'use strict'

entityRegistry['module']['threeDeeMetaBox'] = {
    init: (staticConfig) => {
        const {
            numItems,
            spawnSize,
            spawnHeight,
            frameCount,
            seed,
        } = { ...staticConfig }
        const rng = new Math.seedrandom(seed)

        const world = new CANNON.World()
        world.gravity.set(0, 0, -9.82/2) // m/s²

        // Materials
        var groundMaterial = new CANNON.Material('groundMaterial')
        // Adjust constraint equation parameters for ground/ground contact
        var ground_ground_cm = new CANNON.ContactMaterial(groundMaterial, groundMaterial, {
            friction: .8,
            restitution: 0.3,
            contactEquationStiffness: 1e8,
            contactEquationRelaxation: 3,
            frictionEquationStiffness: 1e8,
            frictionEquationRegularizationTime: 3,
        });
        // Add contact material to the world
        world.addContactMaterial(ground_ground_cm)

        // // Create a slippery material (friction coefficient = 0.0)
        // var slipperyMaterial = new CANNON.Material("slipperyMaterial");
        // // The ContactMaterial defines what happens when two materials meet.
        // // In this case we want friction coefficient = 0.0 when the slippery material touches ground.
        // var slippery_ground_cm = new CANNON.ContactMaterial(groundMaterial, slipperyMaterial, {
        //     friction: 0,
        //     restitution: 0.3,
        //     contactEquationStiffness: 1e8,
        //     contactEquationRelaxation: 3
        // });
        // // We must add the contact materials to the world
        // world.addContactMaterial(slippery_ground_cm);

        const sphereBody = []
        for (let i = 0; i < numItems; ++i) {
            const body = new CANNON.Body({
                mass: 5.25,
                position: new CANNON.Vec3((rng()-.5)*spawnSize[0], (rng()-.5)*spawnSize[2], spawnHeight + (rng()-.5)*spawnSize[1]),
                quaternion: new CANNON.Quaternion(rng(), rng(), rng(), rng()),
                // shape: new CANNON.Box(new CANNON.Vec3(1,1,1)),
                shape: new CANNON.Sphere(1.0),
                material: groundMaterial
            })
            world.addBody(body)
            body.applyLocalForce(new CANNON.Vec3(rng()*2-1, rng()*2-1, -(rng()*5+5)), new CANNON.Vec3(rng()*2-1, rng()*2-1, rng()*2-1))
            sphereBody.push(body)
        }

        const bars = [
            [[0,+5,+5], [5,.25,.25]],
            [[0,-5,+5], [5,.25,.25]],
            [[0,+5,-5], [5,.25,.25]],
            [[0,-5,-5], [5,.25,.25]],

            [[+5,0,+5], [.25,5,.25]],
            [[-5,0,+5], [.25,5,.25]],
            [[+5,0,-5], [.25,5,.25]],
            [[-5,0,-5], [.25,5,.25]],

            [[+5,+5,0], [.25,.25,5]],
            [[-5,+5,0], [.25,.25,5]],
            [[+5,-5,0], [.25,.25,5]],
            [[-5,-5,0], [.25,.25,5]],
        ]
        bars.forEach((bar) => {
            const pos = bar[0]
            const size = bar[1]
            world.addBody(new CANNON.Body({
                mass: 0, // mass == 0 makes the body static
                position: new CANNON.Vec3(pos[0], pos[1], pos[2]),
                shape: new CANNON.Box(new CANNON.Vec3(size[0], size[1], size[2])),
                material: groundMaterial,
            }))
        })

        var fixedTimeStep = 1.0 / 100.0 // seconds

        const bodies = []
        for (let i = 0; i < frameCount; ++i) {
            for (let j = 0; j < sphereBody.length; ++j) {
                bodies[j] = bodies[j] || []
                const pos = [sphereBody[j].position.x, -sphereBody[j].position.z, sphereBody[j].position.y]
                const quat = sphereBody[j].quaternion
                const rot = m4.quaternionToEuler([-quat.y, quat.z, -quat.x, quat.w])
                bodies[j].push([pos, [-rot[0], -rot[1], -rot[2]]])
            }
            world.step(fixedTimeStep)
        }

        return {
            bodies
        }
    },
    staticConfig: [
        { paramName: 'numItems', displayName: 'Items', type: 'int', defaultValue: 100, triggerInit: true },
        { paramName: 'spawnSize', displayName: 'Spawn Size', type: 'float3', defaultValue: [50, 300, 50], triggerInit: true },
        { paramName: 'spawnHeight', displayName: 'Spawn Height', type: 'float', defaultValue: 300, triggerInit: true },
        { paramName: 'frameCount', displayName: 'Frame Count', type: 'int', defaultValue: 1000, triggerInit: true },
        { paramName: 'seed', displayName: 'Seed', type: 'string', defaultValue: 'seed', triggerInit: true },
    ],
    dynamicConfig: [
        { paramName: 'model', displayName: 'Model', type: 'model', defaultValue: '' },
        { paramName: 'itemScale', displayName: 'Item Scale', type: 'float', defaultValue: 1 },
        { paramName: 'animation', displayName: 'Animation', type: 'float', defaultValue: 0 },
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                frameCount,
                model,
                itemScale,
                animation,
                scale,
            } = { ...config }

            const colorBuffer = renderer.getCurrentBuffer('color')
            const depthBuffer = renderer.getCurrentBuffer('depth')
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')

            const scaleMat = m4.scaling(itemScale, itemScale, itemScale)
            const clampedAnimation = clamp(animation, 0, 1)
            const frame = Math.floor(clampedAnimation * (frameCount-1))
            const frameAlpha = (clampedAnimation * (frameCount-1)) % 1
            const insideBalls = []
            self.bodies.forEach(body => {
                const pos1 = body[frame][0]
                const pos2 = body[frame+1][0]
                const pos = m4.lerpVectors(pos1, pos2, frameAlpha)
                if (pos[1] > -13 && pos[1] < 20) {
                    const cagePos = [pos[0]*1, pos[1]*1, [pos[2]*1]]
                    const insideCage = Math.abs(cagePos[0])<11 && Math.abs(cagePos[1])<11 && Math.abs(cagePos[2])<11
                    const fullyInsideCage = Math.abs(cagePos[0])<5 && Math.abs(cagePos[1])<5 && Math.abs(cagePos[2])<5
                    if (insideCage) {
                        const mat = m4.translation(cagePos[0], cagePos[1], cagePos[2])
                        insideBalls.push(m4.inverse(mat))
                    }
                    if (!fullyInsideCage) {
                        const rot1 = body[frame][1]
                        const rot2 = body[frame+1][1]
                        const rot = [
                            Math.abs(rot1[0]-rot2[0])>Math.PI ? rot1[0] : lerp(rot1[0],rot2[0],frameAlpha),
                            Math.abs(rot1[1]-rot2[1])>Math.PI ? rot1[1] : lerp(rot1[1],rot2[1],frameAlpha),
                            Math.abs(rot1[2]-rot2[2])>Math.PI ? rot1[2] : lerp(rot1[2],rot2[2],frameAlpha),
                        ]
    
                        const translationMat = m4.translation(pos[0], pos[1], pos[2])
                        const rotationMat = m4.zRotate(m4.yRotate(m4.xRotation(rot[0]), rot[1]), rot[2])
                        const worldMat = m4.multiply(m4.multiply(translationMat, rotationMat), scaleMat)

                        renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
                    }
                }
            })

            function getIso(p, points, ballSize) {
                const p4 = [p[0], p[1], p[2], 1]
                let isoBalls = 100
                points.forEach(point => {
                    const newIso = getIsoSphere(m4.transformVector(point, p4), ballSize)
                    isoBalls = Math.min(isoBalls, newIso)
                })
                const isoBox = getIsoBox(p, [5.5, 5.5, 5.5])
                return Math.max(isoBalls, isoBox)
            }
    
            if (insideBalls.length > 0) {
                const fieldStrength = 0.06
        
                const fieldSize = 12/2
                const isoField = makeIsoField(fieldSize, (x, y, z) => getIso([x, y, z], insideBalls, 1.25))
                const lists = renderIsoField(fieldSize, isoField, fieldStrength)

                const metaModel = {
                    verts: lists.verts,
                    objects: [
                        {
                            name: 'balls',
                            subObjects: [
                                {
                                    diffuse: [.4, .5, .6],
                                    ambient: [1, 0, 0],
                                    emissive: [0.05, 0.1, 0.2],
                                    lines: [],
                                    tris: lists.tris,
                                    quads: [],
                                }
                            ]
                        }
                    ]
                }

                renderer.decorateModel(metaModel)
                renderer.drawModel(metaModel, m4.identity(), colorBuffer, depthBuffer, brightnessBuffer)
            }
        }
    }
}
