'use strict'

function renderPath (pathVerts, worldMat, colorBuffer, depthBuffer, brightnessBuffer, color, steps=25) {
    const transformedVerts = renderer.transformVerts(worldMat, pathVerts)
    for (let i = 0; i < transformedVerts.length-1; ++i) {
        const p0 = transformedVerts[i]
        const p1 = transformedVerts[i+1]
        for (let j = 0; j < steps; j+=2) {
            const a1 = j/steps
            const a2 = (j+1)/steps
            renderer.clipDrawLine_Depth(
                colorBuffer.data,
                depthBuffer.data,
                brightnessBuffer.data,
                [lerp(p0[0], p1[0], a1), lerp(p0[1], p1[1], a1), lerp(p0[2], p1[2], a1)],
                [lerp(p0[0], p1[0], a2), lerp(p0[1], p1[1], a2), lerp(p0[2], p1[2], a2)],
                color,
                1
            )
        }
    }
}

function pathToSpline (pathVerts, segments=30) {
    const splinePoints = []
    for (let i = 0; i < pathVerts.length-3; ++i) {
        const p0 = pathVerts[i+0]
        const p1 = pathVerts[i+1]
        const p2 = pathVerts[i+2]
        const p3 = pathVerts[i+3]
        for (let j = 0; j < segments; ++j) {
            const a = j/segments
            splinePoints.push(catmullRomSpline3(a, p0, p1, p2, p3))
        }
    }
    return splinePoints
}

function makeCages (pathVerts, up=[0, 1, 0]) {
    const cages = []
    for (let i = 0; i < pathVerts.length-1; ++i) {
        const p0 = pathVerts[i]
        const p1 = pathVerts[i+1]
        const fwd = m4.normalize(m4.subtractVectors(p1, p0))
        const right = m4.normalize(m4.cross(up, fwd))
        up = m4.normalize(m4.cross(fwd, right))
        cages.push([p0, up, right])
    }
    return cages
}

const arrowColors = [
    /*
    [[.7, .4, 1], [0.4, 0, 0]],
    [[1, .7, .4], [0, 0.4, 0]],
    [[.4, 1, .7], [0, 0, 0.4]],
    [[.7, 1, .4], [0.4, 0, 0]],
    [[.4, .7, 1], [0, 0.4, 0]],
    [[1, .4, .7], [0, 0, 0.4]],
    */
    [[.906,.733,.263],[.2,.1,0]],//Gold: E7BB43
    [[.851,.709,.875],[.2,0,0]],//light pink: D9B5DF
    [[.906,.588,.843],[.3,0,0]],//pink: E796D7
    [[.757,.906,.569],[0,.3,0]],//light green: C1E791
    [[.125,.090,.137],[.1,.1,.1]],//black: 201723
]

class ArrowModel {
    cages
    verts
    subObjects

    constructor (pathVerts) {
        this.cages = makeCages(pathVerts)
        this.reset()
    }

    reset () {
        this.verts = []
        this.subObjects = []
    }

    addObject (colorIndex, start, end, widthFunc, heightFunc, hasHead, headWidth, headHeight, headLength) {
        const color = colorIndex % arrowColors.length
        let newSubObject = {
            diffuse: arrowColors[color][0],
            // ambient: [0, 0, 0],
            emissive: arrowColors[color][1],
            lines: [],
            tris: [],
            quads: [],
        }
        let vertIndex = this.verts.length

        const intStart = Math.floor(start)
        const intEnd = Math.floor(end)
        const remainderStart = start % 1
        const remainderEnd = end % 1

        // Interpolated start cap verts
        {
            const i0 = intStart % this.cages.length
            const i1 = (intStart+1) % this.cages.length
            const i = lerp(i0, i1, remainderStart)

            const [p0, up0, right0] = this.cages[i0]
            const [p1, up1, right1] = this.cages[i1]
            const p = m4.lerpVectors(p0, p1, remainderStart)
            const up = m4.lerpVectors(up0, up1, remainderStart)
            const right = m4.lerpVectors(right0, right1, remainderStart)

            const width = widthFunc(0, i/(this.cages.length-1))
            const height = heightFunc(0, i/(this.cages.length-1))
            const cageUp = m4.multiplyVector(up, height)
            const cageDown = m4.multiplyVector(up, -height)
            const cageLeft = m4.multiplyVector(right, -width)
            const cageRight = m4.multiplyVector(right, width)

            this.verts.push(m4.addVectors(p, m4.addVectors(cageUp, cageRight)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageUp, cageLeft)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageDown, cageLeft)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageDown, cageRight)))
        }       
        // Middle section verts
        for (let i = intStart+1; i < intEnd; ++i) {
            const width = widthFunc((i-intStart-1) / (intEnd-intStart), i/(this.cages.length-1))
            const height = heightFunc((i-intStart-1) / (intEnd-intStart), i/(this.cages.length-1))
            const [p0, up, right] = this.cages[i % this.cages.length]
            const cageUp = m4.multiplyVector(up, height)
            const cageDown = m4.multiplyVector(up, -height)
            const cageLeft = m4.multiplyVector(right, -width)
            const cageRight = m4.multiplyVector(right, width)
            this.verts.push(m4.addVectors(p0, m4.addVectors(cageUp, cageRight)))
            this.verts.push(m4.addVectors(p0, m4.addVectors(cageUp, cageLeft)))
            this.verts.push(m4.addVectors(p0, m4.addVectors(cageDown, cageLeft)))
            this.verts.push(m4.addVectors(p0, m4.addVectors(cageDown, cageRight)))
        }
        // Interpolated end cap verts
        {
            const i0 = intEnd % this.cages.length
            const i1 = (intEnd+1) % this.cages.length
            const i = lerp(i0, i1, remainderEnd)

            const [p0, up0, right0] = this.cages[i0]
            const [p1, up1, right1] = this.cages[i1]
            const p = m4.lerpVectors(p0, p1, remainderEnd)
            const up = m4.lerpVectors(up0, up1, remainderEnd)
            const right = m4.lerpVectors(right0, right1, remainderEnd)

            const width = widthFunc(1, i/(this.cages.length-1))
            const height = heightFunc(1, i/(this.cages.length-1))
            const cageUp = m4.multiplyVector(up, height)
            const cageDown = m4.multiplyVector(up, -height)
            const cageLeft = m4.multiplyVector(right, -width)
            const cageRight = m4.multiplyVector(right, width)

            this.verts.push(m4.addVectors(p, m4.addVectors(cageUp, cageRight)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageUp, cageLeft)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageDown, cageLeft)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageDown, cageRight)))
        }       


        if (hasHead) {
            const [p0, up0, right0] = this.cages[intEnd % this.cages.length]
            const [p1, up1, right1] = this.cages[(intEnd+1) % this.cages.length]
            const p = m4.lerpVectors(p0, p1, remainderEnd)
            const up = m4.lerpVectors(up0, up1, remainderEnd)
            const right = m4.lerpVectors(right0, right1, remainderEnd)

            const cageUp = m4.multiplyVector(up, headHeight)
            const cageDown = m4.multiplyVector(up, -headHeight)
            const cageLeft = m4.multiplyVector(right, -headWidth)
            const cageRight = m4.multiplyVector(right, headWidth)

            this.verts.push(m4.addVectors(p, m4.addVectors(cageUp, cageRight)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageUp, cageLeft)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageDown, cageLeft)))
            this.verts.push(m4.addVectors(p, m4.addVectors(cageDown, cageRight)))
            
            const headNorm = m4.normalize(m4.subtractVectors(p1, p0))
            const pHead = m4.addVectors(m4.lerpVectors(p0, p1, remainderEnd), m4.multiplyVector(headNorm, headLength))
            this.verts.push(pHead)
        }

        newSubObject.quads.push([vertIndex+0, vertIndex+3, vertIndex+2, vertIndex+1])
        for (let i = 0; i < intEnd-intStart; ++i) {
            newSubObject.quads.push([vertIndex+0, vertIndex+1, vertIndex+5, vertIndex+4])
            newSubObject.quads.push([vertIndex+1, vertIndex+2, vertIndex+6, vertIndex+5])
            newSubObject.quads.push([vertIndex+2, vertIndex+3, vertIndex+7, vertIndex+6])
            newSubObject.quads.push([vertIndex+3, vertIndex+0, vertIndex+4, vertIndex+7])
            vertIndex += 4
        }
        if (hasHead) {
            newSubObject.quads.push([vertIndex+0, vertIndex+1, vertIndex+5, vertIndex+4])
            newSubObject.quads.push([vertIndex+1, vertIndex+2, vertIndex+6, vertIndex+5])
            newSubObject.quads.push([vertIndex+2, vertIndex+3, vertIndex+7, vertIndex+6])
            newSubObject.quads.push([vertIndex+3, vertIndex+0, vertIndex+4, vertIndex+7])
            newSubObject.tris.push([vertIndex+4, vertIndex+5, vertIndex+8])
            newSubObject.tris.push([vertIndex+5, vertIndex+6, vertIndex+8])
            newSubObject.tris.push([vertIndex+6, vertIndex+7, vertIndex+8])
            newSubObject.tris.push([vertIndex+7, vertIndex+4, vertIndex+8])
        } else {
            newSubObject.quads.push([vertIndex+1, vertIndex+2, vertIndex+3, vertIndex+0])
        }

        this.subObjects.push(newSubObject)
    }

    getModel () {
        const model = {
            verts: this.verts,
            objects: [
                {
                    // name: 'bob',
                    subObjects: this.subObjects
                }
            ]
        }
        renderer.decorateModel(model)
        return model
    }
}

