#!/usr/bin/env python3
import subprocess
import json
import glob
import sys
import os
import re

def ffmpeg_error(task, srcfile, out):
    print("[FAILED]")
    print(f"ERROR while {task} '{srcfile}' - FFmpeg output is:")
    print(out)
    sys.exit(1)

def run_ffmpeg(task, srcfile, *opts):
    res = subprocess.run(
        ["ffmpeg", "-nostdin", "-hide_banner", "-nostats", "-y", "-i", srcfile] + list(opts),
        capture_output=True)
    out = (res.stdout + res.stderr).decode(errors='ignore')
    if res.returncode != 0:
        ffmpeg_error(task, srcfile, out)
    return out

class SampleSet:
    def __init__(self, destdir):
        self.name = os.path.basename(destdir)
        self.desc = self.name.capitalize()
        self.order = 0
        self.music = False
        self.files = []

    def add(self, filename):
        self.files.append(os.path.basename(filename))

if __name__ == "__main__":
    sets = []
    for srcdir in glob.glob(os.path.join("_rawsamples", "*")):
        dirbase = os.path.basename(srcdir).lower()
        destdir = os.path.join("samples", dirbase)
        if not os.path.isdir(destdir):
            os.mkdir(destdir)
        ss = SampleSet(destdir)
        
        infofile = os.path.join(srcdir, "_info.json")
        info = {}
        if os.path.exists(infofile):
            try:
                with open(infofile, "r") as f:
                    info = json.load(f)
                assert isinstance(info, dict), "not a valid JSON object"
            except (EnvironmentError, AssertionError) as e:
                print(f"WARNING: invalid info JSON file '{infofile}' - {e}")
                info = {}
        if "name" in info: ss.desc = info["name"]
        ss.order = info.get("order", 0)
        ss.music = info.get("music", False)

        for srcfile in glob.glob(os.path.join(srcdir, "*.wav")):
            destfile = os.path.join(destdir, os.path.splitext(os.path.basename(srcfile))[0].lower() + ".mp3")
            ss.add(destfile)
            if os.path.exists(destfile) and (os.path.getmtime(destfile) > os.path.getmtime(srcfile)):
                print(srcfile, "->", destfile, "[skip]")
                continue
        
            # ReplayGain analysis
            sys.stdout.write(f"{srcfile} ")
            sys.stdout.flush()
            out = run_ffmpeg("analyzing", srcfile,
                "-af", "replaygain",
                "-f", "null", "nul:" if (sys.platform == "win32") else "/dev/null")
            m = re.search(r'track_gain\s*=\s*([-+]?\d+\.\d+)\s*d[bB]', out)
            if not m: ffmpeg_error("analyzing", srcfile, out)
            db = m.group(1)

            # output file generation
            sys.stdout.write(f"({db} dB) -> {destfile} ")
            sys.stdout.flush()
            run_ffmpeg("processing", srcfile,
                "-af", f"volume={db}dB",
                "-c:a", "libmp3lame", "-b:a", "128k",
                "-map_metadata", "-1", "-fflags", "+bitexact", "-flags:a", "+bitexact",
                destfile)
            print("[OK]")
        sets.append(ss)
    sets.sort(key=lambda ss: (ss.order, ss.name))

    with open("samplesets.js", "w") as f:
        f.write("// This file has been automatically generated by " + os.path.basename(sys.argv[0]) + ". DO NOT EDIT!\n")
        f.write("const SampleSets = [\n")
        first = True
        for ss in sets:
            if not first:
                f.write(",\n")
            files = ', '.join(f'"{f}"' for f in ss.files)
            f.write(f'  {{ "name": "{ss.name}",\n')
            f.write(f'    "desc": "{ss.desc}",\n')
            f.write(f'    "music": {str(ss.music).lower()},\n')
            f.write(f'    "files": [{files}]\n')
            f.write("  }")
            first = False
        f.write("\n];\n")
