package main

import (
	"bufio"
	"bytes"
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"math"
	"math/rand"
	"net"
	"os"
	"runtime"
	_ "strconv"
	"time"
)

const PI2 = math.Pi * 2.0
const DG2RD = PI2 / 360.0

const STAR_RADIUS = 0.1

const PLAYER_MAX_SPEED = 0.05
const PLAYER_TICKCOST = 1.0

const COST_NORM = 10.0
const COST_SEEK = 15.0
const COST_MINE = 20.0

const MISSILE_INITIAL_ENERGY = 1000.0
const MISSILE_INITIAL_SPEED = 0.05
const MISSILE_MAX_SPEED = 0.05
const MISSILE_MAX_SPEED_SQ = (MISSILE_MAX_SPEED * MISSILE_MAX_SPEED)
const MISSILE_HIT_DIST = 0.1
const MISSILE_HIT_DIST_SQ = (MISSILE_HIT_DIST * MISSILE_HIT_DIST)
const MISSILE_LOW_ENERGY = 10.0
const MISSILE_LOW_ENERGY_DRAG = 1.01
const MISSILE_TICKCOST = 50.0

const SEEKING_INITIAL_SPEED = 0.03

const MINE_INITIAL_SPEED = 0.005
const MINE_INITIAL_ENERGY = 5000.0
const MINE_VELOCITY = 0.0005

const GRAV_NORM = 1000000.0
const GRAV_SEEK = 100000.0

type Entity struct {
	Energy    float64
	Rotation  float64
	VelocityX float64
	VelocityY float64
	X, Y      float64
	Id        int
}

type Missile struct {
	Entity
	Owner int
	Type  string
}

type Player struct {
	Entity
}

type GameState struct {
	Missiles []*Missile
	Others   []*Player
	You      *Player
}

type Message struct {
	Gamestate   *GameState
	Messagetype string
}

func clamp(n, from, to float64) float64 {
	if n < from {
		return from
	}
	if n > to {
		return to
	}
	return n
}

func rot(r float64) float64 {
	if r < 0 {
		return r + PI2
	} else if r > PI2 {
		return r - PI2
	}
	return r
}

func (m Missile) move() *Missile {
	d := math.Hypot(m.X, m.Y)

	if d < STAR_RADIUS {
		return nil
	}

	vms := m.VelocityX*m.VelocityX + m.VelocityY*m.VelocityY
	if vms > MISSILE_MAX_SPEED_SQ {
		va := math.Atan2(m.VelocityY, m.VelocityX)
		m.VelocityX = math.Cos(va) * MISSILE_MAX_SPEED
		m.VelocityY = math.Sin(va) * MISSILE_MAX_SPEED
	}

	f := d / 1000
	a := math.Atan2(m.Y, m.X)
	m.VelocityX -= math.Cos(a) * f
	m.VelocityY -= math.Sin(a) * f

	m.X += m.VelocityX
	m.Y += m.VelocityY

	if m.X > 1 {
		m.X = -1
	}
	if m.Y > 1 {
		m.Y = -1
	}
	if m.X < -1 {
		m.X = 1
	}
	if m.Y < -1 {
		m.Y = 1
	}

	if m.Type == "NORMAL" {
		m.Rotation = rot(math.Atan2(m.VelocityY, m.VelocityX))
	}

	if m.Energy < MISSILE_LOW_ENERGY {
		m.VelocityX /= MISSILE_LOW_ENERGY_DRAG
		m.VelocityY /= MISSILE_LOW_ENERGY_DRAG
		return &m
	}

	if m.Type == "MINE" {
		m.VelocityX += math.Cos(m.Rotation) * MINE_VELOCITY
		m.VelocityY += math.Sin(m.Rotation) * MINE_VELOCITY
		m.Energy -= 1
	}

	m.Energy -= MISSILE_TICKCOST

	if m.Type == "NORMAL" {
		m.VelocityX += math.Cos(m.Rotation) * (m.Energy / GRAV_NORM)
		m.VelocityY += math.Sin(m.Rotation) * (m.Energy / GRAV_NORM)
	} else if m.Type == "SEEKING" {
		m.VelocityX += math.Cos(m.Rotation) * (m.Energy / GRAV_SEEK)
		m.VelocityY += math.Sin(m.Rotation) * (m.Energy / GRAV_SEEK)
	}
	return &m
}

func (p Player) shoot(t string) (*Missile, *Player) {
	m := Missile{
		Entity: Entity{
			Energy:   MISSILE_INITIAL_ENERGY,
			Rotation: p.Rotation + math.Pi,
			X:        p.X,
			Y:        p.Y},
		Type: t,
	}

	if t == "NORMAL" {
		p.Energy -= COST_NORM
		m.VelocityX = math.Cos(m.Rotation) * MISSILE_INITIAL_SPEED
		m.VelocityY = math.Sin(m.Rotation) * MISSILE_INITIAL_SPEED
	} else if t == "SEEKING" {
		p.Energy -= COST_SEEK
		m.VelocityX = math.Cos(m.Rotation) * SEEKING_INITIAL_SPEED
		m.VelocityY = math.Sin(m.Rotation) * SEEKING_INITIAL_SPEED
	} else if t == "MINE" {
		p.Energy -= COST_MINE
		m.Rotation = rot(math.Atan2(p.Y, p.X))
		m.Energy = MINE_INITIAL_ENERGY
		m.VelocityX = math.Cos(m.Rotation) * MINE_INITIAL_SPEED
		m.VelocityY = math.Sin(m.Rotation) * MINE_INITIAL_SPEED
	}

	return &m, &p
}

func (p Player) move() *Player {
	if p.Energy <= 0 {
		return nil
	}

	angle := math.Atan2(p.Y, p.X)
	distance := math.Hypot(p.X, p.Y)

	if distance < STAR_RADIUS {
		return nil
	}

	force := distance / p.Energy
	p.VelocityX = clamp((p.VelocityX - math.Cos(angle)*force), -PLAYER_MAX_SPEED, PLAYER_MAX_SPEED)
	p.VelocityY = clamp((p.VelocityY - math.Sin(angle)*force), -PLAYER_MAX_SPEED, PLAYER_MAX_SPEED)
	p.X += p.VelocityX
	p.Y += p.VelocityY

	if p.X > 1 {
		p.X = -1
	} else if p.X < -1 {
		p.X = 1
	}
	if p.Y > 1 {
		p.Y = -1
	} else if p.Y < -1 {
		p.Y = 1
	}

	p.Energy -= PLAYER_TICKCOST
	return &p
}

func p(s string) {
	log.Println(s)
	os.Stderr.Sync()
	os.Stdout.Sync()
}

func parse(d []byte) *Message {
	msg := Message{}
	//p("parse")
	json.Unmarshal(d, &msg)
	//p("pdone")

	if msg.Gamestate != nil {
		for _, m := range msg.Gamestate.Missiles {
			m.Rotation *= DG2RD
		}
	}
	return &msg
}

func receiveNshit(buf *bufio.Reader, data chan<- []byte, kill chan<- bool) {
	buffer := bytes.NewBuffer(make([]byte, 0))
	var (
		part   []byte
		prefix bool
		err    error
	)
	for {
		for {
			if part, prefix, err = buf.ReadLine(); err != nil {
				p("!!!")
				break
			}
			//p(".")
			//p(".")
			//log.Print("%v %v", prefix, string(part))
			buffer.Write(part)
			if !prefix {
				data <- buffer.Bytes()
				buffer.Reset()
			}
		}
		p("::")
		_, _, err := buf.ReadLine()

		if err != nil {
			log.Println(err)
			kill <- true
			os.Exit(0)
		}
	}
}

func actLikeCrazy(world <-chan *GameState, cmd chan<- string, kill <-chan bool, wandl, wandr, thrust int) {
	//var seenMissiles map[int]struct{}
idle:
	for {
		// start := time.Now() // time.Since(start)...
		select {
		case <-kill:
			log.Println("actlikedead")
			return
		case state := <-world:
			if state == nil || state.You == nil {
				continue idle
			}
			//if len(state.Missiles) > 0 {
			//      for _, m := range state.Missiles {
			//              if _, seen := seenMissiles[m.Owner]; !seen {
			//                      seenMissiles[m.Owner] = struct{}{}
			//                      for m != nil {
			//                              //log.Println(m)
			//                              m = m.move()
			//                      }
			//              }
			//              //log.Println(m)
			//      }
			//} else {
			//      // cheap clear
			//      seenMissiles = make(map[int]struct{})
			//}

			fut := state.You.move()
			if fut != nil {
				rkt, _ := state.You.move().shoot("NORMAL")
				oth := []*Player{}
				for _, o := range state.Others {
					oth = append(oth, o.move())
				}

				nearfirst := 10.0
				neardist := 10.0
				nx, ny := 0.0, 0.0
				var lrkt *Missile

				//state.Others
				for j := 0; j < 42; j++ {
					lrkt = rkt
					rkt = rkt.move()
					if rkt == nil {
						break
					}

					for i, o := range oth {
						if o == nil {
							continue
						}
						for gx := -2.0; gx < 3; gx += 2 {
							for gy := -2.0; gy < 3; gy += 2 {
								dx, dy := (o.X+gx)-rkt.X, (o.Y+gy)-rkt.Y
								ds := dx*dx + dy*dy
								if (ds * 1.05) < MISSILE_HIT_DIST_SQ {
									cmd <- "MISSILE"
									continue idle
								}
								if ds < neardist {
									neardist = ds
									nx, ny = o.X, o.Y
								}
							}
						}
						oth[i] = o.move()
					}
					if neardist < nearfirst {
						nearfirst = neardist
					} else {
						break
					}
				}
				rkt = lrkt
				dir := rkt.VelocityX*-(ny-rkt.Y) + rkt.VelocityY*(nx-rkt.X)
				if thrust > 0 && rand.Intn(thrust) < 1 {
					cmd <- "ACCELERATE"
				}
				if dir > 0 {
					if wandr > 0 && rand.Intn(wandr) < 1 {
						cmd <- "RIGHT"
					}
				} else {
					if wandl > 0 && rand.Intn(wandl) < 1 {
						cmd <- "LEFT"
					}
				}
			}
		}
	}
}

func worldDecoupler(world <-chan *GameState, lastState chan *GameState, kill <-chan bool) {
	for {
		select {
		case <-kill:
			break
		case state := <-world:
			log.Println("dlvrd state")
			select {
			case lastState <- state:
			default:
				log.Println(state)
				p("stateskip")
				<-lastState
				go func() { lastState <- state }()
			}
		}
	}
}

func main() {
	runtime.GOMAXPROCS(4)
	host := flag.String("host", "127.0.0.1", "the host to connect to")
	port := flag.Int("port", 54321, "the port to connect to")
	name := flag.String("name", "takeonme", "i can hear voices")
	wandl := flag.Int("wandl", 23, "do random turning - sometimes")
	wandr := flag.Int("wandr", 23, "do random turning - sometimes")
	thrust := flag.Int("thrust", 50, "do random thrusting - sometimes")
	flag.Parse()

	log.Println(time.Now())
	fmt.Println("okay - let's do this")

	world := make(chan *GameState)
	kill := make(chan bool)
	cmd := make(chan string)
	data := make(chan []byte)

	conn, _ := net.Dial("tcp", fmt.Sprintf("%v:%v", *host, *port))
	conn.Write([]byte(fmt.Sprintf("NAME %v\n", *name)))

	dcworld := make(chan *GameState)

	go worldDecoupler(world, dcworld, kill)

	go receiveNshit(bufio.NewReader(conn), data, kill)
	go actLikeCrazy(dcworld, cmd, kill, *wandl, *wandr, *thrust)

	for {
		select {
		case c := <-cmd:
			log.Println(cmd)
			conn.Write([]byte(c + "\n"))
		case d := <-data:
			if msg := parse(d); msg.Gamestate != nil {
				world <- msg.Gamestate
			}
		case <-kill:
			log.Println("main rcvd kill")
			return
		default:
		}
	}
}
