#include "action.h"
#include "player.h"
#include "weapon.h"
#include "map.h"
#include "main.h"
#include "minimax.h"
#include "movement.h"

int action_param_count(action_t act)
{
    /* Calculate player count. */
    player_t *player;
    int player_count;
    player = players->next;
    player_count = 0;
    while (player) {
        player_count++;
        player = player->next;
    }

    switch (act) {
        case act_laser:
        case act_move:
            return player_count;

        case act_mortar:
        case act_droid:
        case act_mine:
            return 0;
    }
}

float action_wrap_minimax(action_t act, int param,
    int depth, float alpha, float beta)
{
    player_t *minimax_me = get_player_by_index(
        (depth + me_depth) % player_count, 0);
    player_t *enemy = get_player_by_index(
        param % (player_count - 1), minimax_me);

    debug("wrap_minimax depth %d me %s enemy %s\n",
        depth, minimax_me->name, enemy->name);

    float damage;
    int orig_j, orig_k;
    float score_offset;

    /* Apply action. */

    if (act == act_laser) {
        int laser;
        if (weapon_type(minimax_me->weapons[0]) == LASER) {
            laser = minimax_me->weapons[0];
        } else if (weapon_type(minimax_me->weapons[1]) == LASER) {
            laser = minimax_me->weapons[1];
        } else {
            laser = 0;
        }

        dir_t dir;
        if (laser != 0) {
            dir = laser_direction(laser,
                map_coords(minimax_me->j, minimax_me->k),
                map_coords(enemy->j, enemy->k));
            damage = laser_damage(laser,
                map_coords(minimax_me->j, minimax_me->k),
                map_coords(enemy->j, enemy->k));
        } else {
            printf("Player %s does not have a laser\n", minimax_me->name);
            dir = dir_up;
            damage = 0;
        }

        enemy->hp -= damage;
        debug("Incrementing %s's score from %f with %f\n",
            minimax_me->name, minimax_me->score, damage);
        minimax_me->score += damage;
        debug("Minimax evaluated player '%s' shooting '%s' with laser "
            "for %f damage\n", minimax_me->name, enemy->name, damage);
    } else if (act == act_move) {
        movement_t movement = movement_towards(
            map_coords(minimax_me->j, minimax_me->k),
            map_coords(enemy->j, enemy->k));

        orig_j = minimax_me->j;
        orig_k = minimax_me->k;
        minimax_me->j += dir_dj(movement.moves[0]) +
            dir_dj(movement.moves[1]) + dir_dj(movement.moves[2]);
        minimax_me->k += dir_dk(movement.moves[0]) +
            dir_dk(movement.moves[1]) + dir_dk(movement.moves[2]);

        if (movement.length > 0) {
            score_offset = 10.0 / movement.length;
        } else {
            score_offset = 0;
        }
        debug("Incrementing %s's score", minimax_me->name);
        debug("from %f", minimax_me->score);
        debug("with %f\n", score_offset);
        minimax_me->score += score_offset;
        debug("Movement from %s to %s is length %d\n",
            minimax_me->name, enemy->name, movement.length);
    }

    /* Do minimax inner logic -- evaluate min for the child state. */
    float sub_value = alphabeta(depth + 1, alpha, beta);

    /* Undo action. */

    if (act == act_laser) {
        enemy->hp += damage;
        debug("Decrementing %s's score from %f with %f\n",
            minimax_me->name, minimax_me->score, damage);
        minimax_me->score -= damage;
        debug("Minimax undid player '%s' shooting '%s' with laser "
            "for %f damage\n", minimax_me->name, enemy->name, damage);
    } else if (act == act_move) {
        minimax_me->j = orig_j;
        minimax_me->k = orig_k;
        debug("%s's score before restore is %f\n",
            minimax_me->name, minimax_me->score);
        minimax_me->score -= score_offset;
        debug("Restored %s's score to %f\n",
            minimax_me->name, minimax_me->score);
    }

    return sub_value;
}

void action_send(action_t act, int param)
{
    player_t *enemy = get_player_by_index(
        param % (player_count - 1), me);

    if (act == act_laser) {
        dir_t dir = laser_direction(
            me->weapons[0],
            map_coords(me->j, me->k),
            map_coords(enemy->j, enemy->k));
        debug("Firing laser at %s in direction %s\n",
            enemy->name, dir_to_string(dir));
        fprintf(stream, "{"
            "\"message\":\"action\","
            "\"type\": \"laser\","
            "\"direction\": \"%s\""
            "}\n",
            dir_to_string(dir));
        fflush(stream);
    } else if (act == act_move) {
        movement_t movement = movement_towards(
            map_coords(me->j, me->k),
            map_coords(enemy->j, enemy->k));
        debug("Directions to enemy %s is %s %s %s\n",
            enemy->name,
            dir_to_string(movement.moves[0]),
            dir_to_string(movement.moves[1]),
            dir_to_string(movement.moves[2]));
        if (movement.moves[0] == dir_none) {
            movement.moves[0] = (dir_t)(rand() % 6);
            movement.moves[1] = (dir_t)(rand() % 6);
            movement.moves[2] = (dir_t)(rand() % 6);
        }
        for (int i = 0; i < 3; i++) {
            dir_t dir = movement.moves[i];
            if (dir == dir_none) {
                break;
            }
            debug("Moving to %s\n", dir_to_string(dir));
            fprintf(stream, "{"
                "\"message\":\"action\","
                "\"type\": \"move\","
                "\"direction\": \"%s\""
                "}\n",
                dir_to_string(dir));
            fflush(stream);
        }
    }
}
