//
// Skinned Mesh Effect file
// Copyright (c) 2000-2002 Microsoft Corporation. All rights reserved.
//
texture tDiffuse;

float4 cAmbient : MATERIALAMBIENT = {0.4f, 0.4f, 0.4f, 1.0f};
float4 cDiffuse : MATERIALDIFFUSE = {0.4f, 0.4f, 0.4f, 1.0f};

// Matrix Pallette
static const int MAX_MATRICES = 26;
float4x3    mWorldMatrixArray[MAX_MATRICES] : WORLDMATRIXARRAY;
float4x4    mViewProj : VIEWPROJECTION;

float4      vLightDir;

///////////////////////////////////////////////////////
struct VS_INPUT
{
    float4  vPosition        : POSITION;
    float4  vNormal          : NORMAL0;
    float4  vBlendWeights    : BLENDWEIGHT;
    float4  vBlendIndices    : BLENDINDICES;
    float2  vTexCoords       : TEXCOORD0;
};

struct VS_OUTPUT
{
    float4  vPosition     : POSITION;
    float3  vDiffuse      : COLOR0;
    float2  vTexCoords0   : TEXCOORD0;
};

struct PS_OUTPUT
{
    float4 vColor           : COLOR0;
};

float3 Diffuse(float3 Normal)
{
    // N.L Clamped
    return (max(0.0f, dot(Normal, vLightDir.xyz)));
}

VS_OUTPUT VShade(VS_INPUT i, uniform int NumBones)
{
    VS_OUTPUT   o;
    float3      vPosition  = 0.0f;
    float3      vNormal    = 0.0f;
    float       LastWeight = 0.0f;

    // Compensate for lack of UBYTE4 on Geforce3
    int4 IndexVector = D3DCOLORtoUBYTE4(i.vBlendIndices);

    // cast the vectors to arrays for use in the for loop below
    float BlendWeightsArray[4] = (float[4])i.vBlendWeights;
    int   IndexArray[4]        = (int[4])IndexVector;

    // calculate the pos/normal using the "normal" weights
    //        and accumulate the weights to calculate the last weight
    for (int iBone = 0; iBone < NumBones-1; iBone++)
    {
        LastWeight = LastWeight + BlendWeightsArray[iBone];
        vPosition += mul(i.vPosition, mWorldMatrixArray[IndexArray[iBone]]) * BlendWeightsArray[iBone];
        vNormal   += mul(i.vNormal,   mWorldMatrixArray[IndexArray[iBone]]) * BlendWeightsArray[iBone];
    }
    LastWeight = 1.0f - LastWeight;

    // Now that we have the calculated weight, add in the final influence
    vPosition += (mul(i.vPosition, mWorldMatrixArray[IndexArray[NumBones-1]]) * LastWeight);
    vNormal   += (mul(i.vNormal,   mWorldMatrixArray[IndexArray[NumBones-1]]) * LastWeight); 

    // normalize normals
    vNormal = normalize(vNormal);

    // transform position from world space into view and then projection space
    o.vPosition = mul(float4(vPosition.xyz, 1.0f), mViewProj);

    // Shade (Ambient + etc.)
    o.vDiffuse.xyz = cAmbient.xyz + Diffuse(vNormal) * cDiffuse.xyz;

    // copy the input texture coordinate through
    o.vTexCoords0 = i.vTexCoords.xy;

    return o;
}

int CurNumBones = 2;
VertexShader vsArray[4] = { compile vs_1_1 VShade(1),
                            compile vs_1_1 VShade(2),
                            compile vs_1_1 VShade(3),
                            compile vs_1_1 VShade(4)
                          };

sampler DiffuseMap = sampler_state
{
	Texture   = <tDiffuse>;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
	MipFilter = LINEAR;
};

//////////////////////////////////////
// Techniques specs follow
//////////////////////////////////////
technique t0
{
    pass p0
    {
        VertexShader = (vsArray[CurNumBones]);

        // lighting
        Lighting       = FALSE;
        SpecularEnable = FALSE;

        // samplers
        Sampler[0] = (DiffuseMap);

        // texture stages
        
        ColorOp[0]   = MODULATE;
        ColorArg1[0] = TEXTURE;
        ColorArg2[0] = DIFFUSE;
        AlphaOp[0]   = MODULATE;
        AlphaArg1[0] = TEXTURE;
        AlphaArg2[0] = DIFFUSE;

        //ColorOp[0]   = SELECTARG1;
        //ColorArg1[0] = TEXTURE;

        ColorOp[1]   = DISABLE;
        AlphaOp[1]   = DISABLE;

        // shaders
        VertexShader = (vsArray[CurNumBones]);
        PixelShader  = NULL;
    }
}

technique t1
{
    pass p0
    {
        VertexShader = (vsArray[CurNumBones]);

        // Pixel shader
        PixelShader =
            asm
            {
                ps_1_1
                tex t0

				// Texture * Diffuse + Ambient
                mad_sat r0, t0, c1, c0
            };

        PixelShaderConstant1[0] = (cAmbient);
        PixelShaderConstant1[1] = (cDiffuse);

        Sampler[0] = (DiffuseMap);
    }
}

