{$N+}
program PxdTut8;
uses cossin, crt,dos;

{  degrees :                                 map :
                      270
                     1440
                                                 | --------------> X-axis
                                                 |
             180                 0/360           |
             960                 0/1920          |
                                                 |
                                                 |
                       90                        V  Y-axis
                      480



As the field of view is 60 degrees we increase the raycasting angle with
60 / 320 = 0.1875 degrees - therefore there are 360 / 0.1875 = 1920 entries
in the look-up tables and we also base our player-movement on these 1920
possible angles.

The world is a 100 X 100 grid with a grid-size of 64 X 64.
Texture-size is 128 X 128


New features from PXDTUT7 :
           - New SQRT function - integers only, much faster!
           - Interpolate between heights instead of calculating each
             height. Faster & smoother graphic
           - Resizeable screen-width. Can have any width with only a
             single change in the code.
           - Clipping to a defined game-window. Allows a frame around the
             game window.
           - Floor / Ceiling mapping
           - Accelleration on both movement and turning
           - Ice surface.
           - FrameRate is displayed when game is exited for benchmarking

  }




CONST
 {VIEW CONSTANTS}
 SCREEN_SIZE = 217;                  {How wide is the screen}
 SCREEN_Y_SIZE = 200;
 HALFSCREEN = SCREEN_SIZE DIV 2;
 FOV = 60;
 SCREEN_X_START = 7;                 {left clipping}
 YTOPCLIP = 8;                       {top clipping}
 YBOTCLIP = 191;                     {bottom clipping}

 {WORLD CONSTANTS}
 GRID_SIZE = 64;          {each cell in the world is 64 X 64 units}
 WORLD_SIZE = 100;        {the world is 100 X 100 cells}
 WORLD_UNIT_SIZE = WORLD_SIZE * GRID_SIZE;
 CEILING_COLOR = 35;
 FLOOR_COLOR   = 165;
 MAX_DOORS     = 10;      {max number of doors active at the same time}

 CLOSEST_WALL  = 32;      {how close can we go to a wall}
 HORIZON       = 100;     {guess....}
 VGA           = $A000;   {Segment of our VGA screen}

 DOOR_CODE     = 5;
 FLOOR         = 6;
 FLOOR_ICE     = 7;
 CEILING       = 8;       {some texture numbers}

 DOOR_SPEED    = 4;       {how many columns pr. frame do we move our doors}

 {CONSTANTS FOR KEYBOARD-HANDLER - THE SCANCODES :}
 {notice that when you release a key the interrupt sends a 128 + scancode -}
 {hence the stop-codes}
 MOVE_FWD     = 72;          {forward arrow}
 STOP_FWD     = 128 + 72;
 MOVE_BACK    = 80;          {back arrow}
 STOP_BACK    = 128 + 80;
 MOVE_LEFT    = 75;          {left arrow}
 STOP_LEFT    = 128 + 75;
 MOVE_RIGHT   = 77;          {right arrow}
 STOP_RIGHT   = 128 + 77;
 QUIT_VALUE   = 1;           {The ESC key}

 OPEN_DOOR    = 57;          {Space}
 TOGGLE_FLOOR = 23;          { I }


TYPE

  Level =  RECORD
             tilename : Array[1..7] of string;
             map : Array[1..100,1..100] of byte
           END;


  DoorInfoT = RECORD
                door_offset : byte;
                status : byte;           {1 = open, 2 = close, 0 = still}
                delay : integer;
                XMapPos, YMapPos : byte;
                DoorType : byte;
              End;



  LevelFileT = File of level;

  {the table types}

  Table1920realT = Array[0..1920] of double;
  HeightTabT = Array[0..9050] of word;

  Table1920fixedT = Array[0..1920] of longint;



  Virtual = Array[1..64000] of byte;
  Virscr = ^Virtual;                   {our virtual screen}

  VirtualTex = Array[1..16384] of byte;
  VirTex  = ^VirtualTex;


  {contais the information needed to draw the walls}
  RayBufferT = Array[0..SCREEN_SIZE] of Record
                                 TexNr  : byte;
                                 TexCol : byte;
                                 dist   : word;
                                 height : integer;
                                 Xmap, Ymap : integer;
                                 side : byte; {1 = X, 2 = Y}
                                end;





var
 {angle variables - calculated according to FOV and ScreenSize}
 {In pxdtut7 these were constants - now they are variables as they change }
 {with the screen size}
 ANG_0   : word;
 ANG_3   : word;
 ANG_4   : word;
 ANG_5   : word;
 ANG_6   : word;
 ANG_30  : word;
 ANG_90  : word;
 ANG_180 : word;
 ANG_270 : word;
 ANG_360 : word;

 ANG_STEP  : double;   {in pxdtut7 this was 0.1875}
 NR_ANGLES : word;     {in pxdtut7 this was 1920}

 time  : longint ABSOLUTE $0040:$006C;
 frames,T1 : longint;  {we use these to calculates the FPS rate}

 {world vars}
 i,dummy : integer;
 ch : char;
 quit : boolean;
 ice  : boolean;            {icy or normal surface}
 floorceil : boolean;       {do we render floor / ceilings}
 LevelFile : LevelFileT;
 map : level;
 vaddr : word;
 scr2 : virscr;
 Tex1,Tex2,Tex3,Tex4,Tex5,Tex6,Tex7,Tex8 : virtex;
 TexAddr : Array[1..8] of word;  {Array of adresses of textures     }
 Faddr   : Word;                 {The address of the current texture}

 DoorArray : Array[1..MAX_DOORS+1] of DoorInfoT;
 NumberOfActiveDoors : byte;     {keeps track of active doors - ie. doors}
                                 {that are opening or closing            }

 ACTIVE_FLOOR : byte;            {either this is normal floor or it is icy}
                                 {floor}

 {look-up tables}
 ScaleTable  : Array[0..SCREEN_SIZE] of integer;  {fish-eye table for walls}

 InvCos_Table : Array[0..SCREEN_SIZE] of longint; {InvCos in 22.10 fixed p}
 FloorDist_Table : Array[0..200] of longint;      {Distance to screen rows }
                                                  {in 22.10 fixed point}
 YnextTable  : ^Table1920realT;
 XnextTable  : ^Table1920realT;                   {X & Y steps in Xray & Yray}
 TanTable    : ^Table1920realT;
 HeightTable : ^HeightTabT;
 Cos_Table : ^Table1920fixedT;
 Sin_Table : ^Table1920fixedT;                    {Cos / Sine in 22.10 fixed}

 {player vars}
 PlayerX, PlayerY, PlayerAng : integer;
 Ice_MoveAngle               : integer;
 dx,dy : integer;                  {for movement}
 XMapPos, YMapPos : integer;       {which map pos are player at ?}

 {keyboard handler}
  OLDKbdHandler : procedure;       {store the original BIOS handler}
  turnflag   : byte;
  move_dir   : integer;
  turn_value : double;
  acc        : double;             {these are use by game physics}

  {drawing buffers}
  DrawBuffer : RayBufferT;                 {data for the walls}

  XnewHeight : Array[0..100] of integer;   {data for interpolation of wall}
  XNewHeightPos :  byte;                   {heights}



PROCEDURE CheckDoor(x,y : integer; ang : integer); FORWARD;
FUNCTION  FindActiveDoor(X,Y : byte) : byte; FORWARD;



PROCEDURE CalcAngles;
begin
ANG_STEP  := FOV / SCREEN_SIZE;
NR_ANGLES := Round(360 / ANG_STEP);

ANG_0   := Round((NR_ANGLES / 360) * 0);
ANG_3   := Round((NR_ANGLES / 360) * 3);
ANG_4   := Round((NR_ANGLES / 360) * 4);
ANG_5   := Round((NR_ANGLES / 360) * 5);
ANG_6   := Round((NR_ANGLES / 360) * 6);
ANG_30  := Round((NR_ANGLES / 360) * 30);
ANG_90  := Round((NR_ANGLES / 360) * 90);
ANG_180 := Round((NR_ANGLES / 360) * 180);
ANG_270 := Round((NR_ANGLES / 360) * 270);
ANG_360 := Round((NR_ANGLES / 360) * 360);
end;

Procedure ToggleFloor;
begin
 If ACTIVE_FLOOR = FLOOR then
       begin
        ACTIVE_FLOOR := FLOOR_ICE;
        ice := true;
        Ice_MoveAngle := PlayerAng;
       end
        else
       begin
        ACTIVE_FLOOR := FLOOR;
        ice := false;
       end;
end;



{$F+}
PROCEDURE MyKbdHandler;
INTERRUPT;
VAR
 input : byte;
BEGIN
  input := Port[$60];     {port $60 is the data-port of the keyboard }

  case input of
   MOVE_FWD     : move_dir := 1;
   STOP_FWD     : move_dir := 0;
   MOVE_BACK    : move_dir := 2;
   STOP_BACK    : move_dir := 0;
   MOVE_LEFT    : turnflag := 1;
   STOP_LEFT    : turnflag := 0;
   MOVE_RIGHT   : turnflag := 2;
   STOP_RIGHT   : turnflag := 0;
   QUIT_VALUE   : quit := true;
   OPEN_DOOR    : CheckDoor(PlayerX,PlayerY, PlayerAng);
   TOGGLE_FLOOR : ToggleFloor;
  end;

  port[$20] := $20;                    {acknowledge the interrupt}
END;
{$F-}


{this is our FAST SQRT function}
Function IntSqrt(Const L : LongInt) : Word;
Assembler;
Asm
	Db $66; mov ax,WORD PTR [l]
	Db $66; mov bx,ax
	Db $66; mov cx, $0000; DW $4000;
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over1
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over1:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over2
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over2: Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over3
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over3:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over4
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over4:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over5
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over5:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over6
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over6:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over7
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over7:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over8
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over8:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over9
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over9:	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over10
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over10:
	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over11
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over11:
	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over12
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over12:
	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over13
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over13:
	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over14
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over14:
	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over15
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over15:
	Db $66; shr cx,2
	Db $66; mov dx,cx
	Db $66; add dx,ax
	Db $66; shr ax,1
	Db $66; cmp dx,bx
	ja @over16
	Db $66; sub bx,dx
	Db $66; or ax,cx
@over16:
End;





{******************************************************************}
{*      HERE GOES SOME GRAPHIC PROCEDURES / FUNCTIONS             *}
{******************************************************************}

PROCEDURE Display(FName: string;where:word;X,Y : integer);
var
   xsize,ysize : word;
   LoadFile: file;
   i : byte;

begin
    Assign(LoadFile,FName);
    Reset(LoadFile,1);
    BlockRead(loadfile,xsize,2);
    Blockread(loadfile,ysize,2);
    If Xsize=319 THEN
    BlockRead(LoadFile,mem[Where:0+(y*320)],(xsize+1)*(ysize+1))
    Else
    For i:=0 to ysize DO
      Blockread(Loadfile,Mem[Where:(i*320)+(x+(y*320))],Xsize+1);
    Close(LoadFile);
END;

PROCEDURE LoadPal(PalName: string);
TYPE
  ColorRec = record
              r,g,b: byte;
             end;
  PalType = array[0..255] of ColorRec;
  PalFileType = file of PalType;

Var
    PalFile: PalFileType;
    Pal: ^PalType;
    i: integer;
begin
    New(Pal);
    Assign(PalFile,PalName);
    Reset(PalFile);
    Read(PalFile,Pal^);
    Close(PalFile);
    for i := 0 to 255 do
    BEGIN
        Port[$3C8] := i;
        Port[$3C9] := Pal^[i].r;
        Port[$3C9] := Pal^[i].g;
        Port[$3C9] := Pal^[i].b;
    END;
    Dispose(Pal);
end;

PROCEDURE SetPal(Farvenr: byte;R,G,B : Byte);
Assembler;
   asm
      mov    dx,3c8h
      mov    al,[farvenr]
      out    dx,al
      inc    dx
      mov    al,[r]
      out    dx,al
      mov    al,[g]
      out    dx,al
      mov    al,[b]
      out    dx,al
End;


Procedure Clear (Col : Byte;where:word);
Assembler;
     asm
        mov     cx, 32000;
        mov     ax,where
        mov     es,ax
        xor     di,di
        mov     al,[col]
        mov     ah,al
        rep     stosw
      END;



FUNCTION GetPixel(x,y : integer;where :word) : byte;
Assembler;
asm
    mov     ax,[where]
    mov     es,ax         {es peger p $a000}
    mov     bx,[X]        { bx = x }
    mov     dx,[Y]        { dx = y }
    mov     di,bx         { offset = x}
    mov     bx, dx        { bx = dx = y }
    shl     dx, 8
    shl     bx, 6
    add     dx, bx        {dx = y*320 }
    add     di, dx        {offset = x + (y*320) }
    mov     al, es:[di]     {al = farvenr }
end;

PROCEDURE flip(source,dest:word);
ASSEMBLER;
asm
  mov     bx,ds
  mov     ax, dest
  mov     es, ax
  mov     ax, source
  mov     ds, ax
  xor     si, si
  xor     di, di
  mov     cx, 16000
  db      $66
  rep     movsw
  mov     ds,bx
END;



PROCEDURE Loadtexture(Picture : string;nr : integer);
VAR
 i,j : integer;
BEGIN
 Display(picture,Vaddr,0,0);
 For i:=0 to 128 DO
  For j:=0 to 128 DO
   mem[TexAddr[nr]:j+i*128]:=GetPixel(j,i,Vaddr);
 Clear(0,Vaddr);
END;


{*******************************************************************}
{*             CALCULATE THE TABLES FOR THE ENGINE                 *}
{*******************************************************************}

procedure CalcTables;
var
 i : integer;
 angle : double;
 rad_angle : double;
begin
  {calc scale-table - this will handle the fisheye effect}
  {this is just the sinus value to the difference between PlayerAngle and}
  {RayAngle - think about it... Some people use an invers cosinus wave.. }
  {but they suck 8)   This works !!!}

  Writeln('Calculating fisheye table');
  angle := FOV / 2;
  for i := 0 to HALFSCREEN-1 do
   begin
    ScaleTable[i] := Round(SinDrg(90-angle)*1024);
    angle := angle - ANG_STEP;
   end;
  for i := HALFSCREEN to SCREEN_SIZE do
   begin
    ScaleTable[i] := Round(SinDrg(90-angle)*1024);
    angle := angle + ANG_STEP;
   end;

 {Calc Ynext-table  -  for easy calculation of Y-step value in the Xray-proc}
 Writeln('Calculating Ynext table');
 angle := 0;
 new(YnextTable);
 for i := 0 to NR_ANGLES do
  begin
   If (i = ANG_90) or (i = ANG_270) then YnextTable^[i] := 0 else
   YnextTable^[i] := 64 * TanDrg(angle);
   angle := Angle + ANG_STEP;
  end;

 {Calc Xnext-Table  -  for easy calculation of X-step value in the Yray-proc}
 Writeln('Calculating Xnext table');
 angle := 0;
 new(XnextTable);
 for i := 0 to NR_ANGLES do
  begin
   if (i = ANG_0) or (i = ANG_360) or (i = ANG_180) then XnextTable^[i] := 0
    else
   XnextTable^[i] := 64 / TanDrg(angle);
  angle := angle + ANG_STEP;
  end;

 {Calc Tantable}
 Writeln('Calculating Tangens table');
 angle := 0;
 new(TanTable);
 for i := 0 to NR_ANGLES do
  begin
   If (i = ANG_90) or (i = ANG_270) then TanTable^[i] := 1 else
   Tantable^[i] := TanDrg(angle);
   angle := angle + ANG_STEP;
  end;


 {calc the HeightTable  -  This contains the scaled height of a wall sliver}
 {based on the distance to it}
 Writeln('Calculating Height table');
 new(HeightTable);
 HeightTable^[0] := 32000; {never allow player THIS close to a wall :)}
 for i:= 1 to 9050 do      {9050 is the max distance in a 100 X 100 world}
  begin
   HeightTable^[i] := Round(10000 / i);
  end;


 Writeln('Calculating Floor InvCos Table');
 {this table is a longint table stored as 22:10 fixed point values}
 {used for calculating distance to a specific pixel}
 for i := -ANG_30 to ANG_30 do
  begin
   rad_angle := (3.272e-4) + i*2*pi/ANG_360;
   InvCos_table[i+ANG_30] := Round(1024/cos(rad_angle));
  end;

 {This table contains the distance to a Screen row - straight ahead.    }
 {5100 - experiment with this value... it calibrates the distance to the}
 {first row                                                             }
 Writeln('Calculating Floor Row Table');
 for i := 200 downto HORIZON + 1 do
  begin
    FloorDist_Table[i] := Round((5100 * 1024) / (i-HORIZON));
  end;
 for i := HORIZON - 1 downto 0 do
  begin
    FloorDist_Table[i] := Round((5100 * 1024) / (HORIZON-i));
  end;


 {A sinus table in 22.10 fixed point}
 Writeln('Calculating Floor Sinus Table');
 New(Sin_table);
 for i := 0 to NR_ANGLES do
  begin
   rad_angle := (3.272e-4) + i*2*pi/ANG_360;
   Sin_Table^[i] := Round(sin(rad_angle) * 1024);
  end;

 {A cosinus table in 22.10 fixed point}
 Writeln('Calculating Floor Cosinus Table');
 New(Cos_Table);
 for i := 0 to NR_ANGLES do
  begin
   rad_angle := (3.272e-4) + i*2*pi/ANG_360;
   Cos_Table^[i] := Round(cos(rad_angle) * 1024);
  end;


Writeln('Done calculating tables...');
end;



PROCEDURE DoBackGround;
Assembler;
asm
 mov ax,Vaddr
 mov es,ax
 mov dx, 2567             {top-right corner = 8,8... ie. dx = 8*320 + 8 }

 mov bx, HORIZON          {how many lines to clear with SKY_COLOR}
 sub bx, YTOPCLIP
 mov al, CEILING_COLOR
 mov ah, CEILING_COLOR
 @loop1:
 mov cx,109               {218 bytes, but only 109 words }
 mov di,dx
 rep stosw                {draw a word at a time}
 add dx,320               {jump to next row in view-field in frame}
 dec bx
 cmp bx,0
 jnz @loop1

 mov bx, YBOTCLIP         {how many lines to clear with FLOOR_COLOR}
 sub bx, HORIZON
 mov al, FLOOR_COLOR
 mov ah, FLOOR_COLOR
 @loop2:
 mov cx,109               {218 bytes, but only 109 words}
 mov di,dx
 rep stosw                {draw a word at a time}
 add dx,320               {jump to next row in view-field in frame}
 dec bx
 cmp bx,0
 jnz @loop2
end;  {clear view-area}



{This functions scans through the map to find X-walls where a X-wall is}
{defined as the sides of the cells pictured below :                    }
{                   _______                                            }
{                  |       |                                           }
{      X-wall ---> |       | <--- X-wall                               }
{                  |-------|                                           }

FUNCTION Xray(x,y : integer; angle : integer; var Xhit : word;
        var Yhit : word;var texcol : byte;var Xmap, Ymap : integer) : byte;
var
 Xnext : integer;
 Xpos,RYpos : word;
 Ypos, Ynext : double;
 Xbeg : integer;
 XmapPos, YmapPos : integer;
 SaveXmap,SaveYmap : integer;
 wallfound,doorhit : boolean;
 ActiveDoorNr : byte;


BEGIN
wallfound := false;
doorhit := false;
Xbeg := x AND $FFC0;  {Xbeg is now left side of current cell}

if (angle = ANG_0 ) or (angle = ANG_180) then
 begin
  Ynext := 0
 end
  else
 begin
  Ynext := YnextTable^[angle];   {get out Ystep value based on ray-angle}
 end;

if (angle > ANG_270) or (angle < ANG_90) then   {looking to the right ?}
 begin    {then we start our casting in RIGHT side of current cell...}
  Xpos := Xbeg + GRID_SIZE;
  Xnext := GRID_SIZE;   {... AND move the the right on map.}
 end
  else
 begin    {looking to the left ?}
  xpos := Xbeg;    {then we start at the LEFT side of current cell...}
  Xnext := - GRID_SIZE; {... AND move to the left on map.}
  Ynext := -Ynext; {this is because Tangens somehow has wrong sign when}
                   {you look to the left.}
 end;

 if (angle = ANG_0) or (angle = ANG_180) then
  Ypos := y
 else
  Ypos := (Xpos-x) * tantable^[angle] + y;  {where in current cell do we}
                                            {begin our ray-casting ??   }

If (Ypos  > WORLD_UNIT_SIZE) or (Ypos < 0) then Ypos := 0; {just to be sure...}

repeat
 XmapPos := Xpos shr 6;
 if (angle > ANG_270) or (angle < ANG_90) then inc(XmapPos);

 RYpos := Round(Ypos);     {only round this value ONCE}

 YmapPos := RYpos shr 6+1; {which cell have we hit with our ray ??}
 If (RYpos MOD 64 = 0) AND (angle > ANG_180) then
  begin  {this fixes the corners of the walls - remember the bug from tut7 ?}
   Dec(YmapPos);
   if (map.map[XmapPos,YmapPos] = 0) then Inc(YmapPos);
  end;

 If (map.map[XmapPos, YmapPos] > 0) then {we have hit a wall !! :) }
  begin
    wallfound := true;
    Xray := map.map[XmapPos, YmapPos]; {texture number of hit}
    XHit := Xpos;         {X-world coordinate of hit - for distance calc.}
    YHit := RYpos;        {Y-world coordinate of hit - for distance calc.}
    Xmap := XmapPos;
    Ymap := YMapPos;
    if (angle > ANG_90) and (angle < ANG_270) then
      TexCol := 128-((Yhit MOD 64) shl 1)
    else
      TexCol := (Yhit MOD 64) shl 1;

    if (map.map[XmapPos,YmapPos] = DOOR_CODE)  then
      begin  {adjust position of door / texture column}
        Xhit := Xhit + Xnext DIV 2;
        Yhit := Round(Ypos  + Ynext / 2);

        if (angle > ANG_90) and (angle < ANG_270) then
          TexCol := 128 - ((Yhit MOD 64) shl 1)
        else
          TexCol := (Yhit MOD 64) shl 1;

        ActiveDoorNr := FindActiveDoor(XmapPos,YmapPos);
           if (ActiveDoorNr  > 0)
             then  {we have found the door in the list of active doors}
               begin
                if ((TexCol - DoorArray[ActiveDoorNr].Door_offset) < 0) then
                   begin {cast ray beyond door}
                     SaveXmap := XmapPos;
                     SaveYmap := YmapPos;
                     Doorhit := true;
                     wallfound := false;
                     map.map[XmapPos,YmapPos] := 0;
                    end
                     else
                      TexCol := TexCol - DoorArray[ActiveDoorNr].Door_offset;
               end;
         end;
      end;

   Inc(Xpos,Xnext);
   Ypos := Ypos + Ynext;  {advance the ray one step on the map.}

until (wallfound) or (Xpos<0) or (Xpos > WORLD_UNIT_SIZE) or (Ypos <0) or
       (Ypos > WORLD_UNIT_SIZE);

 If not(wallfound) then Xray := 0;
 if doorhit then map.map[SaveXmap,SaveYMap] := DOOR_CODE;

{repeat until we have found a wall,door - or is outside the world}
end;



{This functions scans through the map to find Y-walls where an Y-wall is}
{defined as the sides of the cells pictured below :                     }
{                                                                       }
{                      | an Y-wall                                      }
{                   ___V___                                             }
{                  |       |                                            }
{                  |       |                                            }
{                  |-------|                                            }
{                                                                      }
{                      | an Y-wall                                      }



FUNCTION Yray(x,y : integer; angle : integer;var Xhit : word;
         var Yhit : word; var texcol : byte;var Xmap,Ymap : integer) : byte;
{Not commented.... Check the comments for Xray - basicly the same...    }
{only difference is that we move in fixed Yvalues and check for Ywalls  }

VAR
 Ynext, Ypos,RXpos, Ybeg : integer;
 Xnext, Xpos : double;
 XmapPos, YmapPos : integer;
 SaveXmap, SaveYmap : integer;
 wallfound,doorhit : boolean;
 ActiveDoorNr : byte;



BEGIN
 wallfound := false;
 doorhit := false;
 Ybeg := Y AND $FFC0;

 if (angle = ANG_90 ) or (angle = ANG_270) then
 begin
  Xnext := 0
 end
  else
 begin
  Xnext := XnextTable^[angle];
 end;

 If (angle < ANG_180) then
  begin
   Ypos := Ybeg + GRID_SIZE;
   Ynext := GRID_SIZE;
  end
   else
  begin
   Ypos := Ybeg;
   Ynext := - GRID_SIZE;
   Xnext := -Xnext;
  end;

if (angle = ANG_90) or (angle = ANG_270) then
  Xpos := x
 else
  Xpos := (Ypos-y) / TanTable^[angle] + x;

If (Xpos > WORLD_UNIT_SIZE) or (Xpos < 0 ) then Xpos := 0;

repeat
    RXpos := Round(Xpos);
    XmapPos := RXpos shr 6 + 1;
    YmapPos := Ypos shr 6;
    if (angle < ANG_180) then inc(YmapPos);

    If (RXpos MOD 64 = 0) AND (angle > ANG_90) AND (angle < ANG_270)
     then
      begin
       Dec(XmapPos);
       if (map.map[XmapPos,YmapPos] = 0) then Inc(XmapPos);
      end;

  If (map.map[XmapPos, YmapPos] > 0) then
    begin
     wallfound := true;
     Yray := map.map[XmapPos, YmapPos];
     XHit := RXpos;
     YHit := Ypos;
     Xmap := XMapPos;
     Ymap := YMapPos;
     TexCol := (Xhit MOD 64) shl 1;

     if (map.map[XmapPos,YmapPos] = DOOR_CODE)  then
      begin  {adjust position of door / Texture Column}
        Xhit := Round(Xpos + Xnext / 2);
        Yhit := Yhit + Ynext DIV 2;

        TexCol := (Xhit MOD 64) shl 1;

        ActiveDoorNr := FindActiveDoor(XmapPos,YmapPos);
           if (ActiveDoorNr  > 0)
             then  {we have found the door in the list of active doors}
               begin
                 if ((TexCol-DoorArray[ActiveDoorNr].Door_offset) < 0) then
                   begin
                     SaveXmap := XmapPos;
                     SaveYmap := YmapPos;
                     Doorhit := true;
                     wallfound := false;
                     map.map[XmapPos,YmapPos] := 0;
                   end
                    else
                      TexCol := TexCol - DoorArray[ActiveDoorNr].Door_offset;
               end;
      end;
    end;

   Inc(Ypos,Ynext);
   Xpos := Xpos + Xnext;

 until (wallfound) or (Xpos < 0) or (Xpos > WORLD_UNIT_SIZE) or (Ypos <0) or
       (Ypos > WORLD_UNIT_SIZE);

 If not(wallfound) then Yray := 0;
 if doorhit then map.map[SaveXmap,SaveYmap] := DOOR_CODE;
END;



{**********************************************************}
{**                                                      **}
{**            DRAW THE WALLS FROM DRAWBUFFER            **}
{**                                                      **}
{**********************************************************}


PROCEDURE DrawScreen;
VAR
 i,j : integer;
 TexNr : byte;
 TexCol : byte;
 height : integer;
 Ytop : integer;
 ScaleStep : word;
 Texpos : word;


BEGIN
for i := 0 to SCREEN_SIZE do
  begin
     TexCol := Drawbuffer[i].TexCol;
     TexNr  := Drawbuffer[i].TexNr;
     Faddr := TexAddr[TexNr];

     height := Drawbuffer[i].height;
     ScaleStep := 16384 DIV height;       {step in 7.9 fixed-point}
     ScaleStep := ScaleStep + ScaleStep;  {step in 8.8 fixed-point}

     Ytop := HORIZON - height shr 1;  {find top Yvalue of wall-sliver}
                                      {this is just half the height above}
                                      {the horizon}


    if (Ytop < YTopClip) then
      begin
        TexPos := ((0-Ytop+YTopClip) * ScaleStep) shr 8; {shr 8 because Scalestep is}
                                              {in 8.8 fixed point        }
        Ytop := YTopClip;
        height := SCREEN_Y_SIZE-YTopClip- (200-YBOTCLIP);
      end
       else
        TexPos := 0;

     {Here we do the texturemap - it's in assembler but don't worry - }
     {there are PLENTY of coments in the code 8)                      }

     asm
      push ds              {push the data segment - don't forget 8) }

      mov al,TexCol
      cmp al,0           {is the texture Xpos < 0 - then we clip it to 0 }
      ja @OK1
      mov TexCol,0
    @OK1:
      cmp al, 125        {is the texture Xpos > 125 - then we clip to 125}
      jb @OK2
      mov TexCol,125
    @OK2:

      mov bx,vaddr
      mov es,bx          {es = our virtual screen = where to draw        }
      mov bx, Faddr
      mov ds,  bx          {ds = the frame page = where to read the texture}

      mov bx, [Ytop]
      mov di, bx
      shl bx,8           {bx = Ytop * 256                              }
      shl di,6           {di = Ytop * 64                               }
      add di,bx          {di = Ytop * 256 + Ytop * 64 = Ytop * 320     }
      add di,[i]         {di = Ytop * 320 + screenX = screenoffset     }
      add di,SCREEN_X_START {this is so we can adjust Screen to the right}

      mov bx, [TexPos]   {bx = texture starting Ypos                   }
      shl bx,7           {bx = texture starting Ypos * 128             }
      add bl, [TexCol]  {bx = YTexPos * 128 + TexXpos = texture offset}
      mov si,bx          {si = texture offset                          }

      xor bx,bx          {clear bx register}
      mov bx,[TexPos]    {load bx with starting YTexPos                }
      shl bx, 8          {covert the starting YTexPos to 8.8 fixed-p   }
                         {this is because we wanna add a 8.8 fixed-p   }
                         {step value to this register -  things has to }
                         {fit together :)                              }
      mov cx, [height]   {cx = loop register - we loop until height is }
                         {reached}
     @again:
      mov al, ds:[si]    {load the color from the texture into al          }



      add bx, [ScaleStep]{add scalestep (8.8 FP) to the Ypos in the texture}
      xor dx,dx          {clear dx reg                                     }
      mov dl,bh          {bh = bx / 256 = actual value of Ypos in texture  }
                         {this is just a faster way of doing :             }
                         {        mov dx,bx                                }
                         {        shr dx,8                                 }

      shl dx,7           {dx = Ypos * 128                                  }
      add dl,[TexCol]   {add The Xpos in the texture found by Yray        }
      mov si,dx          {si = new offset in texture                       }


      mov es:[di], al    {and draw it to the screen                        }
      add di,320         {go 1 pixel down on the screen - add 320 to offset}

      dec cx             {decrease or counter register                     }
      cmp cx,0           {has it reaced zero yet ?                         }
      jne @again         {if not jump to the @again label and do another   }
                         {pixel                                            }

      pop ds             {when we're done remember to restore ds           }
     end;
 end;
END;




{*********************************************************}
{**                                                     **}
{**        FLOOR & CEILING RENDERING AND DRAWING        **}
{**                                                     **}
{*********************************************************}


PROCEDURE DoFloorCeiling(x,y : integer; PlayerA : integer);
VAR
 ViewAngle : integer;
 i,j       : integer;
 SinVal, CosVal, InvCosVal : longint;
 Ytop, Ybot : integer;
 distance : longint;
 xv,yv    : longint;
 Xmap, Ymap : integer;
 TexOfs : word;
 BotScrOfs, TopScrOfs : word;
 BotStartOfs, TopStartOfs : word;
 FloorAdd, CeilingAdd : word;


BEGIN
 ViewAngle := PlayerA - ANG_30; {start looking 30 degrees left from player}
 If (ViewAngle < ANG_0) then ViewAngle := viewangle + ANG_360;

 Flooradd := TexAddr[ACTIVE_FLOOR];
 Ceilingadd := TexAddr[CEILING];    {set up memory locations of the floor/ }
                                    {ceiling                               }

 BotStartOfs := (YBOTCLIP shl 8) + (YBOTCLIP shl 6) + SCREEN_X_START;
 TopStartOfs := (YTOPCLIP shl 8) + (YTOPCLIP shl 6) + SCREEN_X_START;
 {the starting offsets in our virtual screen}


 for i:=0 to SCREEN_SIZE do    {cast 320 rays...}
 begin
   SinVal    := Sin_Table^[ViewAngle];
   CosVal    := Cos_Table^[ViewAngle];
   InvCosVal := InvCos_Table[i]; {these only change with the column - not row}
   TopScrOfs := TopStartOfs;
   BotScrOfs := BotStartOfs;
   Ybot      := HORIZON + (DrawBuffer[i].height shr 1);
   YTop      := Ybot - DrawBuffer[i].height;

      for j := YBOTCLIP downto Ybot do
        begin
          distance := (FloorDist_Table[j] * InvCosVal) SHR 10;
          yv := ((distance * SinVal) SHR 20) + y;
          xv := ((distance * CosVal) SHR 20) + x;
           {the world coord of the pixel}

          asm
           MOV cx, ds                 {push ds                            }

           MOV ax, Vaddr
           MOV es, ax                 {es:[di] = target screen - vaddr    }
           MOV di, BotScrOfs          {di = offset to Floor-pixel         }
           MOV ax, FloorAdd
           MOV ds,ax                  {ds:[si] = texture space            }
           MOV ax, WORD ptr yv
           MOV bx, WORD ptr xv
           AND ax,63                  {AND the textureCoords to get them  }
           AND bx,63                  {from World-space to texture space  }
           ADD ax,ax
           ADD bx,bx                  {MUL by 2 because texture is 128X128}
           SHL ax,7
           ADD ax,bx
           MOV si, ax                 {si = offset in texture space       }
           MOV al, ds:[si]            {get floor color from floor texture }
           MOV bx, CeilingAdd
           MOV ds, bx                 {ds:[si] is now Ceiling texture     }

           MOV es:[di], al            {draw floor pixel                   }
           SUB BotScrOfs,320
           ADD TopScrOfs,320          {Move pixel positions in Vaddr}
           MOV al, ds:[si]            {Load Ceiling color - notice that si}
                                      {is the same for floor/ceiling      }
           MOV di, TopScrOfs
           MOV es:[di], al            {draw the ceiling pixel             }

           MOV ds,cx                  {pop ds                             }
          end;
        end;

  Inc(TopStartOfs);
  Inc(BotStartOfs);
  inc(ViewAngle);
  If (ViewAngle > ANG_360) then ViewAngle := ViewAngle - ANG_360;
 end;
END;



{***********************************************************}
{**                                                       **}
{**            RENDER WALLS AND FILL DRAW BUFFER          **}
{**                                                       **}
{***********************************************************}



PROCEDURE CalcView(x,y : integer; PlayerA : integer);
Var
 ViewAngle : integer;
 i,j : integer;
 XrayXhit, XrayYhit, YrayXhit, YrayYhit : word;
 xdist, ydist : longint;
 Xnr,Ynr : byte;
 XTexCol, YTexCol : byte;
 XReturnXMap,XReturnYmap : integer;
 YReturnXMap,YReturnYmap : integer;

 XLastXmap, XLastYmap : byte;
 LastSide : byte;
 StepValue, Position : longint;


BEGIN
ViewAngle := PlayerA - ANG_30; {start looking 30 degrees left from player}
If (ViewAngle < ANG_0) then ViewAngle := viewangle + ANG_360;


for i:=0 to SCREEN_SIZE do    {cast 320 rays...}
begin
 Xdist := 9050;  {set xdistance to a ridiculous length}
 Ydist := 9050;  {set Ydistance to a ridiculous length}
                 {9050 is max. dist in a 100 X 100 world of 64 X 64 cells}

 if(ViewAngle <> ANG_90) and (ViewAngle <> ANG_270) then
   begin {only fire X-ray if X wall can be found}
    Xnr := Xray(x,y,ViewAngle,XrayXhit,XrayYhit,XtexCol,XReturnXmap,XReturnYMap);
    if (Xnr <> 0) then
       Xdist := INTSQRT((XrayXhit - X)*(XrayXhit - X) +
                        (XrayYhit - Y)*(XrayYhit - Y));
   end;

  if(ViewAngle <> ANG_0) and (ViewAngle <> ANG_180) and (ViewAngle <> ANG_360) then
   begin {only fire Y-ray if Ywall can be found}
    Ynr := Yray(x,y,ViewAngle,YrayXhit, YrayYhit,YTexCol,YReturnXmap,YReturnYmap);
    if (Ynr <> 0) then
       Ydist := INTSQRT((YrayXhit - X)*(YrayXhit - X) +
                        (YrayYhit - Y)*(YrayYhit - Y));
   end;


   if (Ydist < xdist) then   {compare distance and save closest hit}
   begin
    DrawBuffer[i].dist   := Ydist;
    DrawBuffer[i].TexNr  := Ynr;
    DrawBuffer[i].TexCol := YTexCol;
    DrawBuffer[i].Xmap   := YReturnXmap;
    DrawBuffer[i].Ymap   := YReturnYMap;
    DrawBuffer[i].side   := 2;

   end
    else
   begin
    DrawBuffer[i].dist   := Xdist;
    DrawBuffer[i].TexNr  := Xnr;
    DrawBuffer[i].TexCol := XTexCol;
    DrawBuffer[i].Xmap   := XReturnXmap;
    DrawBuffer[i].Ymap   := XReturnYMap;
    DrawBuffer[i].side   := 1;
   end;

    Inc(ViewAngle);
    If (ViewAngle > ANG_360) then ViewAngle := ViewAngle - ANG_360;
  end;

 {take care of height calculations}
 {first check how many different map-tiles we have hit....}

XLastXmap := 255;
XLastYmap := 255;
XnewHeightPos := 0;   {set to impossible values to make sure column 0 is }
                      {recorded as a new wall                            }

  for i := 0 to SCREEN_SIZE do
   begin
     if (DrawBuffer[i].Xmap <> XLastXmap) or (DrawBuffer[i].Ymap <> XLastYmap) or
        (DrawBuffer[i].side <> LastSide)
      then {if new wall hit}
       begin
        XlastXmap := Drawbuffer[i].Xmap;
        XlastYmap := Drawbuffer[i].Ymap;
        LastSide  := DrawBuffer[i].side;  {update last-hit variables}

        XnewHeight[XnewHeightPos] := i;
        inc(XNewHeightPos);
        DrawBuffer[i].height :=
         HeightTable^[(DrawBuffer[i].dist*ScaleTable[i]) shr 10];

        {update XnewHeight according to the new wall found}

        if (i>0) then
        {calc last height in last wall - but only if there IS a last wall :) }
        DrawBuffer[i-1].height :=
         HeightTable^[(Drawbuffer[i-1].dist*ScaleTable[i-1]) shr 10];


       end;
   end;

 {deal with column SCREEN_SIZE / SCREEN_SIZE-1}
 {this is to make sure the last column is marked as exit for the last wall}

 XnewHeight[XnewHeightPos] := SCREEN_SIZE;
 inc(XnewHeightPos);
 DrawBuffer[SCREEN_SIZE].height :=
    HeightTable^[(Drawbuffer[SCREEN_SIZE].dist*ScaleTable[SCREEN_SIZE]) shr 10];
 DrawBuffer[SCREEN_SIZE-1].height :=
    HeightTable^[(Drawbuffer[SCREEN_SIZE-1].dist*ScaleTable[SCREEN_SIZE-1]) shr 10];


 {interpolate the heights}

 for i := 0 to XnewHeightPos-2 do
  begin
   if((XnewHeight[i+1] - XNewHeight[i]) > 2) then
    begin
     StepValue := ((DrawBuffer[XnewHeight[i+1]-1].height -
                   DrawBuffer[XNewHeight[i]].height));
     StepValue := (StepValue * 65536) DIV (XnewHeight[i+1]-1 - XnewHeight[i]);
     Position := DrawBuffer[XNewHeight[i]].height;
     Position := Position * 65536;

       for j := XnewHeight[i]+1 to XnewHeight[i+1]-2 do
        begin
          Position := Position + StepValue;
          DrawBuffer[j].height := Position shr 16;
        end;
    end;
  end;
END;


{**************************************************************}
{**                                                          **}
{**                   DOOR ROUTINES                          **}
{**                                                          **}
{**************************************************************}


{returns the door-number from the DoorArray - if map position is not}
{in the Array it returns 0                                          }
FUNCTION FindActiveDoor(X,Y : byte) : byte;
VAR
 i : byte;
 found : boolean;
BEGIN
 i := 0;
 Found := false;
 repeat
  inc(i);
  if (DoorArray[i].XmapPos = X) AND (DoorArray[i].YmapPos = Y) then
   begin
    found := true;
    FindActiveDoor := i;
   end;
 until (i = NumberOfActiveDoors) or (found = true);
if not(found) then FindActiveDoor := 0;
END;


{Checks if a door is in front of the player. If there IS a door and it's}
{not allready active we set it to ACTIVE                                }

PROCEDURE CheckDoor(x,y : integer; ang : integer);
VAR
 Xcheck, Ycheck : byte;
 XrayXhit, XrayYhit, YrayXhit, YrayYhit : word;
 XtexCol, YTexCol : byte;
 Xdist, Ydist : word;
 Doornr : byte;
 XReturnXmap,XReturnYmap : integer;
 YReturnXmap,YReturnYmap : integer;

BEGIN
 Xcheck := Xray(x,y,Ang,XrayXhit,XrayYhit,XtexCol,XReturnXmap,XReturnYmap);
 Ycheck := Yray(x,y,Ang,YrayXhit,YrayYhit,YtexCol,YReturnXmap,YReturnYmap);
 Xdist := INTSQRT((XrayXhit - X)*(XrayXhit - X) + (XrayYhit - Y)*(XrayYhit - Y));
 Ydist := INTSQRT((YrayXhit - X)*(YrayXhit - X) + (YrayYhit - Y)*(YrayYhit - Y));

 If (Xcheck = DOOR_CODE) AND (Xdist < GRID_SIZE DIV 2 + CLOSEST_WALL+10) then
  begin
   Doornr := FindActiveDoor(XmapPos,YmapPos);
      If(map.map[XReturnXmap,XReturnYmap] = DOOR_CODE) AND (DoorNr = 0) then
       begin  {activate door - but only new doors (DoorNr = 0)}
         inc(NumberOfActiveDoors);
         DoorArray[NumberOfActiveDoors].status := 1; {now opening}
         DoorArray[NumberOfActiveDoors].Door_offset := 0; {still closed}
         DoorArray[NumberOfActiveDoors].XmapPos := XReturnXmap;
         DoorArray[NumberOfActiveDoors].YmapPos := XReturnYmap;
         DoorArray[NumberOfActiveDoors].DoorType := DOOR_CODE;
       end;
  end;

 If (Ycheck = DOOR_CODE) AND (Ydist < GRID_SIZE DIV 2 + CLOSEST_WALL+10) then
  begin
   DoorNr := FindActiveDoor(XmapPos,YmapPos);
      If(map.map[YReturnXmap,YReturnYmap] = DOOR_CODE) AND (DoorNr = 0) then
       begin  {activate door}
         inc(NumberOfActiveDoors);
         DoorArray[NumberOfActiveDoors].status := 1;
         DoorArray[NumberOfActiveDoors].Door_offset := 0;
         DoorArray[NumberOfActiveDoors].XmapPos := YReturnXmap;
         DoorArray[NumberOfActiveDoors].YmapPos := YReturnYmap;
         DoorArray[NumberOfActiveDoors].DoorType := DOOR_CODE;
       end;
  end;
END;



{This procedure updates the doors in DoorArray }
PROCEDURE UpdateDoors;
VAR
 i,j : byte;

BEGIN
 for i := 1 to NumberOfActiveDoors do
   begin

    if (DoorArray[i].Door_offset = 128) AND (DoorArray[i].status = 1) then
     begin {door is open - initialize delay before close. And set map pos}
           {to 0 to allow player to walk through the door                }
      DoorArray[i].status := 0;
      DoorArray[i].delay := 150;
      Map.map[DoorArray[i].Xmappos,DoorArray[i].Ymappos] := 0;
     end;

    if (DoorArray[i].status = 0) then
     begin  {door is standing still}
      dec(DoorArray[i].delay);
      if (DoorArray[i].delay = 0) then
       begin {begin close door}

        DoorArray[i].status := 2;
        Map.Map[DoorArray[i].XmapPos, DoorArray[i].YMapPos] :=
                DoorArray[i].Doortype;

       end;
     end;

    if (DoorArray[i].status = 2) and (DoorArray[i].Door_offset = 0) then
     begin {door is closed.... remove from active list}

      for j := i to NumberOfActiveDoors do
            DoorArray[j] := DoorArray[j+1];

      Dec(i);  {so we'll check the new active door on current pos in array}
      Dec(NumberOfActiveDoors);
     end;


    {these two if-statements moves the door position by door-speed}
    if (DoorArray[i].status = 1) and (DoorArray[i].Door_offset < 128)
     then Inc(DoorArray[i].Door_offset,DOOR_SPEED);
    if (DoorArray[i].status = 2) and (DoorArray[i].Door_offset > 0)
     then Dec(DoorArray[i].Door_offset,DOOR_SPEED);
    end;
END;



{**********************************************************}
{**                                                      **}
{**                 MOVEMENT ROUTINES                    **}
{**                                                      **}
{**********************************************************}

{This is supposed to be a normal accelleration scheme :) Remember it's}
{tuned for MY computer though - as you might have different framerates on}
{your computer the effect might not be as nice on you computer - change }
{these numbers yourself to get a nice movement                          }
PROCEDURE MoveNorm(var Dx,Dy : integer);
Begin
if (turnflag = 1) AND (turn_value = 0) then turn_value := -1;
if (turnflag = 2) AND (turn_value = 0) then Turn_value := 1;
if (turnflag = 1) AND (turn_value < 0) AND (Turn_value > -ANG_5) then
   turn_value :=  turn_value * 1.2;  {if turning left and we have not yet}
                                     {reached maximum turn speed                }
if (turnflag = 2) AND (turn_value > 0) AND (Turn_value < ANG_5) then
   turn_value :=  turn_value * 1.2;  {if turning right and we have not   }
                                     {yet reached maximum speed          }

if (turnflag = 1) AND (turn_value > 0) then turn_value := turn_value-1;
if (turnflag = 2) AND (turn_value < 0) then turn_value := turn_value+1;
{if player pressed a direction opposite to the direction we are currently}
{turning we add the player pressure to the normal de-accelleration       }

if (turnflag = 0) AND (turn_value > 1) then turn_value := turn_value * 0.85;
if (turnflag = 0) AND (turn_value < -1) then Turn_value := turn_value * 0.85;
if (turnflag = 0) AND (turn_value < 1) AND (turn_value > -1) then
   turn_value := 0;
{de-accellerate the turn-rate. If between -1 and 1 set it to 0 }


if (turnflag = 1) AND (move_dir <> 0) then turn_value := -ANG_4;
if (turnflag = 2) AND (move_dir <> 0) then turn_value := ANG_4;
if (turnflag = 0) AND (move_dir <> 0) then turn_value := 0;
{if we're moving then turn with a constant speed}


{moving acceleration}
if (move_dir = 1) AND (acc = 0) then acc := 1.5;
if (move_dir = 2) AND (acc = 0) then acc := -1;
if (move_dir = 1) AND (acc > 0) AND (acc < 5) then
   acc := acc * 1.4;
if (move_dir = 2) AND (acc < 0) AND (acc > -5) then
   acc := acc * 1.4;

if (move_dir = 1) AND (acc < 0) then acc := acc + 0.40;
if (move_dir = 2) AND (acc > 0) then acc := acc - 0.40;
{if player change direction before acc = 0 then modify the acc based on
 the direction the player is trying to go}

if (move_dir = 0) AND (acc > 1) then acc := acc * 0.92;
if (move_dir = 0) AND (acc < -1) then acc := acc * 0.92;
if (move_dir = 0) AND (acc <= 1) AND (acc >= -1) then acc := 0;
{do the de-accelleration. Stop player when acc is between -1 and 1}


dx := Round(cos(2*pi * PlayerAng / ANG_360)*acc);
dy := Round(sin(2*pi * PlayerAng / ANG_360)*acc);
{calculate the dx and dy values based on the turn rate and accelleration}

PlayerAng := Round(PlayerAng + turn_value);
if (PlayerAng < ANG_0) then inc(PlayerAng,ANG_360);
if (PlayerAng > ANG_360) then dec(PlayerAng,ANG_360);
end;





{the Icy game physics. Not very commented - check out the normal settings}
{and the pxdtut8.txt                                                     }
Procedure MoveIce(var dx, dy : integer);
begin
if (turnflag = 1) AND (turn_value = 0) then turn_value := -1;
if (turnflag = 2) AND (turn_value = 0) then Turn_value := 1;
if (turnflag = 1) AND (turn_value < 0) AND (Turn_value > -ANG_5) then
   turn_value :=  turn_value * 1.1;
if (turnflag = 2) AND (turn_value > 0) AND (Turn_value < ANG_5) then
   turn_value :=  turn_value * 1.1;
if (turnflag = 1) AND (turn_value > 0) then turn_value := turn_value-0.2;
if (turnflag = 2) AND (turn_value < 0) then turn_value := turn_value+0.2;
if (turnflag = 0) AND (turn_value > 1) then turn_value := turn_value * 0.98;
if (turnflag = 0) AND (turn_value < -1) then Turn_value := turn_value * 0.98;
if (turnflag = 0) AND (turn_value < 1) AND (turn_value > -1) then
   turn_value := 0;

{if moving then turn with a constant speed}
if (turnflag = 1) AND (move_dir <> 0) then turn_value := -ANG_4;
if (turnflag = 2) AND (move_dir <> 0) then turn_value := ANG_4;
if (turnflag = 0) AND (move_dir <> 0) then turn_value := 0;

{moving acceleration}
if (move_dir = 1) AND (acc = 0) then acc := 1.5;
if (move_dir = 2) AND (acc = 0) then acc := -1;
if (move_dir = 1) AND (acc > 0) AND (acc < 5) AND
   (ABS(Ice_MoveAngle-PlayerAng) < ANG_30) then acc := acc * 1.1;
if (move_dir = 2) AND (acc < 0) AND (acc > -5) AND
   (ABS(Ice_MoveAngle-PlayerAng) < ANG_30) then acc := acc * 1.1;
if (move_dir = 1) AND (acc < 0) then acc := acc + 0.09;
if (move_dir = 2) AND (acc > 0) then acc := acc - 0.09;
{if player change direction before acc = 0 then modify the acc based on
 the direction the player is trying to go}
if (move_dir = 0) AND (acc > 1) then acc := acc * 0.99;
if (move_dir = 0) AND (acc < -1) then acc := acc * 0.99;
if (move_dir = 0) AND (acc <= 1) AND (acc >= -1) then acc := 0;
{do the de-accelleration. Stop player when acc is between -1 and 1}
if (acc = 0) then Ice_MoveAngle := PlayerAng;
{only allow change of moving direction when player stands still}

dx := Round(cos(2*pi * Ice_MoveAngle / ANG_360)*acc);
dy := Round(sin(2*pi * Ice_MoveAngle / ANG_360)*acc);

PlayerAng := Round(PlayerAng + turn_value);
if (PlayerAng < ANG_0) then inc(PlayerAng,ANG_360);
if (PlayerAng > ANG_360) then dec(PlayerAng,ANG_360);
end;




{************************************************************************}
{**                                                                    **}
{**                         MAIN PROGRAM                               **}
{**                                                                    **}
{************************************************************************}

begin
ClrScr;
Writeln('      ****************************************************************');
Writeln('      *                                                              *');
Writeln('      *             3D ENGINE - ADVANCED WOLFENSTEIN STYLE           *');
Writeln('      *                      by : Telemachos                         *');
Writeln('      *                                                              *');
Writeln('      ****************************************************************');
Writeln;
Writeln('      Hiya!');
Writeln('      Welcome to the Peroxide Programming Tips #8');
Writeln('      This one is on advanced raycasting.');
Writeln('      Not much to be said about this program. Controls :');
Writeln;
Writeln('      Movement  : Arrow keys');
Writeln('      Open door : Space');
Writeln('      Quit game : ESC');
Writeln('      In the last minute I decided to add a fun little feature in the');
Writeln('      sample program - namely the ICE effect! During play press ''I'' to');
Writeln('      toggle between normal and icy floor :)');
Writeln;
Writeln('      Start the program with ''PXDTUT8.EXE -nofloor'' to remove floor/ceil.');
Writeln;
Writeln('      Have fun ! ');
Writeln('      Telemachos^Peroxide                  Press a key...');
readkey;


quit := false;
Frames := 0;
turn_value := 0;
acc        := 0;
ACTIVE_FLOOR  := FLOOR;
ice := false;
if paramstr(1) ='-nofloor' then floorceil := false else floorceil := true;


CalcAngles;     {calculate angle values based on FOV and SCREEN_SIZE}
CalcTables;     {calculate all the look-up tables}


Assign(levelfile,'pxdtut8.map');
Reset(levelfile);
Read(levelfile,map);
Close(levelfile);         {load game level                       }

PlayerAng := ANG_270;     {we start looking directly to the north}
PlayerX := 416;
PlayerY := 352;           {set the initial player position }
turn_value := 0;          {we are NOT turning now :)...    }
move_dir := 0;            {...and NOT moving...            }


GetIntVec(9, @OLDKbdHandler);     {save the old BIOS keyboard handler }
SetIntVec(9, Addr(MyKbdHandler)); {and set up our own customized one..}

asm
 mov ax,13h
 int 10h
end;                      { switch to VGA-mode          }

GetMem (Scr2,64000);
vaddr := seg(Scr2^);      { set up our virtual screen   }

GetMem(Tex1,16384);
TexAddr[1] := Seg(Tex1^);
GetMem(Tex2,16384);
TexAddr[2] := Seg(Tex2^);
GetMem(Tex3,16384);
TexAddr[3] := Seg(Tex3^);
GetMem(Tex4,16384);
TexAddr[4] := Seg(Tex4^);
GetMem(Tex5,16384);
TexAddr[5] := Seg(Tex5^);
GetMem(Tex6,16384);
TexAddr[6] := Seg(Tex6^);  {floor}
GetMem(Tex7,16384);
TexAddr[7] := Seg(Tex7^);  {ice floor}
GetMem(Tex8,16384);
TexAddr[8] := Seg(Tex8^);  {ceiling}

{set up memory for our 8 textures}


Loadpal('pxdtut8.pal');   { Load the game palette       }
SetPal(255,0,0,0);        { black out palette entry 255 }

 For i:=1 to 4 DO
    If map.tilename[i] <> '' THEN
        LoadTexture(map.tilename[i],i);  {Load the textures}

LoadTexture('PXDTEX5.VGA',5);  {door}
LoadTexture('PXDTEX6.VGA',6);  {floor}
LoadTexture('PXDTEX7.VGA',7);  {ice-floor}
LoadTexture('PXDTEX8.VGA',8);  {ceiling}

Display('bkg.vga',vaddr,0,0);            {display frame around viewarea}

{**************************************************************}
{*               HERE IS THE INNER LOOP                       *}
{**************************************************************}



NumberOfActiveDoors := 0;
t1 := time;
repeat


CalcView(PlayerX,PlayerY,PlayerAng);       { calculate the view          }
if floorceil then
DoFloorCeiling(PlayerX,PlayerY,PlayerAng)  { render floor/ceilings       }
else
DoBackGround;                              {do floor / ceiling with constant}
                                           {colors                          }


DrawScreen;                                { draw the walls              }
Flip(vaddr,VGA);                           { flip it to VGA              }
inc(frames);
UpdateDoors;                               {Update status of active doors}


if not(ice) then
MoveNorm(dx,dy)                            {handle the move flags from the}
else                                       {keyboard interrupt            }
MoveIce(dx,dy);


XMapPos := (PlayerX) shr 6 + 1;
YMapPos := (PlayerY) shr 6 + 1;    {where on Map is player ?}

PlayerX := PlayerX + dx;
PlayerY := PlayerY + dy;           {move player}


{Clip player position to the walls and the CLOSEST_WALL constant }
{The idea is to check in the direction we are moving.. If we gets}
{too close to a wall - don't allow that move.                    }


 if (dx > 0) then {walking to the right}
   begin
     if ( map.map[(PlayerX + CLOSEST_WALL) shr 6 + 1][YmapPos] <> 0)
       then
        begin
          PlayerX := XmapPos shl 6 - CLOSEST_WALL;
          Ice_MoveAngle := PlayerAng;
        end;
   end;

 if (dx < 0) then {walking to the left}
  begin
    if (map.map[(PlayerX - CLOSEST_WALL) shr 6 + 1][YmapPos] <> 0)
     then
      begin
       PlayerX := (XmapPos-1) shl 6 + CLOSEST_WALL;
       Ice_MoveAngle := PlayerAng;
      end;
  end;


 if (dy > 0) then {walking down on map}
   begin
   if ( map.map[XmapPos][(PlayerY+CLOSEST_WALL) shr 6 + 1] <> 0)
     then
      begin
       PlayerY := YMapPos shl 6 - CLOSEST_WALL;
       Ice_MoveAngle := PlayerAng;
      end;
   end;

 if (dy < 0) then {walking up on map}
   begin
    if ( map.map[XMapPos][(PlayerY-CLOSEST_WALL) shr 6 + 1] <> 0) then
     begin
      PlayerY := (YmapPos-1) shl 6 + CLOSEST_WALL;
      Ice_MoveAngle := PlayerAng;
     end;
   end;


until quit;  {repeat until quit is set from the keyboard handler}

t1 := time - T1; {get time used in inner loop}


asm
 mov ax,03h
 int 10h
end;

SetIntVec(9, @OLDKbdHandler);  {restore the original keyboard handler}

Dispose(YnextTable);
Dispose(XnextTable);
Dispose(TanTable);
Dispose(Heighttable);
Dispose(Cos_Table);
Dispose(Sin_Table);             {dispose of out lookup tables}


FreeMem (Scr2,64000);  {release the memory we have allocated for the virtual}
                       {screen                                              }

Writeln('All done... ');
Writeln('CU in my next tutorial - Telemachos^PXD');
writeln;
writeln(frames, ' frames rendered...');
writeln(Frames/(T1/18.2):1:2,' Frames per second.');
readkey;

end.