/*-------------------------------------------------------------------------
	File    :	d:\source\tracker_release\isPlayer\isPlayer.cpp
    Author  :   Zyrax, Gnilk
	Orginal :	-- Along time ago --
	Descr   :	Player and sound mixer for beaver sweeper

	This file handles the direct sound code, and also some
	pattern effect stuff.

	to add special effects, look around line 389

--------------------------------------------------------------------------- 
    Todo [-:undone,+:inprogress,!:done]:
	

Changes: 

-- Date -- | -- Name ------- | -- Did what...                              
2005-10-23 | Gnilk           | English comments

---------------------------------------------------------------------------*/

#include <windows.h>
#include "isPlayer.h"

#pragma comment (lib,"dsound.lib")

#define TIMER_CALL_INTERVAL 10

struct InterpolatedEffect {
	float lerp_start, lerp_stop, lerp_cur;	
	DWORD effect_bias;	
};

struct ChannelPlayInfo {
	// Current pattern
	GTK_PATTERN *playpattern;
	char transpose;
	
	// Countdown & current pos
	DWORD tickstobeat, currentpos;
	DWORD ticksperbeat;

	// Last note
	short lastnote[ GTK_DEFAULT_PATCHANNELS ];

	// Interpolated effects
	InterpolatedEffect interpol_fx[256];
};

struct PlayInfo {
	// Module currently being played
	GTK *module;
	// Position in sequencer
	DWORD sequencerpos;
	// Song position in "ticks"
	DWORD tickcount;
	//
	ChannelPlayInfo cpi[ GTK_MAX_CHANNELS ];
	// mixtime 
	DWORD mixtime;
};

struct PrecalcedFrame {
	// FrameLength
	DWORD m_framelength;
	// Snapshot for player, and at what time it should be fetched
	PlayerStateSnapshot m_playersnapshot;
};


DWORD WINAPI AutoUpdateThread( LPVOID passed );

isPlayer::isPlayer( HWND hwnd, isSynth *synth )
{	
	m_playinfo = new PlayInfo;
	m_hwnd = hwnd;
	m_synth = synth;
	m_isplaying = FALSE;
	m_lastcpumeasure = 0;
	InitializeCriticalSection( &m_playerlock );	
	m_processevent = CreateEvent( 0,0,0, 0);

	
#ifndef PLAYER_MINI_EDITION

	m_puc = NULL;
	m_pause_reference = 0;

#else
	m_isprecalced = FALSE;

#endif

	// will initiate buffer sizes
	SetBPM(100);

	// allocate a queue with PSS (player snapshots) for synchronization
	m_allistlen = 200;
	m_allist = (PlayerStateSnapshot *) malloc( sizeof(PlayerStateSnapshot) * m_allistlen );
	memset( m_allist, 0x00, sizeof(PlayerStateSnapshot) * m_allistlen );
	m_last_backsnap.time = 1;
	

	// DirectSound..
	if (DS_OK != DirectSoundCreate8( NULL, &m_ds8, NULL )) {
		MessageBox( NULL, "Could not create dx8 object", "Error!", MB_OK );
	}	

	if (DS_OK != m_ds8->SetCooperativeLevel( m_hwnd, DSSCL_PRIORITY )) {
		MessageBox( NULL, "Could not set cooperative level", "Error!", MB_OK );
	}

	// ... 
	WAVEFORMATEX wf;
	wf.wFormatTag = WAVE_FORMAT_PCM; 
	wf.nChannels = 2;
	wf.wBitsPerSample = 16;
	wf.nSamplesPerSec = 44100;
	wf.nBlockAlign = wf.wBitsPerSample * wf.nChannels / 8;
	wf.nAvgBytesPerSec = wf.nSamplesPerSec * wf.nBlockAlign;
	wf.cbSize = 0;

	// ..
	DSBUFFERDESC dsbdesc;
	dsbdesc.dwSize = sizeof( DSBUFFERDESC );
	dsbdesc.dwFlags = DSBCAPS_GETCURRENTPOSITION2 + DSBCAPS_STICKYFOCUS;
	dsbdesc.dwBufferBytes = 44100 * 4 * 2;
	dsbdesc.dwReserved = 0;
	dsbdesc.lpwfxFormat = &wf;
	dsbdesc.guid3DAlgorithm = GUID_NULL;

	// ...
	if (DS_OK != m_ds8->CreateSoundBuffer( &dsbdesc, (LPDIRECTSOUNDBUFFER *) &m_sec8, NULL )) {
		MessageBox( NULL, "secbuf", "Error!", MB_OK );
	}

	// Better check how big thing we actually got
	DSBCAPS sbcaps;
	sbcaps.dwSize = sizeof( DSBCAPS );
	m_sec8->GetCaps( &sbcaps );
	m_buffersize = sbcaps.dwBufferBytes;

	// allocate synth buffers (yes, we are using floats)
	m_tempbuf_l = (float*) malloc( 8*65536 * sizeof(float) );	
	m_tempbuf_r = (float*) malloc( 8*65536 * sizeof(float) );

#ifndef PLAYER_MINI_EDITION

	// default latency, msec
	SetLatency( 40 );

	// empty secondary buffers
	LPVOID buf1, buf2;
	DWORD size1, size2;

	m_sec8->Lock( 0, 0, &buf1, &size1, &buf2, &size2, DSBLOCK_ENTIREBUFFER  );

	if ( NULL != buf1 )
		memset( buf1, 0x0, size1 );
	if ( NULL != buf2 )
		memset( buf2, 0x0, size2 );

	m_sec8->Unlock( buf1, size1, buf2, size2 );

	// start rocking
	m_sec8->SetCurrentPosition(0);

	if (DS_OK != m_sec8->Play( 0, 0, DSBPLAY_LOOPING )) {
		MessageBox( NULL, "Could not play buffer!", "Error!", MB_OK );
	}

	// dsound update thread
	m_stopthread = FALSE;
	m_firsttime = TRUE;	
	m_threadhandle = CreateThread( NULL, 0, &AutoUpdateThread, this, 0, &m_threadid );
	SetThreadPriority( m_threadhandle, THREAD_PRIORITY_TIME_CRITICAL );

#else

	// In small version it is nice with big latency
	SetLatency( 500 );
	m_hearingtick = 0;

#endif

	


	
}

isPlayer::~isPlayer()
{

#ifndef PLAYER_MINI_EDITION

	// stop it...
	m_sec8->Stop();	
	m_stopthread = TRUE;	
	WaitForSingleObject( m_threadhandle, INFINITE );
	DeleteCriticalSection( &m_playerlock );

#else

	if (TRUE == m_isplaying)
		Stop();

#endif



	m_ds8->Release();

	free( m_tempbuf_l );
	free( m_tempbuf_r );
	free( m_allist );
	delete m_playinfo;	
	CloseHandle( m_processevent );
}



void isPlayer::SetLatency( int latency )
{
	// Set write cursor position, more or less
	m_latencybytes = 4*44100 * latency / 1000;
	m_latencyms = latency;
	m_firsttime = TRUE;
}


void isPlayer::SetBPM( int bpm )
{
	// Update channel rendering block sizes
	m_synthbufsize = 60*44100 / ( 32 * bpm );
	m_synth->SetBufferSize( m_synthbufsize );
}

#ifndef PLAYER_MINI_EDITION

// Not part of "little" (release version)
void isPlayer::SetUpdateCallback( POSITIONUPDATECALLBACK puc, LPVOID paramtopass )
{
	m_puc = puc;
	m_pucparam = paramtopass;
}

// Gets the latency from the player, needed for editor
// only in "big" version
int isPlayer::GetLatency (void)
{
    return m_latencyms;
}


#endif


//////////////////////////////////////////////////////////////////////////
/**
	\brief do one tick of playing

	Execute a tick, here you may put new pattern and sequencer commands
	if you wish...
	
*/
void isPlayer::DoTick(void)
{
	
	BOOL posupdated = FALSE;

	///////////////////////////////////////////////////////////
	//				Uppdatera sequencern
	///////////////////////////////////////////////////////////

	//
	// have we just started a tick
	//
	if (!(m_playinfo->tickcount % (m_playinfo->module->def_seqtick) )) {
		
		for (int i=0;i<m_playinfo->module->num_channels;i++) {

			// get some pointers
			GTK_SEQCHANNEL *seqchan = &m_playinfo->module->channels[i];
			ChannelPlayInfo *cpi = &m_playinfo->cpi[i];

			// dont play outside stuff...
			if (m_playinfo->sequencerpos < seqchan->len) {

				// fetch
				GTK_SEQPOS *curpos = &seqchan->positions[ m_playinfo->sequencerpos ];

				// time to start a new pattern
				if (curpos->pattern) {
					// this pattern
					cpi->playpattern = &m_playinfo->module->patterns[ curpos->pattern ];
					cpi->transpose = curpos->transpose;
					// start right away
					cpi->tickstobeat = 0;
					cpi->ticksperbeat = m_playinfo->module->def_pattick;
					cpi->currentpos = 0;

					for (int i=0;i<256;i++) 
						cpi->interpol_fx[i].lerp_stop = cpi->interpol_fx[i].lerp_start = cpi->interpol_fx[i].lerp_cur;				

				}

				// check for interpolation of values
				if (curpos->fx) {
					cpi->interpol_fx[curpos->fx].lerp_start = curpos->param1;
					cpi->interpol_fx[curpos->fx].lerp_stop = curpos->param2;
					if (curpos->param1 == curpos->param2)
						cpi->interpol_fx[ curpos->fx ].effect_bias |= PLAYER_EFFECTBIAS_CHANGED;
				}

				if (curpos->param3) {
					cpi->interpol_fx[curpos->param3].lerp_start = curpos->param4;
					cpi->interpol_fx[curpos->param3].lerp_stop = curpos->param5;					
					if (curpos->param4 == curpos->param5)
						cpi->interpol_fx[ curpos->fx ].effect_bias |= PLAYER_EFFECTBIAS_CHANGED;
				}
				
				
			}
		}

		// move forward in sequencer
		m_playinfo->sequencerpos ++;
		posupdated = TRUE;
	}

	/////////////////////////////////////////////////////////////
	//			move forward in all patterns
	/////////////////////////////////////////////////////////////

	for (int i=0;i<m_playinfo->module->num_channels;i++) {
		// 
		ChannelPlayInfo *cpi = &m_playinfo->cpi[i];

		if (NULL != cpi->playpattern) {

			// time to step forward in this pattern
			if (cpi->tickstobeat <= 0) {
	
				if (cpi->currentpos < cpi->playpattern->len) {
	
					for (int c=0;c<GTK_DEFAULT_PATCHANNELS;c++) {
	
						GTK_PATTERNPOS *pp = &cpi->playpattern->patterndata[ cpi->currentpos ][ c ];
	
						isEVENT event;
						memset(&event,0,sizeof(isEVENT));
	
						// have a note
						if (pp->note) 
						{		
							if (pp->note != GTK_NOTE_OFF) 
							{
								if ( cpi->lastnote[c] ) 
								{
									event.type = NOTE_OFF;
									event.param1 = cpi->lastnote[c];
									event.param2 = pp->velocity;
									cpi->lastnote[c] = 0;									
									m_synth->Event( i, event );																		
								}							
								event.type = NOTE_ON;
								event.param1 = pp->note + cpi->transpose;
								event.param2 = pp->velocity;
								cpi->lastnote[c] = event.param1;							
							} else 
							{
								if ( cpi->lastnote[c] ) 
								{
									event.type = NOTE_OFF;
									event.param1 = cpi->lastnote[c];
									event.param2 = pp->velocity;
									cpi->lastnote[c] = 0;									
								}
							};	
							m_synth->Event( i, event );
						}

						//////////////////////////////////////////////////////////////////////////
						//
						// handle any pattern effects
						//
						if (pp->fx) {

							switch ((unsigned char)pp->fx) {

								case 0xff:
									cpi->ticksperbeat = pp->param;	
									break;
								default:
									cpi->interpol_fx[ pp->fx ].effect_bias = pp->param | PLAYER_EFFECTBIAS_CHANGED;
									break;
							}
						
						}


					}					
					// Reset counter, so it will start counting down tickperbeat again
					// This equals PatTick
					cpi->tickstobeat = cpi->ticksperbeat - 1;

					// Move forward in the pattern
					cpi->currentpos++;

					posupdated = TRUE;

				}

			} else 
				cpi->tickstobeat--;			

		// Send all effects thats need to be sent, here we send interpolated stuff
		// as well as thoose with "effect"-commands in patterns
		for (int q=0;q<256;q++)
			if ( (cpi->interpol_fx[q].effect_bias & PLAYER_EFFECTBIAS_CHANGED) || cpi->interpol_fx[q].lerp_start != cpi->interpol_fx[q].lerp_stop ) {

				float t = 1;

				cpi->interpol_fx[q].effect_bias &= 0xffffffff^PLAYER_EFFECTBIAS_CHANGED;

				if (cpi->playpattern && cpi->playpattern->len >= 0)
					t = (float)cpi->currentpos / (float)cpi->playpattern->len;

				isEVENT event;
				cpi->interpol_fx[q].lerp_cur = (float)cpi->interpol_fx[q].lerp_start + t * ((float)cpi->interpol_fx[q].lerp_stop - (float)cpi->interpol_fx[q].lerp_start);

				float w = cpi->interpol_fx[q].lerp_cur + cpi->interpol_fx[q].effect_bias;
				if (w > 255)
					w = 255;
				if (w < 0)
					w = 0;				
				event.type = CONTROLLER;
				event.param1 = q;
				event.param2 = (unsigned char)w;
				m_synth->Event( i, event ); 
			}

		}
	}

	m_playinfo->tickcount++;

	// Insert into latency list. This has is only interesting if not precalcing,
	// but wont hurt even if we are...
	if (TRUE == posupdated ) {
		// construct an entry
		PlayerStateSnapshot pss;
		pss.time = m_playinfo->mixtime + m_latencyms;
		pss.sequencerpos = m_playinfo->sequencerpos;
		for (int i=0;i<m_playinfo->module->num_channels;i++) {
			pss.currentpos[i] = m_playinfo->cpi[i].currentpos;
			pss.playpattern[i] = m_playinfo->cpi[i].playpattern;
		}

		// A pss is finished. Fix a free place for it
		for (i=0;i<m_allistlen;i++)
			if (!m_allist[i].time) {// empty?
				m_allist[i] = pss;
				break;
			};
	}

}

float isPlayer::GetCPUUsage(void)
{
	return m_lastcpumeasure;
}

#ifdef PLAYER_MINI_EDITION

LRESULT WINAPI ProgressWindowProc(HWND hWnd, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
  switch (iMessage)
  {

	case WM_CREATE:
		SetWindowLong( hWnd, 0, 0 );
		break;

	case WM_DESTROY:
		PostQuitMessage(0);
		return (FALSE);

	case WM_CLOSE:
		DestroyWindow( hWnd );
		return (FALSE);

	case WM_PAINT:
		{
			
			int progress = GetWindowLong( hWnd, 0 );
			PAINTSTRUCT paint;

			BeginPaint( hWnd, &paint );	
			RECT r;
			GetClientRect( hWnd, &r );

			HBRUSH brush1 = CreateSolidBrush( RGB(0,128,255) );
			HBRUSH brush2 = CreateSolidBrush( RGB(64,0,0) );
			
			RECT rect;
			rect = r;
			rect.right = r.left + (r.right - r.left) * progress / 100;
			FillRect( paint.hdc, &rect, brush1 );
			rect = r;
			rect.left = r.left + (r.right - r.left) * progress / 100;
			FillRect( paint.hdc, &rect, brush2 );

			DeleteObject( brush1 );
			DeleteObject( brush2 );
			
			EndPaint( hWnd, &paint );

			// Validate whole window, then we dont steal as much CPU
			ValidateRect( hWnd, &rect );
			return 0;
		}

  }
  return DefWindowProc (hWnd, iMessage, wParam, lParam); 
}


void isPlayer::PrecalcSong(GTK *playmodule, DWORD startseqpos, DWORD stopseqpos, BOOL showdialog,  int maxframes, PRECALCPROGRESS callback, void *callback_ptr)
{	
	WNDCLASS wnd;
	memset( &wnd, 0x00, sizeof( WNDCLASS ) );	
	wnd.hbrBackground = (HBRUSH)(COLOR_DESKTOP + 2);
	wnd.hCursor = LoadCursor(NULL,IDC_ARROW);
	wnd.hInstance = GetModuleHandle(0);
	wnd.lpfnWndProc = ProgressWindowProc;
	wnd.cbWndExtra = 4;
	wnd.lpszClassName = "PRGESS";
	// Register our nice window class
	RegisterClass(&wnd);			

	RECT bkgr;
	GetClientRect( GetDesktopWindow(), &bkgr );
	int sx = 300;
	int sy = 50;
	HWND hwnd;
	
	if (showdialog)
		hwnd = CreateWindow( "PRGESS", "", WS_VISIBLE, (bkgr.right-sx)/2, (bkgr.bottom-sy)/2, sx, sy, 0,0,GetModuleHandle(0), 0 );	
	else
		hwnd = 0;

	m_precalcframe = (PrecalcFrame*) malloc( sizeof(PrecalcFrame) * maxframes );
	m_precalcframes = 0;

	// Set default bpm
	SetBPM( playmodule->def_bpm );

	// setup default values
	m_playinfo->module = playmodule;
	m_playinfo->sequencerpos = startseqpos;
	m_playinfo->tickcount = 0;


	int smallbuffers = 100;
	char *buffer = 0;

	// Precalc until finished
	while (m_playinfo->sequencerpos < stopseqpos) {

		if (smallbuffers == 100) {
			buffer = (char*) malloc( m_synthbufsize * 4 * 100 );
			smallbuffers = 0;
		}

		m_precalcframe[ m_precalcframes ].pf_samples = (short*)&buffer[ smallbuffers++ * m_synthbufsize * 4 ];
		DoTick();
		m_synth->GetSamples( m_tempbuf_l, m_tempbuf_r );

		// Convert to correct frames
		for (DWORD i=0;i<m_synthbufsize;i++) {
			m_precalcframe[ m_precalcframes ].pf_samples[2*i] = (short) ((m_tempbuf_l[i] * 32767));
			m_precalcframe[ m_precalcframes ].pf_samples[2*i+1] = (short) ((m_tempbuf_r[i] * 32767));
		}

		// What are we listening at, right now!

		m_precalcframes++;
		if (m_precalcframes == maxframes) {
			MessageBox( NULL, "increase maxframes in your precalc!","lamer!",MB_OK);
		}

		if (m_precalcframes % 70 == 0) {
			char str[256];					
			int progress = m_playinfo->tickcount * 100 / ((stopseqpos-1) * playmodule->def_seqtick);

			if (callback)
				callback( callback_ptr, (float)m_playinfo->tickcount / (float)((stopseqpos-1) * playmodule->def_seqtick) );

			sprintf(str,"Making music... %d%% done", progress);
			if (hwnd) {
				SetWindowText( hwnd, str );
				SetWindowLong( hwnd, 0, progress );
				InvalidateRect( hwnd, 0, FALSE );
				UpdateWindow( hwnd );
			}
		}

		if (hwnd) {
			MSG msg;
			while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
				TranslateMessage(&msg);
				DispatchMessage(&msg);	
			}
		}

	}
	
	if (hwnd) {
		DestroyWindow( hwnd );
		MSG msg;
		while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);	
		}
	}

	// we shall all be finished
	m_isprecalced = TRUE;
	
}




#endif

void isPlayer::process(void)
{
	SetEvent( m_processevent );
}

void isPlayer::Start(GTK *playmodule, DWORD startseqpos )
{

#ifndef PLAYER_MINI_EDITION

	EnterCriticalSection( &m_playerlock );

	m_isplaying = TRUE;
	// Initialize
	m_playinfo->module = playmodule;
	m_playinfo->sequencerpos = startseqpos;
	m_playinfo->tickcount = 0;

	// Set bpm to default
	SetBPM( m_playinfo->module->def_bpm );

	// Reset everything
	memset( &m_playinfo->cpi, 0x00, GTK_MAX_CHANNELS * sizeof( ChannelPlayInfo ) );

	LeaveCriticalSection( &m_playerlock );


#else

	// If the tune is precalced, initialize stuff with respect to that
	if (TRUE == m_isprecalced) {

		m_currentframe = 0;
		m_hearingtick = startseqpos * playmodule->def_seqtick;

	} else {
		

		EnterCriticalSection( &m_playerlock );

		m_isplaying = TRUE;
		// Default...
		m_playinfo->module = playmodule;
		m_playinfo->sequencerpos = startseqpos;
		m_playinfo->tickcount = 0;
		SetBPM( m_playinfo->module->def_bpm );
		memset( &m_playinfo->cpi, 0x00, GTK_MAX_CHANNELS * sizeof( ChannelPlayInfo ) );


		LeaveCriticalSection( &m_playerlock );
		

	}

	// empty secondary buffer
	LPVOID buf1, buf2;
	DWORD size1, size2;

	m_sec8->Lock( 0, 0, &buf1, &size1, &buf2, &size2, DSBLOCK_ENTIREBUFFER  );

	if ( NULL != buf1 )
		memset( buf1, 0x0, size1 );
	if ( NULL != buf2 )
		memset( buf2, 0x0, size2 );

	m_sec8->Unlock( buf1, size1, buf2, size2 );


	// start plauing
	m_sec8->SetCurrentPosition(0);

	if (DS_OK != m_sec8->Play( 0, 0, DSBPLAY_LOOPING )) {
		MessageBox( NULL, "Could not play buffer!", "Error!", MB_OK );
	}

	// Create player update thread
	m_stopthread = FALSE;
	m_firsttime = TRUE;	
	m_threadhandle = CreateThread( NULL, 0, &AutoUpdateThread, this, 0, &m_threadid );
	SetThreadPriority( m_threadhandle, THREAD_PRIORITY_HIGHEST );

	Sleep(500);


#endif
	
}

void isPlayer::Stop(void)
{

#ifndef PLAYER_MINI_EDITION

	if (FALSE == m_isplaying)
		return;

	EnterCriticalSection( &m_playerlock );	

	// Make sure we stop everything, this does not really work...
	for (int i=0;i<m_playinfo->module->num_channels;i++) {
		ChannelPlayInfo *cpi = &m_playinfo->cpi[i];
		for (int c=0;c<GTK_DEFAULT_PATCHANNELS;c++) 
				if ( cpi->lastnote[c] ) {
					isEVENT event;
					event.type = NOTE_OFF;
					event.param1 = cpi->lastnote[c];					
					cpi->lastnote[c] = 0;									
					m_synth->Event( i, event );																		
				}							
	}

	LeaveCriticalSection( &m_playerlock );	

#else

	// stop thread, wait for settling done
	m_sec8->Stop();	
	m_stopthread = TRUE;	
	WaitForSingleObject( m_threadhandle, INFINITE );

#endif

	m_isplaying = FALSE;
}

void isPlayer::GetCurrentPosition( int activechannel, int &sequencerrow, int &pattern, int &patternrow )
{
	sequencerrow = 0;
	pattern = 0;
	patternrow = 0;

	if (FALSE == m_isplaying)
		return;

	// ..
	if (m_last_backsnap.time) {

		pattern = 1;
		sequencerrow = m_last_backsnap.sequencerpos - 1;
		patternrow = m_last_backsnap.currentpos[activechannel] - 1;

		for (int i=0;i<m_playinfo->module->num_patterns;i++)
			if (&m_playinfo->module->patterns[i] == m_last_backsnap.playpattern[activechannel]) {
				pattern = i;
				break;
			}	

		if (sequencerrow < 0)
			sequencerrow = 0;
	}
}

DWORD isPlayer::GetCurrentTick( void )
{
	if (FALSE == m_isplaying)
		return -1;

	return m_playinfo->tickcount;
}

#ifdef PLAYER_MINI_EDITION

void isPlayer::GetSyncData( int *tickcount, int *seqpos )
{
	*tickcount = m_hearingtick;
	*seqpos = m_hearingtick / m_playinfo->module->def_seqtick;
}

#endif


// ---------------------------------------------------------------------------
//
// Here is the sound server code and dsound feeding
//
// ---------------------------------------------------------------------------
void isPlayer::update( DWORD timedelta_ms )
{
	
	DWORD bytestowrite = 0;
	DWORD playpos, writepos;
	BOOL restore = FALSE;

	m_sec8->GetCurrentPosition( &playpos, &writepos );

	// Reset statistics ?
	if (TRUE == m_firsttime) {
		m_firsttime = FALSE;
		restore = TRUE;
		// update write positions, wont skew next round
		m_lastwritecursorpos = writepos;
	}

	if (FALSE == restore) {		
		if (writepos >= m_lastwritecursorpos)
			bytestowrite = writepos - m_lastwritecursorpos;
		else
			bytestowrite = writepos + m_buffersize - m_lastwritecursorpos;

		if (bytestowrite > m_buffersize)
			bytestowrite = 0;
	}

	// restore time ?
	if (TRUE == restore) {		
		m_lastownwritepos = writepos;	
		m_lastwritecursorpos = writepos;
		m_bytesbehind = m_latencybytes; // be a little before
	}

	m_lastwritecursorpos = writepos;
	m_bytesbehind += bytestowrite;
	


#ifdef PLAYER_MINI_EDITION

	if (TRUE == m_isprecalced) {

		// just stream boy...
		DWORD framestowrite = m_bytesbehind / (m_synthbufsize * 4);

		for (DWORD i=0;i<framestowrite;i++) {
			int frame = m_currentframe++;

			LPVOID buf1, buf2;
			DWORD size1, size2;

	
			// shall never happen, but you can never be sure..
			if (DS_OK != m_sec8->Lock( m_lastownwritepos, 4 * m_synthbufsize, &buf1, &size1, &buf2, &size2, 0)) {
				m_firsttime = TRUE;
				return;
			}	
	
			short *buf1_s = (short*) buf1;
			short *buf2_s = (short*) buf2;

			float *temp_l = m_tempbuf_l;
			float *temp_r = m_tempbuf_r;

			if (frame < m_precalcframes) {

				memcpy( buf1_s, &m_precalcframe[frame].pf_samples[ 0 ], size1 );
				memcpy( buf2_s, &m_precalcframe[frame].pf_samples[ size1 / 2 ], size2 );

			} else {

				memset( buf1_s, 0x00, size1 );
				memset( buf2_s, 0x00, size2 );
			}

			float temp = (float) frame;
			temp *= ((float)m_synthbufsize / 44100);
			temp -= (float)(m_latencyms * 0.001);
			temp /= ((float)m_synthbufsize / 44100);			
			if (temp<0) temp = 0;
			m_hearingtick = (int) temp;

	
			m_sec8->Unlock( buf1, size1, buf2, size2 );

			m_lastownwritepos = (m_lastownwritepos + 4 * m_synthbufsize ) % m_buffersize;
			m_bytesbehind -= m_synthbufsize * 4;
		}

	} else {

#endif


		// how many blocks ?
		DWORD blockstowrite = m_bytesbehind  / (m_synthbufsize * 4);
		
		LPVOID buf1, buf2;
		DWORD size1, size2;

		m_playinfo->mixtime = timeGetTime();

		while ( blockstowrite-- ) {
			
			if (TRUE == m_isplaying)
				DoTick();
			
			m_synth->GetSamples( m_tempbuf_l, m_tempbuf_r );	

			m_playinfo->mixtime += m_synthbufsize / 44100;

			// shall never happen
			if (DS_OK != m_sec8->Lock( m_lastownwritepos, 4 * m_synthbufsize, &buf1, &size1, &buf2, &size2, 0)) {
				m_firsttime = TRUE;
				return;
			}	
			
			short *buf1_s = (short*) buf1;
			short *buf2_s = (short*) buf2;
		
			float *temp_l = m_tempbuf_l;
			float *temp_r = m_tempbuf_r;
		
			DWORD i;
		
			for (i=0;i<(size1/4);i++) {
				buf1_s[2*i] = (short) ((*temp_l++) * 32767);
				buf1_s[2*i+1] = (short) ((*temp_r++) * 32767);
				};		
		
			for (i=0;i<(size2/4);i++) {
				buf2_s[2*i] = (short) ((*temp_l++) * 32767);
				buf2_s[2*i+1] = (short) ((*temp_r++) * 32767);
			}; 

			m_sec8->Unlock( buf1, size1, buf2, size2 );

#ifdef PLAYER_MINI_EDITION
			float temp = (float) m_playinfo->tickcount;
			temp *= ((float)m_synthbufsize / 44100);
			temp -= (float)(m_latencyms * 0.001);
			temp /= ((float)m_synthbufsize / 44100);			
			if (temp<0) temp = 0;
			m_hearingtick = (int) temp;
#endif

			m_lastownwritepos = (m_lastownwritepos + 4 * m_synthbufsize ) % m_buffersize;
			m_bytesbehind -= m_synthbufsize * 4;
		}

#ifdef PLAYER_MINI_EDITION
	}
#endif


}

#ifndef PLAYER_MINI_EDITION

float *isPlayer::GetLeftPlayingBuffer (DWORD *l)
{
  *l = m_synthbufsize;
  return m_tempbuf_l;
}
float *isPlayer::GetRightPlayingBuffer (DWORD *l)
{
  *l = m_synthbufsize;
  return m_tempbuf_r;
}

#endif

// -------------------------------------------
//
//   dsound update thread
//		can be executed with higher priority
//
// -------------------------------------------

DWORD WINAPI AutoUpdateThread( LPVOID passed )
{
	isPlayer *player = (isPlayer*) passed;

	DWORD lasttime = timeGetTime();	

	DWORD updatetime = 0;
	DWORD intervaltime = 0;
	
	do {

		DWORD ctime = timeGetTime();
		DWORD sincelast = ctime - lasttime;
		lasttime = ctime;		

		EnterCriticalSection( &player->m_playerlock );

		if (player->m_pause_reference<=0) {
			player->update( sincelast );
		}
			
		DWORD deltaupdatetime = timeGetTime() - ctime;
		updatetime += deltaupdatetime;
		intervaltime += sincelast;
		
		if (intervaltime > 100) {	
			player->m_lastcpumeasure = player->m_lastcpumeasure*0.9f+((float)updatetime / (float)intervaltime)*0.1f;
			intervaltime = 0;
			updatetime = 0;
		}		

		// unlock CSection when PUC shall be called, otherwise the player will crash
		// and might hang quite nice...
		LeaveCriticalSection( &player->m_playerlock );		

		//
		// in the little player we dont have a PUC
		//
		#ifndef PLAYER_MINI_EDITION
		
		// check events
		for (int i=0;i<player->m_allistlen;i++)
			if (player->m_allist[i].time && ctime >= player->m_allist[i].time) {
				player->m_last_backsnap = player->m_allist[i];
				player->m_allist[i].time = 0;		
				player->m_puc( player->m_pucparam );
			}

		#endif			

		WaitForSingleObject( player->m_processevent, TIMER_CALL_INTERVAL );

		
	} while (FALSE == player->m_stopthread);

	return 0;
}

#ifndef PLAYER_MINI_EDITION

void isPlayer::Pause(void)
{
	EnterCriticalSection( &m_playerlock );
	m_pause_reference++;
	LeaveCriticalSection( &m_playerlock );
}

void isPlayer::Resume(void)
{
	EnterCriticalSection( &m_playerlock );
	m_pause_reference--;
	LeaveCriticalSection( &m_playerlock );
}

#endif