/*
 * Copyright (C) 1998, 1999 Wolfgang Moser
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Commodore CBM 1581 floppy disk copy util for PC's, FDCLOWLV.C
 *
 * Redirected contact addresses:
 *   Wolfgang Moser <womo@mindless.com>
 *   Wolfgang Moser <womo@fairlight.to>
 *    http://www.dollar8000.de/womo
 *
 * Current workplace (up to Dezember 1999):
 *   Wolfgang Moser <womo@advm2.gm.fh-koeln.de>
 *    http://advm2.gm.fh-koeln.de/~womo
 *
 *
 * Basic informations from Dan Fandrich <dan@fch.wimsey.bc.ca>.
 *   His readme of the cbmfs-0.3 driver for Linux explained me, what the
 *   difference between a DOS formatted 800 kb disk and a CBM 1581 disk is.
 *   (check: http://vanbc.wimsey.com/~danf/software/)
 *
 *
 * Basic implementations by Ciriaco Garca de Celis <ciri@gui.uva.es>
 *   His util 765Debug, Version 5.0 is great for learning dma based
 *   direct floppy disk controller programming.
 *   (check: ftp://ftp.gui.uva.es/pub/pc/2m/765d50sr.zip)
 *
 * Check out for his floppy disk utils 2M and 2MGUI, the last words
 * in improving floppy disk storage capacity.
 *   http://www.gui.uva.es/2m, ftp://ftp.gui.uva.es/pub/pc/2m
 *
 *
 * For additional informations to FDC programming check:
 *     http://developer.intel.com/design/periphrl/datashts/290468.htm
 *   and get the intel 82078 CHMOS Single-Chip Floppy Disk Controller
 *   PDF document:
 *     http://www.intel.nl/design/periphrl/datashts/29047403.pdf
 *   National Semiconductor has also some pages about their PC
 *   compatible controllers:
 *     http://www.national.com/pf/DP/DP8473.html
 *     http://www.national.com/pf/DP/DP8477B.html
 *
 * Another good source for floppy disk controller programming information
 * are the linux kernal sources, you could have a look into:
 *     http://www.cs.utexas.edu/users/peterson/linux/include/linux/fdreg.h
 *     http://www.cs.utexas.edu/users/peterson/linux/drivers/block/floppy.c
 *
 * Now, that I started talking about Linux: Check out the most massive
 * attack against floppy disks ever made, Alain Knaff's fdutils:
 *     http://fdutils.linux.lu/
 *     http://alain.knaff.linux.lu/
 */


#include "fdclowlv.h"
/* later: #ifdef DJGPP
   unsigned char peekb(unsigned long Segment, unsigned short Offset){
      unsigned char ret;
      dosmemget((Segment<<4)+Offset,1,&ret);
      return ret;
      }

   void pokeb(unsigned long Segment, unsigned short Offset, unsigned char Value){
      dosmemput(&Value,1,(Segment<<4)+Offset);
      }
#endif
*/


#ifdef USE_TSC
#ifdef USE_P_RDTSC
// special: Pentium Time Stamp Counter read routine
// !!! doesn't work under MS EMM !!! (but with QEMM 7.04)
// reads only the lower dword

unsigned long int read_tsc(void){
	unsigned long int l,h;

	asm{
		db 0x0f, 0x31						// opcode:   rdtsc
		db 0x66; mov word ptr l, ax	// opcode:   mov l, eax
		db 0x66; mov word ptr h, dx   // opcode:	 mov h, eax
		}
	return l;
	}

unsigned long int diff_TSC(unsigned long int lastTSC, unsigned long int newTSC){
	unsigned long int diff;

	diff=newTSC-lastTSC;
   return diff;
	}
// special

#else
unsigned long int read_tsc(void){
	union{
		unsigned long int tsc;
		struct{
			signed short int l;
			signed short int h;
			} lh;
		}Llh;

	asm	pushf					// Save current interrupt flag
	asm	cli					// Disable interrupts

	// using system timer and ticks counter instead of rdtsc
	asm	mov	ax,	0x40
	asm	mov	es,	ax
	asm	mov	dx,	es:[0x6C]
		//	Llh.lh.h=peek(0x40, 0x6C);

	asm	mov	al,	0x00
	asm	out	0x43,	al		// Latch timer 0

	asm	in		al,	0x61	// Waste some time (waitstated)

	asm	in		al,	0x40	// Counter --> bx
	asm	mov	bl,	al		// LSB in BL

	asm	in		al,	0x61	// Waste some time (waitstated)

	asm	in		al,	0x40
	asm	mov	bh,	al		// MSB in BH
	asm	not	bx				// Need ascending counter

	asm	popf					// Restore interrupt flag


	asm   mov	cl,	15
	asm	mov	di,	bx    // If the MSB of the system timer
	asm	sar	di,	cl		// is set, then the second

	asm	and	dx,	di		// measured ticks counter value
	asm	not	di				// is selected, otherwise the

	asm	mov	ax,	0x40
	asm	mov	es,	ax
	asm	mov	cx,	es:[0x6C]
   		//	h=peek(0x40, 0x6C);
	asm	and	cx,	di		// first measured ticks counter
	asm	or		dx,	cx		// value is select as high word

	asm   mov	ax,	bx		// system timer value is low word

	Llh.lh.l=_BX;
	Llh.lh.h=_DX;

	return Llh.tsc;
	}

unsigned long int diff_TSC(unsigned long int lastTSC, unsigned long int newTSC){
	unsigned long int diff;

	diff=newTSC-lastTSC;
 	diff+=(diff&0x80000000L)>>15;	// undetected overflow fix
   return diff;
	}
#endif
static void print_Tz_since(void){
	static unsigned long int lastTsc;
	unsigned long int newTsc;

	newTsc=read_tsc();
	printf(" system clock cycles since last: %10ld\n", diff_TSC(lastTsc,newTsc));
	lastTsc=newTsc;
	}
// special
#endif

// CBM1581 floppy disk parameter set

// Linux: /etc/fdprm entry for cbm1581 filesystem driver
// #               size sec/t hds trk stre gap  rate spec1 fmt_gap
// cbm1581         1600    10   2  80    2 0x2A 0x02 0xDF     0x2E
//                                      ^
// Bit 2 of the stretch parameter controls on Linux systems, that
// the disk side numbers in the sector header ID fields are swapped.

// The cbmfs-0.3 GAP sizes are not perfect. The original CBM 1581
// format GAP size is: 0x23 and a better working read/write GAP size
// is 0x0C. I take these values as standards for the following
// default initialisation of the internal FDC parameter set table.

FDCparameterSet FDprm={
	  80,	// Cylinders:   80 cylinders per disk
	   2,	// Heads:        2 sides (tracks) per cylinder
	  10,	// Sectors:     10 sectors per track (side)
	   2,	// BpS:        512 bytes per sector
	   2,	// Rate:       250 kilobit per second
	   1,	// Modulation: MFM
	0xDF,	// SpecifyCMD
	0x0C,	// RWGAP3:      12
	0x23,	// FmtGAP:      35
	0x00	// Pattern:      0
	};

static uint8 FDCcommandString[9];

// fills up a 9 byte command sequence with the standard parameters
//
// input:  none
// output: none
//
static void prefill9cmd(void){
   FDCcommandString[5]=FDprm.BpS;
   FDCcommandString[6]=FDprm.Sectors;
   FDCcommandString[7]=FDprm.RWGAP3;
   FDCcommandString[8]=128;
	}


// waits until the FDC is ready (command phase)
//
// input:  a pointer to a byte, where the last status is stored to
//         this pointer can be NULL, too
// output: 1, if the FDC is ready for command transfers
//         0, if a timeout occured
//
int waitUntilReady4Transfer(uint8 *STAT){	// time out after 440 ms
	int   t,i=FDC_TIMEOUT;
   uint8 dummy;

   if(STAT==NULL) STAT=&dummy;
	do{
		t=peekb(0x40, 0x6C);
		while(t==peekb(0x40, 0x6C)){
			if((*STAT=inportb(FD_STATUS))&0x80) return 1;
			}
		}while (i-->0);
	return 0;
	}

// read a byte from the FDC data register
//
// input:  none
// output: 0...255: the correct byte from the FDC
//         -1:      a timeout occured (read operation)
//
int inFDC(void){
	uint8 status;
	return waitUntilReady4Transfer(&status)?inportb (FD_DATA):-1;
	}

// clears the FDC command registers, when a previous operation
// was aborted for some reason and the FDC couldn't return to
// the command phase
//
// input:  none
// output: none
//
void clearFDCresults(void){
	while((inportb(FD_STATUS)&0xC0)==0xC0){
		delay(2);
		inFDC();						// clear FDC data register,
		delay(2);					// get all return values
		}
   }

// write a byte to the FDC data register
//
// input:  the byte to write
// output:  1: operation was performed correctly
//         -1: a timeout occured (write operation)
//
int outFDC(uint8 data){
	uint8 status;

   do{
   	clearFDCresults();	// get old result bytes, if there are any
		if(!waitUntilReady4Transfer(&status)) return -2;	// FDC not responding (outFDC)
   	}while(status&0x40);
	outportb(FD_DATA, data);
	return 1;
	}

// prepare the DMA controller for the transfer to/from the FDC
//
// input:  mode: Bitfields for DMA channel 0-3 mode register (port 0x0B):
//
//				Bit(s)	Description
//
//					7-6  transfer mode		  5  direction
//             ----------------------------------------------------
//					 00 demand mode			  0 address increment select
//					 01 single mode			  1 address decrement select
//					 10 block mode
//					 11 cascade mode
//
//					3-2	operation			1-0  channel number
//             ----------------------------------------------------
//					 00 verify operation     00 channel 0 select
//					 01 write to memory      01 channel 1 select
//					 10 read from memory     10 channel 2 select
//					 11 reserved             11 channel 3 select
//
//         bytes:  the number of bytes, that have to be transferred
//         DMAbuf: the pointer to the DMA transfer buffer
// output: none
//
void prepareDMA(uint8 mode, uint16 bytes,
					 uint8 huge *DMAbuf){

	unsigned long	flat;
	unsigned short	page, offs;
/* later
#ifdef DJGPP
      // probably completely wrong
   flat=(long)DMAbuf;
#else
#endif
*/
		// normalize pointer
	flat= ((unsigned long)FP_SEG(DMAbuf)<<4)+FP_OFF(DMAbuf);
	page= flat>>16;
	offs= flat&0xFFFF;
		// decrement to the highest ("array") index number
	bytes--;

#ifdef DISABLE_INTS
disable();
#endif
	outportb (0x81, page);			// DMA channel 2 address byte 2
	outportb (0x0B, mode);			// DMA channel 0-3 mode register
	outportb (0x0C, 0);				// DMA clear byte pointer flip-flop
	outportb (0x04, offs&0xFF);   // DMA channel 2 address byte 0
	outportb (0x04, offs>>8);     // DMA channel 2 address byte 1
	outportb (0x05, bytes&0xFF);	// DMA channel 2 word count byte 0
	outportb (0x05, bytes>>8);		// DMA channel 2 word count byte 1
	outportb (0x0A, 2);				// clear mask bit of DMA channel 2
#ifdef DISABLE_INTS
enable();
#endif
	}

// wait until the FDC sends an IRQ
//
// input:  timeout	the number of timer ticks at least to wait for an IRQ
// output: none
//
void waitUntilInterrupt(int timeout){	// time out after 2 seconds
	int t;// ,i=OP_TIMEOUT;

	t=peekb(0x40, 0x6C);
	do{
		if(t!=peekb(0x40, 0x6C)){
			t=peekb(0x40,0x6C);
			if(timeout--<0) break;			// wait _at least_ the number of ticks
         }
		}while(!(peekb(0x40,0x3E)&0x80));
	pokeb(0x40,0x3E,peekb(0x40,0x3E)&0x7F);
	}

// switch the motor from the selected drive to on
//
// input:  the drive selected
// output: none
//
void switchMotorOn(uint8 drive){
	int i;
			// 0040h:0040h - DISKETTE - MOTOR TURN-OFF TIMEOUT COUNT
	pokeb(0x40,0x40,0xFF);
			// 0040h:003Fh - DISKETTE - MOTOR STATUS
	if(((i=peekb(0x40,0x3F))&(1<<drive))==0){
		outportb(FD_DOR,(1<<(drive+4))|(4+8)|drive);
		pokeb(0x40,0x3F,i|(1<<drive));
		delay(1000);
		pokeb(0x40,0x40,0xFF);
		}
	}

// switch the motors from all drives to off
//
// input:  none
// output: none
//
void switchMotorOff(void){
			// 0040h:0040h - DISKETTE - MOTOR TURN-OFF TIMEOUT COUNT
	pokeb(0x40,0x40,55);
	}

// send all specifications for the current drive to the FDC
//
// input:  none
// output: none
//
void specifyDensity(void){
	outportb(FD_DCR, FDprm.Bitrate);		// select bitrate (disk density)
	if(outFDC(3)<0 ||							// specify command
		outFDC(FDprm.SpecifyCMD)<0 ||		// head unload time, step rate time
		outFDC(0x02)<0							// head load time, switch to DMA mode
		);		// this is a trick: if one operation fails, the following are not done
	}

// selects a drive for the following operations
//
// input:  none
// output: none
//
void selectDrive(uint8 drive){
		// 0040h:0040h - DISKETTE - MOTOR TURN-OFF TIMEOUT COUNT
	pokeb(0x40,0x40,0xFF);
		// select drive, reset FDC, motor on
	outportb(FD_DOR,(1<<(drive+4))|(4+8)|drive);
	pokeb(0x40,0x3F,peekb(0x40,0x3F)|(1<<drive));

	specifyDensity();
	}

// issue a FDC command with DMA transfer (internal lib use only)
//
// input:  CMDlen:       the length of the command string
//         DMAmode:      the DMA mode to select for this command
//         DMAbytes:     the number of bytes to transfer with DMA
//         DMAbuffer:    the DMA transfer buffer
// output: 0 - timout while sending a command to the FDC
//         1 - operation was successful
static int issueCmd(uint8 CMDlen,
	uint8 DMAmode, uint16 DMAbytes, uint8 huge *DMAbuf){
	int i;

	selectDrive(FDCcommandString[1]&3);
	switchMotorOn(FDCcommandString[1]&3);			// set again timeout count


#if (MESSAGES >= 6)
	fprintf(stderr,"FDC command: 0x");
   for(i=0;i<CMDlen;i++) fprintf(stderr,"%02X.",FDCcommandString[i]);
   fprintf(stderr,"  DMA: 0x%02X, 0x%04X, 0x%05lX\n",DMAmode,DMAbytes,
			((unsigned long)FP_SEG(DMAbuf)<<4)+FP_OFF(DMAbuf));
//   printf("\n");
#endif


	prepareDMA(DMAmode, DMAbytes, DMAbuf);
   for(i=0;i<CMDlen;i++)
		if(outFDC(FDCcommandString[i])<0) return 0;
	waitUntilInterrupt(OP_TIMEOUT);
	return 1;
	}

// performs the sense interrupt status command
//
// input:  pointer to the result string structure
// output: 1 - operation was successful
//         0 - operation was unsuccessful
//    the result string holds ST0 and the Cylinder value
//
int senseInterruptStat(resultString *resStr){
	int res;

   if(resStr==NULL) return 0;

	for(res=0;res<8;res++) resStr->Status[res]=0;
	if(outFDC(8)>=0){				// sense interrupt status command
	   resStr->D.ST0=inFDC();
	   if(resStr->D.ST0>0){
		   res=inFDC();
		   if(res>0){
			   resStr->D.Cylinder=res;
			   return 1;
			   }
		   }
      }
	return 0;
	}

// performs a drive reset
//
// input:  the drive, that should be resetted
// output: none
//
void resetDrive(uint8 drive){
	resultString RS;

	pokeb(0x40,0x3E,peekb(0x40,0x3E)&0x7F);		// clear interrupt bit

	outportb(FD_DOR,(1<<(drive+4))|drive|8);		// reset
	delay(2);
	outportb(FD_DOR,(1<<(drive+4))|drive|(8+4));	// finish reset

	waitUntilInterrupt(OP_TIMEOUT);

	senseInterruptStat(&RS);
	}

// gets a seven byte result string from the FDC
//
// input:  pointer to the result string structure
// output: 1 - operation was successful
//         0 - operation was unsuccessful
//    the result string filled up with all the values
//
int valid7ResultString(resultString *resStr){
	int i,res;

   if(resStr==NULL) return 0;

	resStr->D.ST0=inFDC();
	if(resStr->D.ST0<0) return 0;

	for(i=2;i<8;i++){
		res=inFDC();
		if(res<0){
			resStr->D.ST0=-1;
			return 0;
			}
		else resStr->Status[i]=res;
		}
		// wont be needed, because ST0 should reflect the same
		//	if(resStr->D.ST1 || resStr->D.ST2) return 0;
		//
	res=resStr->D.ST0&0xC0;
	return !res;
	}

// prints all values from a result string to the console
//
// input:  pointer to the result string structure
// output: none
//
void printResultString(resultString *resStr){
	int  str, t, bit;
	static char *sterr[2][8]={{
		"MA (Missing address mark)",
		"NW (Non writable: write protect error)",
		"ND (No data)",
		"",
		"OR (Overrun)",
		"DE (Data error)",
		"",
		"EN (End of cylinder)",
		},{
		"MD (Missing address mark in data field",
		"BC (Bad cylinder)",
		"SN (Scan not satisfied)",
		"SE (Scan equal hit)",
		"WC (Wrong Cylinder)",
		"DD (Data Error in Data Field)",
		"CM (Control Mark)",
		""}};

   if(resStr==NULL) return;

	if(resStr->D.ST0<0) printf("[ST0 %4d]   FDC doesn't answer! ERROR!\n",resStr->D.ST0);
	else{
		printf("[ST0 0x%02X] [ST1 0x%02X] [ST2 0x%02X] [Cyl %3d] "
				 "[Side %1d] [Sec %2d] [Byte %3d] : ",
				 resStr->D.ST0, resStr->D.ST1, resStr->D.ST2,
				 resStr->D.Cylinder, resStr->D.Side,
				 resStr->D.Sector, resStr->D.BytesPS);
		resStr->D.ST1&=0xB7;
		resStr->D.ST2&=0x7F;

		if(resStr->D.ST0&0xC0)	printf("ERROR!\n");
		else							printf("OK\n");

		for(str=2; str<=3; str++) for(bit=0, t=1; bit<8; bit++, t<<=1){
			if(resStr->Status[str]&t)
				printf("ST%c(%c): %s\n",str+'0'-1,bit+'0',sterr[str-2][bit]);
			}
		}
	}

// performs a recalibration for the specified drive (seek to track 0)
//
// input:  drive:        the drive, that should be recalibrated
//         side:         the disk side (head), that should be selected
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *recalibrateDrive(uint8 drive, uint8 side){
	static resultString RS;
	int tries;

#ifdef USE_TSC
	fprintf(stderr,"%20s","reaclibrateDrive");
	print_Tz_since();
#endif

	selectDrive(drive);
	switchMotorOn(drive);			// set again timeout count

	resetDrive(drive);
//	resetDrive(drive);
	specifyDensity();

	for(tries=2;tries>0;tries--){
	   RS.D.ST0=-2;					// default: FDC not responding
		if(outFDC(0x07)<0 ||			// recalibrate command
			outFDC((((~side)&1)<<2)|drive)<0
			)return &RS;

		waitUntilInterrupt(OP_TIMEOUT);

		senseInterruptStat(&RS);

		if((RS.D.ST0&0xF0)==0x20 && RS.D.Cylinder==0u){
		   RS.D.ST0=-2;				// default: FDC not responding
			if(outFDC(0x04)<0 ||		// drive status command (read ST3)
				outFDC((((~side)&1)<<2)|drive)<0
				)return &RS;
			if(inFDC()&0x10) break;	// TRACK0 bit is set, that's ok
         }
		}
	return (tries>0)?NULL:&RS;
	}

// performs a seek for the specified drive
//
// input:  drive:        the drive, that should do the seek
//         cyl:          the cylinder to seek to
//         side:         the disk side (head), that should be selected
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *seekToCyl(uint8 drive, uint8 cyl, uint8 side){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","seekToTrack");
	print_Tz_since();
#endif

	selectDrive(drive);
	switchMotorOn(drive);			// set again timeout count

   RS.D.ST0=-2;						// default: FDC not responding
   if(outFDC(0x0F)<0 ||				// seek command
   	outFDC((((~side)&1)<<2)|drive)<0 ||
		outFDC(cyl)<0
		)return &RS;

	waitUntilInterrupt(OP_TIMEOUT);

	senseInterruptStat(&RS);
	if(RS.D.ST0&0xC0){
		delay(750);
		senseInterruptStat(&RS);
		}
	return((RS.D.ST0&0xF0)==0x20 && RS.D.Cylinder==cyl)?NULL:&RS;
	}

// get the disk status of the specified drive
//
// input:  drive:        the drive, that should do the seek
// output: 0 - there's no disk in the drive
//         1 - the old disk is in (some operations have been done before)
//         2 - a new disk was inserted into the drive
//
int senseDiskStatus(uint8 drive){
	switchMotorOn(drive);				// set again timeout count
   if(inportb(FD_DCR)&0x80){			// read "disk change" bit

		delay(2);
      switch(inportb(FD_STATUS)&0xC0){
      	case 0xC0:							// there are FDC results to read
				pokeb(0x40,0x3E,peekb(0x40,0x3E)&0x7F);	// clear interrupt (to be sure)
				seekToCyl(drive,1,0);		// clear "disk change" bit, this
				recalibrateDrive(drive,0);	// can only be done by doing a seek
					// if disk change bit is cleared ==> there's a new disk is in
	         if(!(inportb(FD_DCR)&0x80)) return 2;
					// issue readSectorID
         case 0x80:							// FDC ready, but no results
				recalibrateDrive(drive,0);	// go to a mostly formatted track
   				// issue a new readSectorID command
               // and don't wait for any results
				readSectorID(drive,0,NULL,-1);
      	}
        	// after executing readSectorID, or when the FDC
			// is busy (0b0xxxxxxx), the disk is assumed as "out"
     	return 0;
   	}
   return 1;	// disk change bit cleared ==> old disk is in
	}

// read the next possible sector header ID from the current cylinder
//
// input:  drive:        the drive to read from
//         side:         the disk side (head), that should be selected
//         header:       a pointer to a result structure
//         timeout:      the number of timer ticks to wait at least for a
//                       sector header (timeout<0 means to _never_ wait)
// output: header:       the result string pointed to filled up with the result
//         return:  0 - operation failed
//                  1 - operation was successful
//
int readSectorID(uint8 drive, uint8 side, resultString *header, int timeout){
#ifdef USE_TSC
	fprintf(stderr,"%20s","readSectorID");
	print_Tz_since();
#endif
	if(header==NULL && timeout>=0) return 0;
	selectDrive(drive);
	switchMotorOn(drive);			// set again timeout count

   if(header!=NULL) header->D.ST0=-2;	// default: FDC not responding
   if(outFDC(0x0A | (FDprm.Modulation<<6))<0 ||
		outFDC((((~side)&1)<<2)|drive)<0
		)return 0;

	if(timeout<0) return 1;			// no valid result, only command issueing

	waitUntilInterrupt(timeout);
	return valid7ResultString(header);
	}

// format a track (one side of a cylinder)
//
// input:  drive:        the drive to format a track at
//         cyl:          the cylinder number to be written into the headers
//         side:         the disk side number to be written into the headers
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *formatCBMtrack(uint8 drive, uint8 cyl, uint8 side){
	static resultString RS;
							// 4 ID bytes per sector
	uint8 FmtBuffer[MaxSectors * 4];		// MaxSectors is defined in CENTRAL.H
	int	i,j;


#ifdef USE_TSC
	fprintf(stderr,"%20s","formatC1581track");
	print_Tz_since();
#endif

	for(i=j=0;i<FDprm.Sectors;){				// fill up the ID bytes
   	i++;											// start with sector 1
		FmtBuffer[j++]=cyl;						// cylinder descriptor
#ifdef CreateMaverickDisk
		FmtBuffer[j++]=CreateMaverickDisk;	// head descriptor
#else
		FmtBuffer[j++]=side;						// head descriptor
#endif
		FmtBuffer[j++]=i;							// sector No.
		FmtBuffer[j++]=FDprm.BpS;
		}

   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x0D;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=FDprm.BpS;
   FDCcommandString[3]=FDprm.Sectors;
   FDCcommandString[4]=FDprm.FmtGAP;
   FDCcommandString[5]=FDprm.Pattern;

	if(!issueCmd(6,0x4A,FDprm.Sectors<<2,FmtBuffer)){
	   RS.D.ST0=-2;	// FDC not responding
      return &RS;
   	}
   else{
		return valid7ResultString(&RS)?NULL:&RS;
   	}
	}

// read a sector
//
// input:  drive:        the drive to read from
//         cyl:          the cylinder number to read from
//         side:         the disk side number to read from
//         sector:       the sector number to read
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to read the data in
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *readCBMsector(uint8 drive, uint8 cyl, uint8 side,
									 uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","readCBMsector");
	print_Tz_since();
#endif

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x06;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;

   RS.D.ST0=-2;	// default: FDC not responding
	if(issueCmd(9,0x46,128u<<FDprm.BpS,buffer) &&
			valid7ResultString(&RS)
		)return NULL;
   else return &RS;
	}

// write a sector
//
// input:  drive:        the drive to write to
//         cyl:          the cylinder number to write to
//         side:         the disk side number to write to
//         sector:       the sector number to write
//         sector:       the sector number to write
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to write the data from
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *writeCBMsector(uint8 drive, uint8 cyl, uint8 side,
									  uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","writeCBMsector");
	print_Tz_since();
#endif

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x05;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;

   RS.D.ST0=-2;	// default: FDC not responding
	if(issueCmd(9,0x4A,128u<<FDprm.BpS,buffer) &&
			valid7ResultString(&RS)
		)return NULL;
   else return &RS;
	}

#ifdef ScanEqual
// verify a sector
//
// input:  drive:        the drive to verify at
//         cyl:          the cylinder number to verify at
//         side:         the disk side number to verify at
//         sector:       the sector number to verify
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to verify the data with
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
resultString *verifyCBMsector(uint8 drive, uint8 cyl, uint8 side,
									   uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;
	unsigned char compbuf[TrBufSize/MaxSectors];
   int i;

#ifdef USE_TSC
	fprintf(stderr,"%20s","compareCBMsector");
	print_Tz_since();
#endif

	prefill9cmd();		// fill up bytes 5 to 8
#if 1
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x06;	// read
#else
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x11;	// scanEqual
#endif
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;

   RS.D.ST0=-2;	// default: FDC not responding

#if 1
	if(issueCmd(9,0x46,128u<<FDprm.BpS,compbuf) &&
#else
	if(issueCmd(9,0x42,128u<<FDprm.BpS,buffer) &&
//	if(issueCmd(9,0x42,128u,buffer) &&	// konstant _nur_ 128 Byte vergleichen
//    funktioniert alles nicht, irgendwas is an den Scan...-Rutinen faul!
//	if(issueCmd(9,0x4A,128u<<FDprm.BpS,buffer) &&
#endif
			valid7ResultString(&RS)
		){
//      for(i=(128u<<FDprm.BpS)-1;i>=0;i--) if(compbuf[i]!=buffer[i]){
		if(memcmp(compbuf,buffer,128u<<FDprm.BpS)!=0){
      	RS.D.ST2|=0x04;
      	return &RS;
      	}
		return NULL;
      }
   else return &RS;
	}
#else
// verify a sector
//
// input:  drive:        the drive to verify at
//         cyl:          the cylinder number to verify at
//         side:         the disk side number to verify at
//         sector:       the sector number to verify
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to verify the data with
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
resultString *verifyCBMsector(uint8 drive, uint8 cyl, uint8 side,
									   uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","verifyCBMsector");
	print_Tz_since();
#endif

   	// Let the FDC CRC check the written data
      // "Real" comparison cannot be done

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x06;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;

   RS.D.ST0=-2;	// default: FDC not responding
	if(issueCmd(9,0x42,128u<<FDprm.BpS,buffer) &&
			valid7ResultString(&RS)
		)return NULL;
   else return &RS;
	}
#endif

// read a track with the multiple sector reading feature of the FDC
//
// input:  drive:        the drive to read from
//         cyl:          the cylinder number to read from
//         side:         the disk side number to read from
//         sector:       the sector number to start reading from
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to read the data in
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *readCBMtrack(uint8 drive, uint8 cyl, uint8 side,
									uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","readCBMtrack");
	print_Tz_since();
#endif

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x06;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;		// start sector to read

   RS.D.ST0=-2;	// default: FDC not responding
	if(issueCmd(9,0x46,(128u << FDprm.BpS) * (FDprm.Sectors - sector + 1),
							 buffer + (128u << FDprm.BpS) * (sector - 1)) &&
			valid7ResultString(&RS)
		){	// second half of the track was read with success

		if(sector<=1) return NULL;	// whole track is already in
#ifdef USE_TSC
	fprintf(stderr,"%20s","readCBMtrack");
	print_Tz_since();
#endif
	   FDCcommandString[4]=1;		// begin with the first half of the track

      RS.D.ST0=-2;	// default: FDC not responding
	   if(issueCmd(9,0x46,(128u << FDprm.BpS) * (sector - 1),buffer) &&
			   valid7ResultString(&RS)
			)return NULL;	// first half of track was read with success
		}
   return &RS;	// error
	}

// write a track with the multiple sector reading feature of the FDC
//
// input:  drive:        the drive to write to
//         cyl:          the cylinder number to write to
//         side:         the disk side number to write to
//         sector:       the sector number to start writing from
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to write the data from
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *writeCBMtrack(uint8 drive, uint8 cyl, uint8 side,
									 uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","writeCBMtrack");
	print_Tz_since();
#endif

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x05;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;		// start sector to read

   RS.D.ST0=-2;	// default: FDC not responding
	if(issueCmd(9,0x4A,(128u << FDprm.BpS) * (FDprm.Sectors - sector + 1),
							 buffer + (128u << FDprm.BpS) * (sector - 1)) &&
			valid7ResultString(&RS)
		){	// second half of the track was read with success
		if(sector<=1) return NULL;	// whole track is already in
#ifdef USE_TSC
		fprintf(stderr,"%20s","writeCBMtrack");
		print_Tz_since();
#endif
	   FDCcommandString[4]=1;		// begin with the first half of the track

      RS.D.ST0=-2;	// default: FDC not responding
	   if(issueCmd(9,0x4A,(128u << FDprm.BpS) * (sector - 1),buffer) &&
			   valid7ResultString(&RS)
			){
			return NULL;	// first half of track was read with success
         }
		}
   return &RS;	// error
	}

#ifdef ScanEqual
// verify a track with the multiple sector reading feature of the FDC
//
// input:  drive:        the drive to read from
//         cyl:          the cylinder number to read from
//         side:         the disk side number to read from
//         sector:       the sector number to start verifying from
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to read the data in
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *verifyCBMtrack(uint8 drive, uint8 cyl, uint8 side,
									  uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;
	unsigned char compbuf[TrBufSize];
   int i;

#ifdef USE_TSC
	fprintf(stderr,"%20s","compareCBMtrack");
	print_Tz_since();
#endif

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x06;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;		// start sector to read

   RS.D.ST0=-2;	// default: FDC not responding
	if(!issueCmd(9,0x46,(128u << FDprm.BpS) * (FDprm.Sectors - sector + 1),
							 compbuf + (128u << FDprm.BpS) * (sector - 1)) ||
			!valid7ResultString(&RS)
		)return &RS;
   else if(sector>1){
#ifdef USE_TSC
	fprintf(stderr,"%20s","compareCBMtrack");
	print_Tz_since();
#endif
	   FDCcommandString[4]=1;		// begin with the first half of the track

      RS.D.ST0=-2;	// default: FDC not responding
	   if(!issueCmd(9,0x46,(128u << FDprm.BpS) * (sector - 1),compbuf) ||
			   !valid7ResultString(&RS)
			)return &RS;
      }

//	for(i=((128u<<FDprm.BpS)*FDprm.Sectors)-1;i>=0;i--) if(compbuf[i]!=buffer[i]){
	if(memcmp(compbuf,buffer,(128u<<FDprm.BpS)*FDprm.Sectors)!=0){
  	  	RS.D.ST2|=0x04;
   	return &RS;
     	}
	return NULL;	// first half of track was read with success
	}
#else
// verify a track with the multiple sector reading feature of the FDC
//
// input:  drive:        the drive to read from
//         cyl:          the cylinder number to read from
//         side:         the disk side number to read from
//         sector:       the sector number to start verifying from
//         sideID:       the disk side identifier can given explicitly
//         buffer:       a pointer to a buffer to read the data in
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *verifyCBMtrack(uint8 drive, uint8 cyl, uint8 side,
									  uint8 sector, uint8 sideID, uint8 huge *buffer){
	static resultString RS;

#ifdef USE_TSC
	fprintf(stderr,"%20s","verifyCBMtrack");
	print_Tz_since();
#endif
   	// Let the FDC CRC check the written data
      // "Real" comparison cannot be done

	prefill9cmd();		// fill up bytes 5 to 8
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0x06;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=cyl;
   FDCcommandString[3]=sideID;
   FDCcommandString[4]=sector;		// start sector to read

   RS.D.ST0=-2;	// default: FDC not responding
//	if(issueCmd(9,0x46,(128u << FDprm.BpS) * (FDprm.Sectors - sector + 1),
	if(issueCmd(9,0x42,(128u << FDprm.BpS) * (FDprm.Sectors - sector + 1),
							 buffer + (128u << FDprm.BpS) * (sector - 1)) &&
			valid7ResultString(&RS)
		){	// second half of the track was read with success
		if(sector<=1) return NULL;	// whole track is already in
#ifdef USE_TSC
	fprintf(stderr,"%20s","verifyCBMtrack");
	print_Tz_since();
#endif
	   FDCcommandString[4]=1;		// begin with the first half of the track

      RS.D.ST0=-2;	// default: FDC not responding
	   if(issueCmd(9,0x42,(128u << FDprm.BpS) * (sector - 1),buffer) &&
				valid7ResultString(&RS)
			)return NULL;	// first half of track was read with success
		}
   return &RS;	// error
	}
#endif

#ifdef FDCintel82078
// check if we are using an intel extended controller 82078
//
// input:  none
// output:  -1 (<0)      timeout or general error
//          0            standard NEC PD 765 compatible controller
//          0x01xx       extended controller (xx - stepping result)
//          0x02xx       82078 compatible controller (xx - stepping result)
//          0x03xx       fully 82078 compatible FDC (xx - stepping result)
//
short check4intelFDC(void){
	short res;

#ifdef USE_TSC
	fprintf(stderr,"%20s","check4intelFDC");
	print_Tz_since();
#endif
	if(outFDC(0x10)<0) return -1;	// issue version command
	if((res=inFDC())<0) return -1;
   if(res==0x90){						// this is an extended controller

		if(outFDC(0x18)<0) return -1;	// issue part id command
		if((res=inFDC())<0) return -1;

      if(res>=0x41 && res<=0x5F){
      		// do some more intel FDC compatibility tests
      	if(res==0x41) res|=0x0300;	// original (first stepping)
      	else          res|=0x0200;
      	}
      else res|=0x0100;
      return res;
   	}
   else return 0x0000;
   }

// format'n'write a track (one side of a cylinder)
//
// input:  drive:        the drive to format a track at
//         cyl:          the cylinder number to be written into the headers
//         side:         the disk side number to be written into the headers
//         buffer:       a pointer to a buffer to write the data from
// output: resultString: NULL  - operation was sucessful
//                       other - the filled up result structure
//
resultString *fnwriteCBMtrack(uint8 drive, uint8 cyl, uint8 side, uint8 huge *buffer){
	static resultString RS;
							// 4 ID bytes per sector
	uint8 FnWBuffer[MaxSectors * 4 + TrBufSize];		// MaxSectors and TrBufSize are defined in CENTRAL.H
	int	i,j,k,l;

#ifdef USE_TSC
	fprintf(stderr,"%20s","fnwriteC1581track");
	print_Tz_since();
#endif

	for(i=j=k=0;i<FDprm.Sectors;){	// fill up the ID bytes and encapsulated write data
   	i++;									// start with sector 1
		FnWBuffer[j++]=cyl;				// cylinder descriptor
		FnWBuffer[j++]=side;				// head descriptor
		FnWBuffer[j++]=i;					// sector No.
		FnWBuffer[j++]=FDprm.BpS;
      for(l=(128 << FDprm.BpS);l>0;l--) FnWBuffer[j++]=buffer[k++];
		}
   FDCcommandString[0]=(FDprm.Modulation<<6) | 0xAD;
   FDCcommandString[1]=(((~side)&1)<<2) | drive;
   FDCcommandString[2]=FDprm.BpS;
   FDCcommandString[3]=FDprm.Sectors;
   FDCcommandString[4]=FDprm.FmtGAP;
   FDCcommandString[5]=FDprm.Pattern;

	if(!issueCmd(6,0x4A,FDprm.Sectors*(4+(128 << FDprm.BpS)),FnWBuffer)){
	   RS.D.ST0=-2;	// FDC not responding
      return &RS;
   	}
   else{
		return valid7ResultString(&RS)?NULL:&RS;
   	}
	}
#endif
