/*
 * Copyright (C) 1998, 1999 Wolfgang Moser
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Commodore CBM 1581 floppy disk copy util for PC's, 1581HLVL.C
 *
 * Redirected contact addresses:
 *   Wolfgang Moser <womo@mindless.com>
 *   Wolfgang Moser <womo@fairlight.to>
 *    http://www.dollar8000.de/womo
 *
 * Current workplace (up to Dezember 1999):
 *   Wolfgang Moser <womo@advm2.gm.fh-koeln.de>
 *    http://advm2.gm.fh-koeln.de/~womo
 *
 *
 * Basic informations from Dan Fandrich <dan@fch.wimsey.bc.ca>.
 *   His readme of the cbmfs-0.3 driver for Linux explained me, what the
 *   difference between a DOS formatted 800 kb disk and a CBM 1581 disk is.
 *   (check: http://vanbc.wimsey.com/~danf/software/)
 *
 *
 * Basic implementations by Ciriaco Garca de Celis <ciri@gui.uva.es>
 *   His util 765Debug, Version 5.0 is great for learning dma based
 *   direct floppy disk controller programming.
 *   (check: ftp://ftp.gui.uva.es/pub/pc/2m/765d50sr.zip)
 *
 * Check out for his floppy disk utils 2M and 2MGUI, the last words
 * in improving floppy disk storage capacity.
 *   http://www.gui.uva.es/2m, ftp://ftp.gui.uva.es/pub/pc/2m
 *
 *
 * For additional informations to FDC programming check:
 *     http://developer.intel.com/design/periphrl/datashts/290468.htm
 *   and get the intel 82078 CHMOS Single-Chip Floppy Disk Controller
 *   PDF document:
 *     http://www.intel.nl/design/periphrl/datashts/29047403.pdf
 *   National Semiconductor has also some pages about their PC
 *   compatible controllers:
 *     http://www.national.com/pf/DP/DP8473.html
 *     http://www.national.com/pf/DP/DP8477B.html
 *
 * Another good source for floppy disk controller programming information
 * are the linux kernal sources, you could have a look into:
 *     http://www.cs.utexas.edu/users/peterson/linux/include/linux/fdreg.h
 *     http://www.cs.utexas.edu/users/peterson/linux/drivers/block/floppy.c
 *
 * Now, that I started talking about Linux: Check out the most massive
 * attack against floppy disks ever made, Alain Knaff's fdutils:
 *     http://fdutils.linux.lu/
 *     http://alain.knaff.linux.lu/
 */


#include "1581hlvl.h"
#include <stdio.h>
#include <conio.h>
#include <assert.h>

/*
	  80,	// Cylinders:   80 cylinders per disk
	   2,	// Heads:        2 sides (tracks) per cylinder
	  10,	// Sectors:     10 sectors per track (side)
	   2,	// BpS:        512 bytes per sector
	   2,	// Rate:       250 kilobit per second
	   1,	// Modulation: MFM
	0xDF,	// SpecifyCMD
	0x0C,	// RWGAP3:      12
	0x23,	// FmtGAP:      35
	0x00	// Pattern:      0
*/
static const FDCparameterSet FDprmSets[2]={
	{80,2,10,2,2,1,0xDF,0x0C,0x23,0x00},
//	{81,2,10,3,0,1,0xCF,0x0C,0x64,0xE5}		// GAP and Pattern are from the CMD FD2000 manual
   {81,2,10,3,0,1,0xCF,0x0C,0x64,0x00}		// GAP and Pattern are from the CMD FD2000 manual
	};

static char FilenameRetStr[261];	// output path and filename

// waits for a disk or a new disk in the specified drive
//
// input:  drive:		the drive, where a disk has to be inserted into
//         newDisk:  if not 0, a new disk is required (disk change)
// output: 1 - disk or new disk is in
//         0 - the wait has been aborted by the user (<ESC> pressed)
//
int waitForDiskChng(uint8 drive, int newDisk){
	int key=0x20,o,n;

	if(newDisk) printf("Please change the disk ...\n");
   else printf("Waiting for disk drive ready status, please insert a disk ...\n");

   o=-1;
	do{
   	if(key==0x00) while(kbhit()) getch();	// eat up special combos
	   do{
      	n=senseDiskStatus(drive);
         if(o!=n){
         	printf("\rDisk Status: ");
				switch(n){
	         	case 0: printf("disk is out     "); break;
   	         case 1: printf("(old) disk is in"); break;
      	      case 2: printf("new disk is in  ");
	         	}
            }
		   switch(n){
            case 1:						// (old) disk is in
         	   if(newDisk) break;	// if new disk insert needed, don't return
      	   case 2:						// new disk
					printf("\n");
         	   return 1;
      	   }
         o=n;
   	   }while(!kbhit());
      }while((key=getch())!=0x1B);
	printf("\n");
	return 0;
	}

// prints all values from a result string to the console
//
// input:  pointer to the result string structure
// output: none
//
void CBMprintStatus(resultString *resStr, const char *Job){
   if(resStr==NULL) return;

   if(resStr->D.ST1&0x02) Job="Write protect ";
   else if(Job==NULL) Job="";

	if(resStr->D.ST0&0xFF00){
   	if(resStr->D.ST0>=-2) printf("FDC doesn't answer (%sput timeout).\n",(resStr->D.ST0)==-1?"in":"out");
      else printf("Unknown FDC communication problem\n");
      }
	else printf("0x%02X,0x%02X,0x%02X,%s%s,%2d,%1d,%2d,%1d\n",
					resStr->D.ST0, resStr->D.ST1, resStr->D.ST2,
               Job,	resStr->D.ST0&0xC0?"ERROR":"OK",
					resStr->D.Cylinder+1, resStr->D.Side,
					resStr->D.Sector, resStr->D.BytesPS);
	}

// reset and recalibrate the specified drive
//
// input:  drive:		the drive, that shall be resetted
// output: 1 - the operation was successful
//         0 - the operation was unsucessful
//
int CBMresetDrive(uint8 drive){
	resetDrive(drive);
	return recalibrateDrive(drive,0)==NULL;
	}

// recalibrate the specified drive
//
// input:  drive:		the drive, that shall be recalibrated
// output: 1 - the operation was successful
//         0 - the operation was unsucessful
//
int CBMrecalibrate(uint8 drive){
	resultString *RES;
   uint i;

   for(i=0;i<3;i++){
		RES=recalibrateDrive(drive,0);
		if(RES==NULL) break;
		CBMprintStatus(RES,"Recalibrate ");
      }

	if(RES!=NULL && !CBMresetDrive(drive)){
		printf("Unsuccessful attempt to recalibrate drive %d\n",drive);
//		CBMprintStatus(RES);
		return 0;
		}
	return 1;
	}

// perform a read/write head seek of the specified drive
//
// input:  drive:		the drive, where the seek shall be done
//         track:		the cylinder number, where the seek shall be done
// output: 1 - the operation was successful
//         0 - the operation was unsucessful
//
int CBMseek(uint8 drive, uint8 cyl){
	resultString *RES; //,RS;
	uint8 t=cyl-1;
	uint retry=3;

	while((RES=seekToCyl(drive,t,0))!=NULL){
		CBMprintStatus(RES,"Seek ");
		if(retry--<=0 || !CBMrecalibrate(drive)){
			printf("Unsuccessful attempt to seek to track %2d on drive %d\n",cyl,drive);
//			CBMprintStatus(RES);
			return 0;
			}
		}
	return 1;
	}

// sets the floppy disk parameter table to the selected type
//
// input:  Set:	the name (enum constant) of the predefined parameter set
// output: none
//
void CBMpresetFDprm(uint8 drive, FDprmSet Set){
	if(Set<CBM1581 || Set>=FDUNDEF) return;		// unknown type, don't change anything

	assert(FDprmSets[Set].Sectors<=MaxSectors);
		// increase sector buffer parameter via CENTRAL.H, if assertion fault
	assert((128u << FDprmSets[Set].BpS)*FDprmSets[Set].Sectors<=TrBufSize);
		// increase track buffer size via CENTRAL.H, if assertion fault

	FDprm=FDprmSets[Set];
	CBMrecalibrate(drive);
	}


// checks, if the inserted disk is a CBM formatted disk and detects the format
//
// input:  drive:		the drive, where the disk to be checked is in
//         chkSwap:  0 - don't check for swapped side IDs
//                   other - check, if sectors on head 1 (side 0)
//                           contain sector IDs that are 0
// output: NON_3_5 -	the drive type is no 3,5" type
//         CBM1581 - the disk was detected as CBM 1581 disk format compatible
//         CMDFD2M - the disk was detected as CMD FD2000 disk format compatible
//         FDUNDEF - the disk was not detected as a CBM format type
//
FDprmSet	CBMcheckDisk(uint8 drive, int chkSwap){
// FDprmSet CBMcheckDisk(uint8 drive){
	resultString RS;
	FDCparameterSet oldFDprm=FDprm;	// save current parameter set

   FDprmSet	PSet;
	char		secMap[MaxSectors];		// MaxSectors is defined in CENTRAL.H

   uint8 	wrongID,type,doBR=0xFF;
	int   	i,tmo;

   	// do a CMOS check, if it is a 3,5" drive
      // if not, return -1 (can never be a "correct" disk)
   disable();
	outportb(0x70, 0x10);	// CMOS address 0x10
	type=inportb(0x71);		// read addressed CMOS value
   enable();
   switch(drive){
   	case 0:	type>>=4;	break;	// test upper nibble
      case 1:	type&=0x0F;	break;	// test lower nibble
      default:	type=0x04;				// assume 1,44 MB drive
   	}

   if(type<3 || type>5) return NON_3_5;	// can't support these disk types (5,25")
   if(type==3) doBR&=~1;		// 500 kbps are not supported on 720 kB disk drives

   for(PSet=FDUNDEF-1;PSet>FDUNKNW;PSet--){
   	if(doBR&(1<<FDprmSets[PSet].Bitrate)){			// if the wished bitrate is supported
			CBMpresetFDprm(drive,PSet);					// Set bitrate, modulation and specifyCMD
			if(CBMrecalibrate(drive)){	// track 0 should be selected
            for(i=0;i<FDprm.Sectors;i++) secMap[i]=0;	// delete sector detection map
            wrongID=0;


               // wait for more than 4 timer ticks (222 ms, a disk round)
   	      i=5;
			   tmo=peekb(0x40, 0x6C);
            while(readSectorID(drive,0,&RS,1)){
				   if(tmo!=peekb(0x40, 0x6C)){
					   tmo=peekb(0x40,0x6C);
                  if(i--<0) break;
                  }
           	   if(RS.D.Cylinder!=0) wrongID|=1;	// TrackHeader wrong
           	   if(chkSwap && RS.D.Side!=0)     wrongID|=2;	// SideHeader wrong
				   if(RS.D.Sector<1 || RS.D.Sector>FDprm.Sectors)
						                  wrongID|=4;	// SectorHeader wrong
				   else secMap[RS.D.Sector-1]=1;		// mark sector read
           	   if(RS.D.BytesPS!=FDprm.BpS)
											   wrongID|=8;	// BpSHeader wrong
               }
            if(i>=5){	// readSectorID timeout happened
					   // deselect the desired bitrate
#if (MESSAGES >= 5)
					fprintf(stderr,"Wrong bitrate, deselecting %d\n",FDprmSets[PSet].Bitrate);
#endif
			      doBR&=~(1<<FDprmSets[PSet].Bitrate);
					   // reset controller without recalibrating again
				   resetDrive(drive);
               }
			   else{
		         for(i=0;i<FDprm.Sectors;i++) if(!secMap[i]) break;
   		      if(!wrongID && i>=FDprm.Sectors) break;	// format is detected
#if (MESSAGES >= 5)
					fprintf(stderr,"Format is not detected: 0x%02X, %d of %d\n",wrongID,i,FDprm.Sectors);
#endif
				   }
   			}
			}
      }
	FDprm=oldFDprm;		// restore parameter set (should be drive related)
	CBMrecalibrate(drive);
   return PSet;
	}

/*
// The algorithm above is not very well optimized, especially not, if you
// want to decide between much more disk formats.
// If there are e.g. two formats, that are very similar together, then
// this routine will need too much time for detection, because it scans
// the disk track for each format again and again.
//
// It would be much better to first detect the drive type (3,5" or 5.25")
// from the CMOS entry, so that standard parameters can be specified for
// each bitrate. Then the right bitrate and modulation should be detected
// and if it can be found, the track scan should be done by filling up a
// table, that holds the informations read from each sector header.
// After this, the drive type, bitrate and table entries should be
// compared against the different format specification entries.
// If no format could be detected, the rest of the untested bitrates and
// modulations has to be done (A priority table with the bitrates and
// modulation values would help minimizing the detection time).
//
// This way we have to scan a disk track only once (except, if two or more
// bitrates are detected on the same track).
//
// The algorithm below should work much better, when supporting all the
// different formats is needed, but I figured out, that it is very lame.


// checks, if the inserted disk is a CBM formatted disk and detects the format
//
// input:  drive:		the drive, where the disk to be checked is in
// output: NON_3_5 -	the drive type is no 3,5" type
//         CBM1581 - the disk was detected as CBM 1581 disk format compatible
//         CMDFD2M - the disk was detected as CMD FD2000 disk format compatible
//         FDUNDEF - the disk was not detected as a CBM format type
//
FDprmSet CBMcheckDisk(uint8 drive){
	resultString RS;
	FDCparameterSet oldFDprm=FDprm;	// save current parameter set
   uint8 Rates[4]={2,0,3,1};	// bitrates 250, 500, 1000/125 and 300 kbps

   FDprmSet	PSet;
	char		secMap[MaxSectors];		// MaxSectors is defined in CENTRAL.H

   uint8 	wrongID;
	int   	i,tmo,Rate,Mdltn;

   	// do a CMOS check, if it is a 3,5" drive
      // if not, return -1 (can never be a "correct" disk)
   disable();
	outportb(0x70, 0x10);	// CMOS address 0x10
	wrongID=inportb(0x71);		// read addressed CMOS value
   enable();
   switch(drive){
   	case 0:	wrongID>>=4;	break;	// test upper nibble
      case 1:	wrongID&=0x0F;	break;	// test lower nibble
      default:	wrongID=0x04;				// assume 1,44 MB drive
   	}

   if(wrongID<3 || wrongID>5) return NON_3_5;	// can't support these disk types (5,25")

	FDprm=FDprmSets[0];		// load CBM 1581 parameter set as standard values
   for(Mdltn=0;Mdltn<2;Mdltn++){
		FDprm.Modulation=(~Mdltn)&1;              // test MFM first, then FM
   	for(Rate=0;Rate<4;Rate++){
         FDprm.Bitrate=Rates[Rate];					// 250, 500, 1000/125, 300
         FDprm.Heads=0;
         FDprm.BpS=0;
         FDprm.Sectors=0;
			if(CBMrecalibrate(drive)){
            for(i=0;i<MaxSectors;i++) secMap[i]=0;	// delete sector detection map
            wrongID=0;
               // scan a track
               // wait for more than 4 timer ticks (222 ms, a disk round)
   	      i=5;
			   tmo=peekb(0x40, 0x6C);
            while(readSectorID(drive,0,&RS,1)){
				   if(tmo!=peekb(0x40, 0x6C)){
					   tmo=peekb(0x40,0x6C);
                  if(i--<0) break;
                  }
               if(!FDprm.Sectors++){      // first sector read
               	FDprm.Heads=RS.D.Side;	// all side and BpS ID fields of the
               	FDprm.BpS=RS.D.BytesPS;	// whole track should be the same
               	}

           	   if(RS.D.Cylinder!=0) wrongID|=1;	// TrackHeader wrong
           	   if(RS.D.Side!=FDprm.Heads)
												wrongID|=2;	// SideHeader wrong
				   if(RS.D.Sector<1 || RS.D.Sector>MaxSectors)
						                  wrongID|=4;	// SectorHeader not supported
				   else secMap[RS.D.Sector-1]=1;		// mark sector read
           	   if(RS.D.BytesPS!=FDprm.BpS)
											   wrongID|=8;	// BpSHeader wrong
               }
            if(i>=5){	// readSectorID timeout happened
					   // deselect the desired bitrate
#if (MESSAGES >= 5)
					fprintf(stderr,"No detection with modulation %d and bitrate %d\n",
               					 FDprm.Modulation,FDprm.Bitrate);
					fprintf(stderr,"Timeout values: tmo %d, act: %d\n",tmo,peekb(0x40, 0x6C));
#endif
					   // reset controller without recalibrating again
				   resetDrive(drive);
               }
			   else{
            	if(!wrongID){	// headers are not "symmetric"
            		   // check the read track against the supported formats
				      for(PSet=FDUNDEF-1;PSet>FDUNKNW;PSet--){
               		   // compare the selected bitrate against the specified
               	   if(FDprmSets[PSet].Bitrate!=Rates[Rate]) continue;
               		   // compare the selected modulation against the specified
               	   if(FDprmSets[PSet].Modulation!=FDprm.Modulation) continue;
               		   // compare the read side (must be 0/swapped)
               	   if(FDprm.Heads!=0) continue;
               		   // compare the read BpS against the specified
               	   if(FDprmSets[PSet].BpS!=FDprm.BpS) continue;
                     	// check the sectors
				         for(i=0;i<FDprmSets[PSet].Sectors;i++){
								if(!secMap[i]) break;
                        }
                     if(i<FDprmSets[PSet].Sectors) continue;
                     	// check, if there are not more marked sectors
                     for(;i<MaxSectors;i++){
								if(secMap[i]) break;
                     	}
                     if(i<MaxSectors) continue;

                     	// all tests are successed, this seems
								// to be the right format
							FDprm=oldFDprm;		// restore parameter set (should be drive related)
							CBMrecalibrate(drive);
						   return PSet;
               	   }
                  }
#if (MESSAGES >= 5)
					fprintf(stderr,"Current track format is not supported 0x%02X %d %d %d %d\n",
							 wrongID,FDprm.Modulation,FDprm.Bitrate,FDprm.Heads,FDprm.BpS);
					fprintf(stderr,"Read sector numbers:");
               for(i=0;i<MaxSectors;i++) if(secMap[i]){
						fprintf(stderr,"%d, ",i+1);
                  }
					fprintf(stderr,"\n");
#endif
               }
         	}
      	}
   	}

	FDprm=oldFDprm;		// restore parameter set (should be drive related)
	CBMrecalibrate(drive);
   return FDUNKNW;
	}
*/

// return the sector number of the next available sector header
//
// input:  drive:		the drive, where the disk to read is in
//         cyl:		the cylinder number of the track to read
//         side:		the disk side number of the track to read
//         tries:		the number of retries for this operation
//         secID:    a pointer to the sector ID result
//         sideID:   a pointer to the side ID result
// output: 1						- the operation was successful
//         0						- the operation was unsuccessful
//        -1 						- a possibly not CBM compatible disk was detected
//        -2 						- the operation was aborted by user interaction
static int CBMPreRWV(uint8 drive, uint8 cyl, uint8 side, uint tries,
							uint8 *secID, uint8 *sideID){
	resultString RS, *RES;
	uint retr1, retr2, retr3, retr4, retr5, op;

   if(secID==NULL || sideID==NULL) return 0;
	op=255;
	retr1=retr2=retr3=retr4=retr5=tries;

	while(retr1>0 && retr2>0 && retr3>0 && retr4>0 &&retr5>0){
   	if(kbhit()){
      	switch(getch()){
            case 0x1B: return -2;
         	case 0x00: while(kbhit()) getch();	// eat up special combos
         	}
      	}

		do{
			if(op--==4){
				resetDrive(drive);
				selectDrive(drive);
				if((RES=seekToCyl(drive,1,0))!=NULL){
					CBMprintStatus(RES,"Seek ");
					retr1--;
					break;
					}
				}

			op=1;
			if((RES=seekToCyl(drive,cyl,0))!=NULL){
				CBMprintStatus(RES,"Seek ");
				retr2--;
				break;
				}

			op=2;
         RES=&RS;
			if(!readSectorID(drive,side,&RS,OP_TIMEOUT)){
				CBMprintStatus(&RS,"Read header ID ");
				retr3--;
				break;
				}
#if (MESSAGES >= 5)
			fprintf(stderr,"Drive %d, Track %2d, Side %2d   --> ReadID: T %2d, H %d, S %2d, BPS %d\n",
					  drive, cyl, side, RS.D.Cylinder, RS.D.Side, RS.D.Sector, RS.D.BytesPS);
#endif
			op=3;
			if(RS.D.Cylinder!=cyl){
				retr4--;
				break;
				}
			*secID=RS.D.Sector;
  	      *sideID=RS.D.Side;
			if(/*RS.D.Side!=side || */RS.D.Sector<1 || RS.D.Sector>FDprm.Sectors || RS.D.BytesPS!=FDprm.BpS){
				printf("Possibly no CBM compatible disk inserted\n");
            return -1;
//				RS.D.Sector=120;
				}
			return 1;
//			return RS.D.Sector;

			}while(0);

     	if(RES!=NULL && RES->D.ST0<0) break; // timeout!!!

		if((RES=recalibrateDrive(drive,0))!=NULL){
			CBMprintStatus(RES,"Recalibrate ");
			op=4;
			retr5--;
			}
		}
	if(RES!=NULL && RES->D.ST0<0) printf("FDC is not responding on ");
	else printf("%d unsuccessful attempts to ",tries);
	switch(op){
		case 1: printf("seek to");					break;
		case 2: printf("read sector ID from");	break;
		case 3: printf("get a valid ID for");       break;
		case 4: printf("recalibrate");
		}
	if(op<4) printf(" side %1d of cylinder %2d of",side,cyl);
	printf(" drive %d\n",drive);
	return 0;
	}

// transfer a track with the multiple sector FDC feature
//
// input:  xfer:		the transfer command (Read, Write, Write&Verify)
//         drive:		the drive, where the disk to is in
//         cyl:		the cylinder number of the track to handle
//         side:		the disk side number of the track to handle
//         buffer:   a pointer to the data transfer buffer
//         ileave:	an interleave factor to be used
//         retries:	the number of retries for this operation
// output: 1 - the operation was successful
//         0 - the operation failed
//        -1 - the operation was aborted by user interaction
//        -2 - write protect error
//
// (Note: CBMxferTrackSS will not be invoked automatically anymore)
//
int CBMxferTrackSS(xferM xfer, uint8 drive, uint8 cyl, uint8 side,
						 uint8 huge *buffer, uint8 ileave, uint retries){
	const char  job[3]={'w','v','r'};
	const char *joberr[3]={"Write ","Verify ","Read "};
	resultString *RES;
	uint8 huge *buf;
	uint retry;
	uint8 v,r,i,t=cyl-1,csec,oldcs,sideID=side;
//	int ret;

	char Done[MaxSectors];		// MaxSectors is defined in CENTRAL.H
   	// markers: 0 		- to write
		//          1 		- to verify
		//          2 		- to read
		//          3 		- read/write/verify error
		//			  -1		- done

   switch(xfer&(WrVrfy|Read)){
      default:
      	return -1;
      case 0:
      	r=0;
      	if(xfer&Format) break;
         return -1;
   	case Read:
      	if(xfer&Format){
				return -1;
            }
      	r=2;
         break;
      case Write:
      case WrVrfy:
      	r=0;
         break;
      case Verify:
      	r=1;
   	}
	for(csec=0;csec<FDprm.Sectors;csec++) Done[csec]=r;

  		// clear done buffer
	ileave%=FDprm.Sectors;

  	if(xfer&Format){
  		if(!CBMseek(drive,cyl)) return 0;
  		csec=0;
      // sideID is preinitialised with side
  		}
  	else{
	   if(CBMPreRWV(drive,t,side,retries,&csec,&sideID)<(-1)) return -1;
      // sideID is preinitialised with side for the case of an abort
  		}

	retry=retries;

   while(retry-->0){
		if(kbhit()){
   		switch(getch()){
      		case 0x1B: return -1;
         	case 0x00: while(kbhit()) getch();	// eat up special combos
	         }
	   	}

		if(xfer&Format){
		   printf("%2d %1d f  \r",cyl,side);
#if (MESSAGES >= 2)
   		fprintf(stderr,"%2d %1d f      \n",cyl,side);
#endif
			RES=formatCBMtrack(drive,t,side);
         if(RES!=NULL){
				CBMprintStatus(RES,"Format ");
	         if(RES->D.ST1&0x02) return -2;
				if(!CBMrecalibrate(drive) || !CBMseek(drive,cyl)) break;
				continue;
   	   	}

         switch(xfer&(WrVrfy|Read)){
   	      case Read:
      	      r=2;
               break;
            case Write:
            case WrVrfy:
      	      r=0;
               break;
            case Verify:
      	      r=1;
   	      }
	      for(csec=0;csec<FDprm.Sectors;csec++) Done[csec]=r;

			if(xfer&WrVrfy) csec=FDprm.Sectors-1;
			else return 1;
         }

      v=(xfer&Verify)?1:0;						// ending with verify?
      r=(xfer&Write)?0:1;						// starting with writing?
      if(xfer&Read) v=r=2;						// or is it reading
      for(;r<=v;r++){							// r=0 - write, r=1 - verify
		   printf("%2d %1d %c  \r",cyl,side,job[r]);

			for(i=0;i<FDprm.Sectors;i++){		// one round verify or write
		   	if(kbhit()){
   		   	switch(getch()){
      		      case 0x1B: return -1;
         			case 0x00: while(kbhit()) getch();	// eat up special combos
	         		}
	   	   	}
            oldcs=csec;	// store the old csec for the case of not having
            				// to do more sectors in the current loop

	         	// calculate the following sector number
            csec+=ileave;
            csec%=FDprm.Sectors;
            	// search the next sector to do (dependend on write or verify)
               // if reading or writing, we search for a todo entry (0)
               // if verifying after writing, we search for a written entry (1)
				// searching the rest of the array, if nothing can be found,
				// the old csec has to be restored!
            for(;csec<FDprm.Sectors;csec++) if(Done[csec]==r) break;
            if(csec>=FDprm.Sectors){				// nothing found, search from the beginning
					// searching from the beginning of the array
					for(csec=0;csec<FDprm.Sectors;csec++) if(Done[csec]==r) break;
	            if(csec>=FDprm.Sectors){
		            csec=oldcs;
						break;	// nothing found, nothing more to do
                  }
            	}
					// next sector to do is: csec
               // calculating buf pointer
				buf=buffer+csec*(128u << FDprm.BpS);
#if (MESSAGES >= 2)
				fprintf(stderr,"%2d %1d %c %3d  \n",cyl,side,job[r],csec+1);
#endif
            switch(r){
            	case 0:	// writing
						RES=writeCBMsector(drive,t,side,csec+1,sideID,buf);
               	break;
            	case 1:	//	verifying
						RES=verifyCBMsector(drive,t,side,csec+1,sideID,buf);
               	break;
            	case 2:	// reading
						RES=readCBMsector(drive,t,side,csec+1,sideID,buf);
               	break;
               default:
               	RES=NULL;
            	}
            if(RES!=NULL){
					CBMprintStatus(RES,joberr[r]);
		         if(RES->D.ST1&0x02) return -2;
					Done[csec]=3;
               }
            else{	// success
            	if(r<v) Done[csec]=1;	// writing done, verify follows
            	else Done[csec]=-1;		// Done
            	}
   	   	}
      	}
   		// set all not done jobs (not -1) in the done buffer to to do (0)
      	// if all sectors are done, we can return

   	// markers: 0      - to do
		//          0x10   - read/write/verify error
		//          WrVrfy - written, if verify is done after
		//          other  - done

      r=1;
     	if(v==1 && (xfer&Write)) v=0;	// writing follows on verify, set to "do write"
      for(i=0;i<FDprm.Sectors;i++) if(Done[i]!=-1){
//      	if(v==1 && (xfer&Write)) v=0;
      	Done[i]=v;	// if there are not done or errornous sectors,
  	      r=0;			// we do a retry
      	}
      if(r) return 1;
   	}
	printf("%d unsuccessful attempts to transfer data at side %1d, cylinder %2d, drive %d\n",retries,side,cyl,drive);
   return 0;
	}

// transfer a track with the multiple sector FDC feature
//
// input:  xfer:		the transfer command (Read, Write, Write&Verify)
//         drive:		the drive, where the disk to is in
//         cyl:		the cylinder number of the track to handle
//         side:		the disk side number of the track to handle
//         buffer:   a pointer to the data transfer buffer
//         ileave:	an interleave factor to be used
//         retries:	the number of retries for this operation
// output: 1 - the operation was successful
//         0 - the operation failed
//        -1 - the operation was aborted by user interaction
//        -2 - write protect error
//
// (Note: CBMxferTrackSS will not be invoked automatically anymore)
//
int CBMxferTrackMS(xferM xfer, uint8 drive, uint8 cyl, uint8 side,
						 uint8 huge *buffer, uint8 ileave, uint retries){
	resultString *RES;
   const char *joberr[4]={"Read ","Write ","Verify ","Format "};

	uint retry;
	uint8 t=cyl-1,csec,sideID=side;
	int job;

  	if(xfer&0x01 && xfer&0x0E){
	   printf("Womo isn idiot, please report this!!!\n");
		return -1;
      }

//	if(xfer&Format){
		ileave+=FDprm.Sectors-1;	// interleave is decremented by 1 for multiple
		ileave%=FDprm.Sectors;		// sector operations (default 0)
   	  									// This _can_ increase combined fromatting speed
	if(xfer&Format){
	  	if(!CBMseek(drive,cyl)) return 0;
//		csec=1;
      }
   else{
	   if(CBMPreRWV(drive,t,side,retries,&csec,&sideID)< -1) return -1;
	      // sideID is preinitialised with side for the case of an abort
//		ileave%=FDprm.Sectors;
		csec+=ileave;
      csec++;
//		if(csec>FDprm.Sectors) csec-=FDprm.Sectors;
		while(csec>FDprm.Sectors) csec-=FDprm.Sectors;
   	}

	retry=retries;

   while(retry-->0){
	   if(kbhit()){
   	   switch(getch()){
      	   case 0x1B: return -1;
         	case 0x00: while(kbhit()) getch();	// eat up special combos
	         }
   	   }
	   printf("%2d %1d ",cyl,side);
		if(xfer&Read){
		   printf("r  \r");
#if (MESSAGES >= 2)
	   	fprintf(stderr,"%2d %1d r %3d  \n",cyl,side,csec);
#endif
			job=0;
      	RES=readCBMtrack(drive,t,side,csec,sideID,buffer);
      	}
      else{
         RES=NULL;
			if(xfer&Format){
			   printf("f  \r");
#if (MESSAGES >= 2)
	   		fprintf(stderr,"%2d %1d f      \n",cyl,side);
#endif
				job=4;
				RES=formatCBMtrack(drive,t,side);
				if(xfer&WrVrfy){
               csec=1+ileave;
				   printf("%2d %1d ",cyl,side);
            	}
				}
			if(RES==NULL && xfer&Write){
			   printf("w  \r");
#if (MESSAGES >= 2)
	   		fprintf(stderr,"%2d %1d w %3d  \n",cyl,side,csec);
#endif
				job=1;
				RES=writeCBMtrack(drive,t,side,csec,sideID,buffer);
            }
			if(RES==NULL && xfer&Verify){
				if(xfer&Write){
					csec+=ileave;
					if(csec>FDprm.Sectors) csec-=FDprm.Sectors;
				   printf("%2d %1d ",cyl,side);
            	}
			   printf("v  \r");
#if (MESSAGES >= 2)
	   		fprintf(stderr,"%2d %1d v %3d  \n",cyl,side,csec);
#endif
				job=2;
				RES=verifyCBMtrack(drive,t,side,csec,sideID,buffer);
         	}
      	}
		if(RES==NULL) return 1;
      else{
			CBMprintStatus(RES,joberr[job]);
         if(RES->D.ST1&0x02) return -2;
         if(job==4 && (!CBMrecalibrate(drive) || !CBMseek(drive,cyl))) break;
         }

		csec+=ileave;
		if(csec>FDprm.Sectors) csec-=FDprm.Sectors;
   	}
	printf("%d unsuccessful attempts to transfer data at side %1d, cylinder %2d, drive %d\n",retries,side,cyl,drive);
//	CBMprintStatus(RES);
	return 0;
	}

#ifdef FDCintel82078
// format'n'write a track via intel's extended command
//
// input:  xfer:		the transfer command (Read, Write, Write&Verify)
//         drive:		the drive, where the disk to is in
//         cyl:		the cylinder number of the track to handle
//         side:		the disk side number of the track to handle
//         buffer:   a pointer to the data transfer buffer
//         retries:	the number of retries for this operation
// output: 1 - the operation was successful
//         0 - the operation failed
//        -1 - the operation was aborted by user interaction
//        -2 - write protect error
//
int CBMfwvTrack(xferM xfer, uint8 drive, uint8 cyl, uint8 side,
						 uint8 huge *buffer, uint retries){
	resultString *RES;
   const char *joberr[3]={"Format'n'write ","Verify "};

	uint retry;
	uint8 t=cyl-1;
	int job;

   	// operations allowed:
      // format'n'write
      // format'n'write with verify
  	if(xfer&~Verify) return -1;

	retry=retries;

   while(retry-->0){
	   if(kbhit()){
   	   switch(getch()){
      	   case 0x1B: return -1;
         	case 0x00: while(kbhit()) getch();	// eat up special combos
	         }
   	   }
	   printf("%2d %1d fw \r",cyl,side);
      RES=NULL;
#if (MESSAGES >= 2)
	   fprintf(stderr,"%2d %1d fw     \n",cyl,side);
#endif
		job=0;
		RES=fnwriteCBMtrack(drive,t,side,buffer);

		if(RES==NULL && xfer&Verify){
			printf("%2d %1d v  \r",cyl,side);
#if (MESSAGES >= 2)
	   	fprintf(stderr,"%2d %1d v      \n",cyl,side);
#endif
			job=1;
			RES=verifyCBMtrack(drive,t,side,1,buffer);
      	}
		if(RES==NULL) return 1;
      else{
			CBMprintStatus(RES,joberr[job]);
         if(RES->D.ST1&0x02) return -2;
         }
   	}
	printf("%d unsuccessful attempts to format'n'write at side %1d, cylinder %2d, drive %d\n",retries,side,cyl,drive);
	return 0;
	}
#endif

// generate a new indexed filename from the previous one
//
// input:  inStr:		the old indexed or unindexed file name
// output: not NULL: the new indexed file name
//         NULL:		a new indexed file name couldn't be generated
//
char *buildInxName(char *inStr){
	const int maxLength=8;

	register int i;
	int length,delimiter,indexStr;
	unsigned long int inx;
	char outStr[13], inxStr[11], *ptr, *op, *rs;

	if(inStr==NULL) return NULL;

   rs=FilenameRetStr;
   // separate filename from path and copy path into retString
   	// search end of string and copy path
	for(ptr=inStr,op=FilenameRetStr;*ptr!='\0';ptr++,op++) *op=*ptr;
   	// search last ':', '\' or '/'
   while(ptr>=inStr && *ptr!=':' && *ptr!='\\' && *ptr!='/') ptr--;
   if(ptr>=inStr){	// there has been a path delimiter found
   	ptr++;			// the following char is the first of the filename

			// subraction of two pointers results in an integer
         // the problem is, that it is a long int, but the first
         // pointer "retStr" is no far pointer, so that the cast
			// is needed (perhaps this causes a bug?)
      rs=FilenameRetStr+(int)(ptr-inStr);
      inStr=ptr;
   	}

	delimiter=-1;
	for(i=0,ptr=inStr;*ptr!='\0';i++,ptr++) if(*ptr=='.') delimiter=i;
	length=i;
	if(delimiter==-1) delimiter=length;

#if (MESSAGES >= 3)
		fprintf(stderr,"Delimiter: \"%s\"\n",inStr+delimiter);
#endif

		// get index string part (up to a maximum of 8 chars)
	for(i=delimiter,ptr=inStr+delimiter-1;ptr>=inStr;i--,ptr--)
		if(*ptr<'0' || *ptr>'9' || delimiter-i>=maxLength)
			break;
	indexStr=i;

#if (MESSAGES >= 3)
	fprintf(stderr,"Index: \"%s\"\n",inStr+indexStr);
#endif

		// build new index number string
	if(delimiter==indexStr){
			// new index number is 1
		inxStr[0]='1';
		inxStr[1]='\0';
#if (MESSAGES >= 3)
		fprintf(stderr,"no old index, generating \"1\"\n");
#endif
		}
	else{
   	inx=0L;
   	for(ptr=inStr+indexStr,i=indexStr;i<delimiter;ptr++,i++){
      	inx*=10L;
         inx+=(*ptr-'0');
      	}
//		inx=strtoul(inStr+indexStr,NULL,10);
#if (MESSAGES >= 3)
		fprintf(stderr,"old index number: \"%lu\"\n",inx);
#endif
		inx++;												// build next index number
		sprintf(inxStr, "%lu", inx);	// write index number to string
#if (MESSAGES >= 3)
		fprintf(stderr,"new index number: \"%lu\"\n",inx);
#endif
		}

		// determine the length of the new index string
	for(i=0,ptr=inxStr;*ptr!='\0';i++,ptr++);
	length=i;
#if (MESSAGES >= 3)
	fprintf(stderr,"new index length: \"%d\"\n",i);
#endif
	if(length>maxLength) return NULL;		// new filename will be invalid

		// copying base name to outStr
	for(i=0,ptr=inStr,op=outStr;i<delimiter && i<maxLength;i++,ptr++,op++) *op=*ptr;

		// new combined string will be to long
	i-=length;
	if(i<indexStr) i=indexStr;			// don't eat up file name chars, if not needed
	if(i+length>=maxLength) i=maxLength-length;
	op=outStr+i;

		// copy index number string onto the base name
	for(ptr=inxStr;i<maxLength && *ptr!='\0';i++,ptr++,op++) *op=*ptr;

		// copy extension onto file name
   for(ptr=inStr+delimiter;*ptr!='\0';ptr++,op++) *op=*ptr;

		// setting string end delimiter
	*op='\0';

   	// copy outString into static return String, so the string can
      // againg be the input of this function without conflicts
	for(ptr=outStr,op=rs,i=0;i<maxLength+5;ptr++,op++,i++) *op=*ptr;

	return FilenameRetStr;
	}

// generate a filename with the forced extension
//
// input:  inStr:		the input filename
//         force:    if true, extension changing is forced
// output: not NULL: the new generated file name
//         NULL:		a new file name couldn't be generated
//
char *changeExtension(char *inStr, int force, FDprmSet Set){
	const char *Extensions[2]={"D81","D2M"};
	char *ptr, *op;

	if(inStr==NULL) return NULL;

   if(Set<0 || Set>1){
#if (MESSAGES >= 3)
		fprintf(stderr,"Set invalid, extension not changed, it's \"%s\"\n",inStr);
#endif
		return inStr;	// let it be unchanged
      }

	   // separate filename from path and copy path into retString
   	// search end of string and copy path
	for(ptr=inStr,op=FilenameRetStr;*ptr!='\0';ptr++,op++) *op=*ptr;
   *op='\0';

		   	// search last ':', '\' or '/' in the output string
	while(op>=FilenameRetStr && *op!=':' && *op!='\\' && *op!='/') op--;
   op++;		// select the following char of the delimiter

   ptr=NULL;// search the last delimiting dot '.', ptr marks the dot
	while(*op!='\0'){
   	if(*op=='.') ptr=op;
      op++;
   	}
//   if(ptr!=NULL){		// if a dot was found don't change anything
   if(ptr!=NULL){		// if a dot was found don't change anything
      if(!force){
#if (MESSAGES >= 3)
			fprintf(stderr,"Dot found, extension not changed, it's \"%s\"\n",inStr);
#endif
			return inStr;	// let it be unchanged
         }
		op=ptr;
      }
   *op++='.';
   ptr=(char *)Extensions[Set];
#if (MESSAGES >= 3)
	fprintf(stderr,"Changing extension to \"%s\"\n",ptr);
#endif

   while(*ptr!='\0') *op++=*ptr++;

#if (MESSAGES >= 3)
	fprintf(stderr,"Filename changed to \"%s\"\n",FilenameRetStr);
#endif
	return FilenameRetStr;
	}
