#type vertex
#version 330 core
layout (location = 0) in vec3 aPos;
layout (location = 1) in vec3 aNormal;
layout (location = 2) in vec2 aTexCoords;

// Variables sent to de Fragment Shader
out vec3 oPos;
out vec3 oNormal;
out vec2 oTexCoords;
out vec3 oLightIntensity;

// Light info (to be sent by us in the script)
uniform vec3 L_Position; // Light position in eye coords.
uniform vec3 La; // Ambient light intensity
uniform vec3 Ld; // Diffuse light intensity
uniform vec3 Ls; // Specular light intensity

// Camera position (to be sent by us in the script)
uniform vec3 CameraPosition;

// Material Data (sent by the engine automatically)
uniform vec3 Mat_Ka;			// Ambient reflectivity
uniform vec3 Mat_Kd;			// Diffuse reflectivity
uniform vec3 Mat_Ks;			// Specular reflectivity
uniform float Mat_KsStrenght;	// Specular shininess factor

// Matrix (sent by the engine automatically)
uniform mat4 model;
uniform mat4 view;
uniform mat4 projection;


void main() {
mat4 ModelViewMatrix = view*model;
mat3 NormalMatrix = mat3(transpose(inverse(ModelViewMatrix)));	// Get the normal Matrix (3x3)
mat4 MVP = projection * view * model;

vec3 tnorm = normalize( NormalMatrix * aNormal);
vec4 eyeCoords = ModelViewMatrix * vec4(aPos,1.0);
vec3 s = normalize(vec3(L_Position - eyeCoords.xyz));
vec3 v = normalize(-eyeCoords.xyz);

vec3 r = reflect( -s, tnorm );
vec3 ambient = La * Mat_Ka;
float sDotN = max( dot(s,tnorm), 0.0 );
vec3 diffuse = Ld * Mat_Kd * sDotN;
vec3 spec = vec3(0.0);
if( sDotN > 0.0 )
spec = Ls * Mat_Ks * pow( max( dot(r,v), 0.0 ), Mat_KsStrenght );

// Output values
oLightIntensity = ambient + diffuse + spec;
oTexCoords = aTexCoords;

// compute normal in world space – no scales allowed
oNormal = normalize(vec3(model * vec4(aNormal, 1)));
// compute position in world space
vec3 pos = vec3(model * vec4(aPos,1));
// compute look direction in world space
oPos = pos - CameraPosition;

gl_Position = MVP * vec4(aPos,1.0);
}

#type fragment
#version 330 core
out vec4 FragColor;

in vec3 oPos;
in vec3 oNormal;
in vec2 oTexCoords;
in vec3 oLightIntensity;

uniform float reflectFactor;
uniform sampler2D texture_diffuse1;
uniform samplerCube skybox;
uniform vec3 CameraPosition; // Camera position

//	Simplex 3D Noise 
//	by Ian McEwan, Ashima Arts
//
vec4 permute(vec4 x){return mod(((x*34.0)+1.0)*x, 289.0);}
vec4 taylorInvSqrt(vec4 r){return 1.79284291400159 - 0.85373472095314 * r;}

float snoise(vec3 v){ 
  const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;
  const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);

// First corner
  vec3 i  = floor(v + dot(v, C.yyy) );
  vec3 x0 =   v - i + dot(i, C.xxx) ;

// Other corners
  vec3 g = step(x0.yzx, x0.xyz);
  vec3 l = 1.0 - g;
  vec3 i1 = min( g.xyz, l.zxy );
  vec3 i2 = max( g.xyz, l.zxy );

  //  x0 = x0 - 0. + 0.0 * C 
  vec3 x1 = x0 - i1 + 1.0 * C.xxx;
  vec3 x2 = x0 - i2 + 2.0 * C.xxx;
  vec3 x3 = x0 - 1. + 3.0 * C.xxx;

// Permutations
  i = mod(i, 289.0 ); 
  vec4 p = permute( permute( permute( 
             i.z + vec4(0.0, i1.z, i2.z, 1.0 ))
           + i.y + vec4(0.0, i1.y, i2.y, 1.0 )) 
           + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));

// Gradients
// ( N*N points uniformly over a square, mapped onto an octahedron.)
  float n_ = 1.0/7.0; // N=7
  vec3  ns = n_ * D.wyz - D.xzx;

  vec4 j = p - 49.0 * floor(p * ns.z *ns.z);  //  mod(p,N*N)

  vec4 x_ = floor(j * ns.z);
  vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)

  vec4 x = x_ *ns.x + ns.yyyy;
  vec4 y = y_ *ns.x + ns.yyyy;
  vec4 h = 1.0 - abs(x) - abs(y);

  vec4 b0 = vec4( x.xy, y.xy );
  vec4 b1 = vec4( x.zw, y.zw );

  vec4 s0 = floor(b0)*2.0 + 1.0;
  vec4 s1 = floor(b1)*2.0 + 1.0;
  vec4 sh = -step(h, vec4(0.0));

  vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;
  vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;

  vec3 p0 = vec3(a0.xy,h.x);
  vec3 p1 = vec3(a0.zw,h.y);
  vec3 p2 = vec3(a1.xy,h.z);
  vec3 p3 = vec3(a1.zw,h.w);

//Normalise gradients
  vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));
  p0 *= norm.x;
  p1 *= norm.y;
  p2 *= norm.z;
  p3 *= norm.w;

// Mix final noise value
  vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);
  m = m * m;
  return 42.0 * dot( m*m, vec4( dot(p0,x0), dot(p1,x1), 
                                dot(p2,x2), dot(p3,x3) ) );
}

float rand(vec2 co){
    return fract(sin(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);
}

void main()
{
  // normalize vectors
  vec3 n = normalize(oNormal);
  vec3 e = normalize(oPos);
  // compute reflection vector
  vec3 t = reflect(e, n);
  // access cube map texture
  vec3 cubeColor = texture(skybox, t).rgb;

  //vec4 diffuseColor = texture(texture_diffuse1, oTexCoords);
  // Como el objeto no tiene textura, le he clavado el color q tenias tu puesto
  vec4 diffuseColor = vec4(0.5, 0.2, 0.8, 1);
  vec4 ilumColor = vec4(oLightIntensity, 1.0);

  FragColor = mix (ilumColor * diffuseColor, vec4(cubeColor, 1.0), reflectFactor);

  // Normal render
  //FragColor = vec4(oLightIntensity, 1.0) * diffuseColor;
}