#include "fbo.h"
#include "SDL2/SDL_log.h"
#include "config.h"
#include "gl.h"
#include <stdlib.h>

GLuint create_texture(GLsizei width, GLsizei height, GLint internal_format,
                      GLenum format, GLenum type, GLint min_filter,
                      GLint mag_filter, GLint wrap) {
    GLuint texture;
    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexImage2D(GL_TEXTURE_2D, 0, internal_format, width, height, 0, format,
                 type, NULL);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, min_filter);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, mag_filter);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrap);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrap);
    return texture;
}

GLuint create_depth(GLsizei width, GLsizei height) {
    return create_texture(width, height, GL_DEPTH_COMPONENT32F,
                          GL_DEPTH_COMPONENT, GL_FLOAT, GL_NEAREST, GL_NEAREST,
                          GL_CLAMP_TO_EDGE);
}

fbo_t *fbo_init_with_texture(GLuint texture, GLint level, GLuint depth) {
    fbo_t *fbo = calloc(1, sizeof(fbo_t));
    fbo->texture = texture;
    glGenFramebuffers(1, &fbo->framebuffer);
    glBindFramebuffer(GL_FRAMEBUFFER, fbo->framebuffer);
    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D,
                           fbo->texture, level);

    if (depth != 0) {
        fbo->depth = depth;
        glFramebufferTexture2D(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                               GL_TEXTURE_2D, fbo->depth, 0);
    }

    if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE) {
        SDL_Log("FBO not complete\n");
        return NULL;
    }

    glGetTexLevelParameteriv(GL_TEXTURE_2D, level, GL_TEXTURE_WIDTH,
                             &fbo->width);
    glGetTexLevelParameteriv(GL_TEXTURE_2D, level, GL_TEXTURE_HEIGHT,
                             &fbo->height);

    return fbo;
}

fbo_t *fbo_init(GLsizei width, GLsizei height, GLint filter, GLuint depth) {
    GLuint texture = create_texture(width, height, HDR_INTERNALFORMAT, GL_RGB,
                                    GL_FLOAT, filter, filter, GL_CLAMP_TO_EDGE);
    fbo_t *fbo = fbo_init_with_texture(texture, 0, depth);
    if (fbo == NULL) {
        return NULL;
    }
    fbo->owns_texture = 1;
    return fbo;
}

void fbo_deinit(fbo_t *fbo) {
    if (fbo) {
        if (fbo->framebuffer) {
            glDeleteFramebuffers(1, &fbo->framebuffer);
        }
        if (fbo->texture && fbo->owns_texture) {
            glDeleteTextures(1, &fbo->texture);
        }
        free(fbo);
    }
}
