#include "demo.h"
#include "bloom.h"
#include "cglm/affine.h"
#include "cglm/cam.h"
#include "cglm/vec3.h"
#include "config.h"
#include "fbo.h"
#include "filesystem.h"
#include "gl.h"
#include "mesh.h"
#include "pass.h"
#include "rand.h"
#include "shader.h"
#include "sync.h"
#include "text.h"
#include <GLES3/gl31.h>
#include <SDL2/SDL.h>
#include <stdlib.h>
#include <string.h>

#define PI 3.1415926535897932384626433832795

#define STR_IMPL_(x) #x     // stringify argument
#define STR(x) STR_IMPL_(x) // indirection to expand argument macros

// Allocate this many FBO:s to run render passes.
#define FBS 2

typedef struct {
    int blit_output;
    int width;
    int height;
    output_coord_t coord;
} output_parameters_t;

// This messy struct is the backbone of our renderer.
struct demo_t_ {
    // This holds final output window scaling information
    output_parameters_t output_parameters;
    // This holds GL state for rendering 2D "screen" shader effects
    pass_renderer_t *pass_renderer;
    // Shader programs for render passes. The stars of this show.
    program_t effect_program;
    program_t post_program;
    program_t raster_program;
    // If integer value is 0, there is a problem with the shaders
    int programs_ok;
    // A RGBA noise texture is used in rendering
    GLuint noise_texture;
    // Our FBOs used for rendering every frame
    fbo_t *fbs[FBS];
    GLuint post_fb_tex;
    fbo_t *post_fb;
    // Post processing bloom
    bloom_t *bloom;
    // This integer is the index ([] number) of the FB which holds current
    // frame's "main" target FBO. 0 or 1. This FB gets the base rendered image
    // before any post processing etc, and the other (0 or 1) holds the previous
    // frame's first pass result for feedback effects.
    size_t firstpass_fb_idx;

    scene_t *scene;
    text_t *text;
};

// This function replaces *old with new, but only if new has a non-zero
// handle (meaning, it compiled and linked successfully).
// Return value is 1 if new program is fine to use, 0 otherwise.
static int replace_program(program_t *old, program_t new) {
    if (!new.handle) {
#ifndef DEBUG
        abort();
#endif
        return 0;
    }
    if (old->handle) {
        program_deinit(old);
    }
    *old = new;

    return 1;
}

// This function drives all shader loading. Gets called on initialization,
// and also from event handler (main.c) if R is pressed.
void demo_reload(demo_t *demo) {
    // Load test scene
    scene_deinit(demo->scene);
    demo->scene = scene_init_gltf("data/mehu.glb");
    demo->programs_ok = demo->scene != NULL;

    char *msg = NULL;
    size_t msg_len = read_file("data/scroll.txt", &msg);
    // TODO this is leaking memory on each reload
    text_init(demo->text, msg, msg_len, "data/aaaiight.ttf", 16, 0.1);
    // TODO handle errors
    free(msg);

    GLuint fragment_shader =
        compile_shader_file("shaders/shader.frag", NULL, 0);

    // If replace_program returns 0, programs_ok get set to 0 regardless of
    // it's current value.
    demo->programs_ok &= replace_program(
        &demo->effect_program,
        link_program(
            (GLuint[]){demo->pass_renderer->vertex_shader, fragment_shader},
            2));

    GLuint post_shader = compile_shader_file(
        "shaders/post.frag",
        (shader_define_t[]){(shader_define_t){.name = "BLOOM_LEVELS",
                                              .value = STR(BLOOM_LEVELS)}},
        1);

    // If replace_program returns 0, programs_ok get set to 0 regardless of
    // it's current value.
    demo->programs_ok &= replace_program(
        &demo->post_program, link_program(
                                 (GLuint[]){
                                     demo->pass_renderer->vertex_shader,
                                     post_shader,
                                 },
                                 2));

    // Cleanup shader objects because they have already been linked to
    // programs
    shader_deinit(fragment_shader);
    shader_deinit(post_shader);

    // Setup rasterization program
    GLuint vertex_shader = compile_shader_file("shaders/rast.vert", NULL, 0);
    fragment_shader = compile_shader_file("shaders/rast.frag", NULL, 0);
    // If replace_program returns 0, programs_ok get set to 0 regardless of
    // it's current value.
    demo->programs_ok &= replace_program(
        &demo->raster_program,
        link_program((GLuint[]){vertex_shader, fragment_shader}, 2));

    // Cleanup shader objects because they have already been linked to
    // programs
    shader_deinit(vertex_shader);
    shader_deinit(fragment_shader);
}

static void letterbox_gl_blit_framebuffer(output_parameters_t *param,
                                          int window_width, int window_height) {
    param->coord = (output_coord_t){.x1 = window_width, .y1 = window_height};

    double demo_aspect_ratio = (double)param->width / (double)param->height;
    double window_aspect_ratio = (double)window_width / (double)window_height;

    if (window_aspect_ratio > demo_aspect_ratio) {
        double adjusted = window_height * demo_aspect_ratio;
        int remainder = (window_width - adjusted) / 2;
        param->coord.x0 = remainder;
        param->coord.x1 = remainder + adjusted;
    } else {
        double adjusted = window_width / demo_aspect_ratio;
        int remainder = (window_height - adjusted) / 2;
        param->coord.y0 = remainder;
        param->coord.y1 = remainder + adjusted;
    }
}

static void letterbox_gl_viewport(output_parameters_t *param, int window_width,
                                  int window_height) {
    param->coord = (output_coord_t){.x1 = param->width, .y1 = param->height};

    double demo_aspect_ratio = (double)param->width / (double)param->height;
    double window_aspect_ratio = (double)window_width / (double)window_height;

    if (window_aspect_ratio > demo_aspect_ratio) {
        param->coord.x0 = (window_width - param->width) / 2;
    } else {
        param->coord.y0 = (window_height - param->height) / 2;
    }
}

// This ugly function computes rectangle coordinates for
// scaling/letterboxing output from internal aspect ratio to actual window
// size.
void demo_resize(demo_t *demo, int width, int height) {
    // Enable direct output when possible
    demo->output_parameters.blit_output =
        !((width == demo->output_parameters.width &&
           height >= demo->output_parameters.height) ||
          (height == demo->output_parameters.height &&
           width >= demo->output_parameters.width));

    if (demo->output_parameters.blit_output) {
        letterbox_gl_blit_framebuffer(&demo->output_parameters, width, height);
        SDL_Log("Using framebuffer blit output\n");
    } else {
        letterbox_gl_viewport(&demo->output_parameters, width, height);
        // Delete post processing framebuffer when not needed
        if (demo->post_fb) {
            fbo_deinit(demo->post_fb);
            demo->post_fb = NULL;
            glDeleteTextures(1, &demo->post_fb_tex);
            demo->post_fb_tex = 0;
        }
        SDL_Log("Using direct output\n");
    }
}

// "demo_t's constructor" (if this were C++...)
demo_t *demo_init(int width, int height) {
    demo_t *demo = calloc(1, sizeof(demo_t));
    if (!demo) {
        return NULL;
    }

    demo->output_parameters.width = width;
    demo->output_parameters.height = height;
    demo_resize(demo, width, height);

    // Initialize pass renderer
    demo->pass_renderer = pass_renderer_init();

    // Initialize scroller
    demo->text = malloc(sizeof(text_t));

    // Load shaders and other assets
    demo_reload(demo);

    // Create FBs
    GLuint depth = create_depth(width, height);
    for (size_t i = 0; i < FBS; i++) {
        demo->fbs[i] = fbo_init(width, height, GL_LINEAR, depth);
        if (demo->fbs[i] == NULL) {
            return NULL;
        }
    }

    // Initialize bloom effect
    demo->bloom = bloom_init(demo->pass_renderer, width, height);

    // Allocate noise texture
    demo->noise_texture =
        create_texture(NOISE_SIZE, NOISE_SIZE, GL_RGBA8, GL_RGBA,
                       GL_UNSIGNED_BYTE, GL_NEAREST, GL_NEAREST, GL_REPEAT);

    return demo;
}

// This gets called once per frame from main loop (main.c)
void demo_render(demo_t *demo, struct sync_device *rocket, double rocket_row) {
    static unsigned char noise[NOISE_SIZE * NOISE_SIZE * 4];
    const size_t cur_fb_idx = demo->firstpass_fb_idx;
    const size_t alt_fb_idx = cur_fb_idx ? 0 : 1;

#ifdef DEBUG
    // Early return if shaders are currently unusable
    if (!demo->programs_ok) {
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
        glClearColor(0.3, 0., 0., 1.);
        glClear(GL_COLOR_BUFFER_BIT);
        return;
    }
#endif

    glClearColor(0., 0., 0., 1.);

    // MAKE SOME NOISE !!!! WOOO
    // ------------------------------------------------------------------------

    glActiveTexture(GL_TEXTURE0);
    for (GLsizei i = 0; i < NOISE_SIZE * NOISE_SIZE * 4; i++) {
        noise[i] = rand_xoshiro();
    }
    glBindTexture(GL_TEXTURE_2D, demo->noise_texture);
    glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, NOISE_SIZE, NOISE_SIZE, GL_RGBA,
                    GL_UNSIGNED_BYTE, noise);

    // Initialize pass render parameters
    pass_render_parameters_t parameters = {
        .renderer = demo->pass_renderer,
        .draw_fb = demo->fbs[cur_fb_idx],
        .viewport = NULL,
        .program = &demo->effect_program,
        .rocket = rocket,
        .rocket_row = rocket_row,
        .textures =
            (GLuint[]){demo->fbs[alt_fb_idx]->texture, demo->noise_texture},
        .sampler_ufm_names =
            (const char *[]){"u_FeedbackSampler", "u_NoiseSampler"},
        .n_textures = 2,
    };

    // Effect shader
    // ------------------------------------------------------------------------

    pass_render(&parameters);

    // Rasterized geometry
    // ------------------------------------------------------------------------

    glEnable(GL_DEPTH_TEST);

    glUseProgram(demo->raster_program.handle);
    set_rocket_uniforms(&demo->raster_program, rocket, rocket_row);

    mat4 projection;
    const double aspect_ratio = (double)demo->output_parameters.width /
                                (double)demo->output_parameters.height;
    glm_perspective(GET_VALUE("Cam:fov") * PI / 180., aspect_ratio, 0.1, 1000.,
                    projection);
    mat4 view;
    glm_lookat((vec3){GET_VALUE("Cam:pos.x") + sin(rocket_row * 0.04) * 0.1,
                      GET_VALUE("Cam:pos.y") + sin(rocket_row * 0.07) * 0.1,
                      GET_VALUE("Cam:pos.z") + sin(rocket_row * 0.10) * 0.1},
               (vec3){
                   GET_VALUE("Cam:target.x"),
                   GET_VALUE("Cam:target.y"),
                   GET_VALUE("Cam:target.z"),
               },
               GLM_YUP, view);

    glUniformMatrix4fv(
        glGetUniformLocation(demo->raster_program.handle, "u_Projection"), 1,
        GL_FALSE, (float *)projection);
    glUniformMatrix4fv(
        glGetUniformLocation(demo->raster_program.handle, "u_View"), 1,
        GL_FALSE, (float *)view);

    scene_draw(demo->scene, &demo->raster_program);

    // Scroller
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
    // glBlendFunc(GL_ONE, GL_ONE);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glBlendEquation(GL_FUNC_ADD);
    glm_rotate_make(view, sin(rocket_row * 0.04) * 0.2 - 0.2, GLM_YUP);
    glm_translate(view, (vec3){6. - rocket_row * 0.2, -2., -4.});
    text_draw(demo->text, view, projection, GET_VALUE("Scroller:depth"));
    glDisable(GL_BLEND);

    // Post effects
    // ------------------------------------------------------------------------

    parameters.textures = (GLuint[]){demo->fbs[cur_fb_idx]->texture};
    parameters.n_textures = 1;
    bloom_render(demo->bloom, &parameters);

    // Create framebuffer for post processing output lazily when needed
    if (demo->output_parameters.blit_output && !demo->post_fb) {
        demo->post_fb_tex = create_texture(
            demo->output_parameters.width, demo->output_parameters.height,
            GL_RGB8, GL_RGB, GL_UNSIGNED_BYTE, GL_LINEAR, GL_LINEAR,
            GL_CLAMP_TO_EDGE);
        demo->post_fb = fbo_init_with_texture(demo->post_fb_tex, 0, 0);
    }

    parameters.draw_fb = demo->post_fb;
    if (demo->output_parameters.blit_output == 0) {
        parameters.viewport = &demo->output_parameters.coord;
    }
    parameters.program = &demo->post_program;
    parameters.textures =
        (GLuint[]){demo->fbs[cur_fb_idx]->texture,
                   bloom_get_texture(demo->bloom), demo->noise_texture};
    parameters.sampler_ufm_names =
        (const char *[]){"u_InputSampler", "u_BloomSampler", "u_NoiseSampler"};
    parameters.n_textures = 3;

    pass_render(&parameters);

    if (demo->output_parameters.blit_output) {
        // Output blit
        // --------------------------------------------------------------------
        // This stretches or squashes the post-processed image to the window
        // in correct aspect ratio (framebuffer 0).
        const output_coord_t *coord = &demo->output_parameters.coord;

        glBindFramebuffer(GL_READ_FRAMEBUFFER, demo->post_fb->framebuffer);
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
        glClear(GL_COLOR_BUFFER_BIT);
        glBlitFramebuffer(0, 0, demo->post_fb->width, demo->post_fb->height,
                          coord->x0, coord->y0, coord->x1, coord->y1,
                          GL_COLOR_BUFFER_BIT, GL_LINEAR);
    }

    // Switch fb to keep render results in memory for feedback effects
    demo->firstpass_fb_idx = alt_fb_idx;
}

void demo_deinit(demo_t *demo) {
    if (demo) {
        if (demo->text) {
            free(demo->text);
        }
        free(demo);
    }
}
