#include "bloom.h"
#include "config.h"
#include "fbo.h"
#include "pass.h"
#include "shader.h"
#include <stdio.h>
#include <string.h>

bloom_t *bloom_init(pass_renderer_t *pass_renderer, GLsizei width,
                    GLsizei height) {
    bloom_t *bloom = calloc(1, sizeof(bloom_t));

    const GLuint vertex_shader = pass_renderer->vertex_shader;
    GLuint pre_shader = compile_shader_file("shaders/bloom_pre.frag", NULL, 0);
    bloom->pre_program = link_program(
        (GLuint[]){
            vertex_shader,
            pre_shader,
        },
        2);
    shader_deinit(pre_shader);
    if (bloom->pre_program.handle == 0) {
        return NULL;
    }

    for (size_t i = 0; i < BLOOM_LEVELS; i++) {
        char buf[32];
        memset(buf, 0, 32);
        snprintf(buf, 32, "%zu", i);

        GLuint blur_x_shader = compile_shader_file(
            "shaders/blur.frag",
            (shader_define_t[]){
                (shader_define_t){.name = "HORIZONTAL", .value = "1"},
                (shader_define_t){.name = "MIP_LEVEL", .value = buf}},
            2);
        bloom->blur_x_programs[i] = link_program(
            (GLuint[]){
                vertex_shader,
                blur_x_shader,
            },
            2);
        shader_deinit(blur_x_shader);
        if (bloom->blur_x_programs[i].handle == 0) {
            return NULL;
        }

        GLuint blur_y_shader =
            compile_shader_file("shaders/blur.frag",
                                (shader_define_t[]){(shader_define_t){
                                    .name = "MIP_LEVEL", .value = buf}},
                                1);
        bloom->blur_y_programs[i] = link_program(
            (GLuint[]){
                vertex_shader,
                blur_y_shader,
            },
            2);
        shader_deinit(blur_y_shader);
        if (bloom->blur_y_programs[i].handle == 0) {
            return NULL;
        }
    }

    for (size_t i = 0; i < BLOOM_FBS; i++) {
        bloom->textures[i] = create_texture(
            width / BLOOM_RESOLUTION_SCALE, height / BLOOM_RESOLUTION_SCALE,
            HDR_INTERNALFORMAT, GL_RGB, GL_FLOAT, GL_LINEAR_MIPMAP_NEAREST,
            GL_LINEAR, GL_CLAMP_TO_EDGE);
        glBindTexture(GL_TEXTURE_2D, bloom->textures[i]);
        glGenerateMipmap(GL_TEXTURE_2D);
    }

    for (size_t i = 0; i < BLOOM_LEVELS; i++) {
        for (size_t j = 0; j < BLOOM_FBS; j++) {
            bloom->fbs[i * BLOOM_FBS + j] =
                fbo_init_with_texture(bloom->textures[j], i, 0);
            if (bloom->fbs[i] == NULL) {
                return NULL;
            }
        }
    }

    return bloom;
}

void bloom_deinit(bloom_t *bloom) {
    if (bloom) {
        for (size_t i = 0; i < BLOOM_FBS * BLOOM_LEVELS; i++) {
            fbo_deinit(bloom->fbs[i]);
        }
        glDeleteTextures(BLOOM_FBS, bloom->textures);

        for (size_t i = 0; i < BLOOM_LEVELS; i++) {
            program_deinit(&bloom->blur_y_programs[i]);
            program_deinit(&bloom->blur_x_programs[i]);
        }
        program_deinit(&bloom->pre_program);

        free(bloom);
    }
}

void bloom_render(bloom_t *bloom, pass_render_parameters_t *parameters) {
    parameters->draw_fb = bloom->fbs[0];
    parameters->program = &bloom->pre_program;
    parameters->sampler_ufm_names = (const char *[]){"u_InputSampler"};
    pass_render(parameters);

    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, bloom->fbs[0]->texture);
    glGenerateMipmap(GL_TEXTURE_2D);

    for (size_t i = 0; i < BLOOM_LEVELS; i++) {
        parameters->draw_fb = bloom->fbs[i * BLOOM_FBS + 1];
        parameters->program = &bloom->blur_x_programs[i];
        parameters->textures = (GLuint[]){bloom->fbs[i * BLOOM_FBS]->texture};
        pass_render(parameters);

        parameters->draw_fb = bloom->fbs[i * BLOOM_FBS];
        parameters->program = &bloom->blur_y_programs[i];
        parameters->textures =
            (GLuint[]){bloom->fbs[i * BLOOM_FBS + 1]->texture};
        pass_render(parameters);
    }
}

GLuint bloom_get_texture(bloom_t *bloom) { return bloom->fbs[0]->texture; }
