
\ Forth Language compatibility
\ ============================

decimal

\ This adds the $ prefix for hex numbers

start-prefixes
: $ 16 __tempbase ; immediate
end-prefixes

\ Want to implement CMOVE in terms of MOVE, when I realized
\ MOVE already does what CMOVE does.
\ I can still define a trivial definition of CMOVE and CMOVE>
\ just for fun.

: power-off ;

: 2variable ( "<spaces>name" -- )
  create 0 , 0 , ;

: variable
  create 0 , ;

\ LCD line rendering
\ ==================

variable lcd-x ( -- a-addr )
\ ** holds the x coordinate of the current position
variable lcd-y ( -- a-addr )
\ ** holds the y coordinate of the current position

variable lcd-line-colour ( -- a-addr )
\ ** if this variable is true the drawing colour is black
\ ** false means drawing colour is white

: lcd-black ( -- )
\ ** set the drawing colour for text / lines to black
  true lcd-line-colour !
;

: lcd-white ( -- )
\ ** set the drawing colour for text / lines to white
  false lcd-line-colour !
;

lcd-black

1 lcd-x !
1 lcd-y !

: lcd-move-to ( x y -- )
\ ** set the current position to (x, y)
\ ** note that (0, 0) is the top left of the display
  lcd-y ! lcd-x ! ;

: lcd-line-to ( x y -- )
\ ** draw a line from the current position to (x, y)
  2dup lcd-x @ lcd-y @ lcd-line lcd-move-to ;

: lcd-home ( -- )
\ ** move the cursor to first line, first character
  0 0 lcd-move-to ;

: lcd-line-rel ( dx dy -- )
\ ** draw a line from the current position to (x + dx, y + dy)
  lcd-y @ + swap lcd-x @ + swap
  2dup lcd-x @ lcd-y @ lcd-line lcd-move-to ;

: lcd-move-rel ( dx dy -- )
\ ** set the current position to (x + dx, y + dy)
  lcd-y +! lcd-x +! ;

: lcd-box ( width height -- )
\ ** draw a square or rectangle starting from current postion
\ ** the top right is at (x, y)
\ ** the bottom left is at (w + width, y + height)
    1- swap 1- swap
    over lcd-x @ + lcd-y @ lcd-line-to
    lcd-x @ over lcd-y @ + lcd-line-to
    lcd-x @ rot - lcd-y @ lcd-line-to
    lcd-x @ lcd-y @ rot - lcd-line-to
;

\ LCD direct to framebuffer rendering
\ ===================================

240
constant lcd-width-pixels

208
constant lcd-height-pixels

lcd-width-pixels 31 + 32 / 32 *
constant lcd-vram-width-pixels

lcd-height-pixels
constant lcd-vram-height-pixels

lcd-vram-width-pixels 8 /
constant lcd-vram-width-bytes

lcd-width-pixels 8 /
constant lcd-width-bytes

lcd-vram-width-bytes lcd-vram-height-pixels *
constant lcd-vram-size

: lcd-clear-all ( -- )
\ ** zero the entire frame buffer
\ ** results in a white display
  lcd-vram lcd-vram-size 0 fill
;

: lcd-set-all ( -- )
\ ** set all bits in the entire frame buffer
\ ** results in a black display
  lcd-vram lcd-vram-size 255 fill
;

: lcd-cls ( -- )
\ ** clear the screen, home the cursor and set the colour to black
  lcd-clear-all lcd-black lcd-home ;


\ LCD text rendering
\ ==================
  
8
constant font-width

13
constant font-height

\ character based positioning
lcd-width-pixels font-width /
constant lcd-text-columns
\ ** the number of characters in the fixed width font that fit on one line

lcd-height-pixels font-height /
constant lcd-text-rows
\ ** the number of lines of character in the fixed width font that fit on the LCD

: lcd-at-xy ( x y -- )
\ ** set the curent position to a character position
\ ** 0 <= x <= lcd-last-columns - 1
\ ** 0 <= y <= lcd-text-rows - 1
  font-height * swap
  font-width * swap
  lcd-move-to ;

hex
create font-8x13
\ ** a fixed width 8x13 font
( 0000 ) $00 c, $00 c, $AA c, $00 c, $82 c, $00 c, $82 c, $00 c, $82 c, $00 c, $AA c, $00 c, $00 c,
( 0001 ) $00 c, $00 c, $00 c, $10 c, $38 c, $7C c, $FE c, $7C c, $38 c, $10 c, $00 c, $00 c, $00 c,
( 0002 ) $AA c, $55 c, $AA c, $55 c, $AA c, $55 c, $AA c, $55 c, $AA c, $55 c, $AA c, $55 c, $AA c,
( 0003 ) $00 c, $00 c, $A0 c, $A0 c, $E0 c, $A0 c, $AE c, $04 c, $04 c, $04 c, $04 c, $00 c, $00 c,
( 0004 ) $00 c, $00 c, $E0 c, $80 c, $C0 c, $80 c, $8E c, $08 c, $0C c, $08 c, $08 c, $00 c, $00 c,
( 0005 ) $00 c, $00 c, $60 c, $80 c, $80 c, $80 c, $6C c, $0A c, $0C c, $0A c, $0A c, $00 c, $00 c,
( 0006 ) $00 c, $00 c, $80 c, $80 c, $80 c, $80 c, $EE c, $08 c, $0C c, $08 c, $08 c, $00 c, $00 c,
( 0007 ) $00 c, $00 c, $18 c, $24 c, $24 c, $18 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0008 ) $00 c, $00 c, $00 c, $10 c, $10 c, $7C c, $10 c, $10 c, $00 c, $7C c, $00 c, $00 c, $00 c,
( 0009 ) $00 c, $00 c, $C0 c, $A0 c, $A0 c, $A0 c, $A8 c, $08 c, $08 c, $08 c, $0E c, $00 c, $00 c,
( 000A ) $00 c, $00 c, $88 c, $88 c, $50 c, $50 c, $2E c, $04 c, $04 c, $04 c, $04 c, $00 c, $00 c,
( 000B ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $F0 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 000C ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $F0 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 000D ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $1F c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 000E ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $1F c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 000F ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $FF c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 0010 ) $FF c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0011 ) $00 c, $00 c, $00 c, $FF c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0012 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $FF c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0013 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $FF c, $00 c, $00 c, $00 c,
( 0014 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $FF c,
( 0015 ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $1F c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 0016 ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $F0 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 0017 ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $FF c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0018 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $FF c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 0019 ) $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c,
( 001A ) $00 c, $00 c, $00 c, $00 c, $0E c, $30 c, $C0 c, $30 c, $0E c, $00 c, $FE c, $00 c, $00 c,
( 001B ) $00 c, $00 c, $00 c, $00 c, $E0 c, $18 c, $06 c, $18 c, $E0 c, $00 c, $FE c, $00 c, $00 c,
( 001C ) $00 c, $00 c, $00 c, $00 c, $00 c, $FE c, $44 c, $44 c, $44 c, $44 c, $44 c, $00 c, $00 c,
( 001D ) $00 c, $00 c, $00 c, $04 c, $04 c, $7E c, $08 c, $10 c, $7E c, $20 c, $20 c, $00 c, $00 c,
( 001E ) $00 c, $00 c, $1C c, $22 c, $20 c, $70 c, $20 c, $20 c, $20 c, $62 c, $DC c, $00 c, $00 c,
( 001F ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $18 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0020 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0021 ) $00 c, $00 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $00 c, $10 c, $00 c, $00 c,
( 0022 ) $00 c, $00 c, $24 c, $24 c, $24 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0023 ) $00 c, $00 c, $00 c, $24 c, $24 c, $7E c, $24 c, $7E c, $24 c, $24 c, $00 c, $00 c, $00 c,
( 0024 ) $00 c, $00 c, $10 c, $3C c, $50 c, $50 c, $38 c, $14 c, $14 c, $78 c, $10 c, $00 c, $00 c,
( 0025 ) $00 c, $00 c, $22 c, $52 c, $24 c, $08 c, $08 c, $10 c, $24 c, $2A c, $44 c, $00 c, $00 c,
( 0026 ) $00 c, $00 c, $00 c, $00 c, $30 c, $48 c, $48 c, $30 c, $4A c, $44 c, $3A c, $00 c, $00 c,
( 0027 ) $00 c, $00 c, $10 c, $10 c, $10 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0028 ) $00 c, $00 c, $04 c, $08 c, $08 c, $10 c, $10 c, $10 c, $08 c, $08 c, $04 c, $00 c, $00 c,
( 0029 ) $00 c, $00 c, $20 c, $10 c, $10 c, $08 c, $08 c, $08 c, $10 c, $10 c, $20 c, $00 c, $00 c,
( 002A ) $00 c, $00 c, $24 c, $18 c, $7E c, $18 c, $24 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 002B ) $00 c, $00 c, $00 c, $00 c, $10 c, $10 c, $7C c, $10 c, $10 c, $00 c, $00 c, $00 c, $00 c,
( 002C ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $38 c, $30 c, $40 c, $00 c,
( 002D ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $7C c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 002E ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $10 c, $38 c, $10 c, $00 c,
( 002F ) $00 c, $00 c, $02 c, $02 c, $04 c, $08 c, $10 c, $20 c, $40 c, $80 c, $80 c, $00 c, $00 c,
( 0030 ) $00 c, $00 c, $18 c, $24 c, $42 c, $42 c, $42 c, $42 c, $42 c, $24 c, $18 c, $00 c, $00 c,
( 0031 ) $00 c, $00 c, $10 c, $30 c, $50 c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 0032 ) $00 c, $00 c, $3C c, $42 c, $42 c, $02 c, $04 c, $18 c, $20 c, $40 c, $7E c, $00 c, $00 c,
( 0033 ) $00 c, $00 c, $7E c, $02 c, $04 c, $08 c, $1C c, $02 c, $02 c, $42 c, $3C c, $00 c, $00 c,
( 0034 ) $00 c, $00 c, $04 c, $0C c, $14 c, $24 c, $44 c, $44 c, $7E c, $04 c, $04 c, $00 c, $00 c,
( 0035 ) $00 c, $00 c, $7E c, $40 c, $40 c, $5C c, $62 c, $02 c, $02 c, $42 c, $3C c, $00 c, $00 c,
( 0036 ) $00 c, $00 c, $1C c, $20 c, $40 c, $40 c, $5C c, $62 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 0037 ) $00 c, $00 c, $7E c, $02 c, $04 c, $08 c, $08 c, $10 c, $10 c, $20 c, $20 c, $00 c, $00 c,
( 0038 ) $00 c, $00 c, $3C c, $42 c, $42 c, $42 c, $3C c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 0039 ) $00 c, $00 c, $3C c, $42 c, $42 c, $46 c, $3A c, $02 c, $02 c, $04 c, $38 c, $00 c, $00 c,
( 003A ) $00 c, $00 c, $00 c, $00 c, $10 c, $38 c, $10 c, $00 c, $00 c, $10 c, $38 c, $10 c, $00 c,
( 003B ) $00 c, $00 c, $00 c, $00 c, $10 c, $38 c, $10 c, $00 c, $00 c, $38 c, $30 c, $40 c, $00 c,
( 003C ) $00 c, $00 c, $02 c, $04 c, $08 c, $10 c, $20 c, $10 c, $08 c, $04 c, $02 c, $00 c, $00 c,
( 003D ) $00 c, $00 c, $00 c, $00 c, $00 c, $7E c, $00 c, $00 c, $7E c, $00 c, $00 c, $00 c, $00 c,
( 003E ) $00 c, $00 c, $40 c, $20 c, $10 c, $08 c, $04 c, $08 c, $10 c, $20 c, $40 c, $00 c, $00 c,
( 003F ) $00 c, $00 c, $3C c, $42 c, $42 c, $02 c, $04 c, $08 c, $08 c, $00 c, $08 c, $00 c, $00 c,
( 0040 ) $00 c, $00 c, $3C c, $42 c, $42 c, $4E c, $52 c, $56 c, $4A c, $40 c, $3C c, $00 c, $00 c,
( 0041 ) $00 c, $00 c, $18 c, $24 c, $42 c, $42 c, $42 c, $7E c, $42 c, $42 c, $42 c, $00 c, $00 c,
( 0042 ) $00 c, $00 c, $78 c, $44 c, $42 c, $44 c, $78 c, $44 c, $42 c, $44 c, $78 c, $00 c, $00 c,
( 0043 ) $00 c, $00 c, $3C c, $42 c, $40 c, $40 c, $40 c, $40 c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 0044 ) $00 c, $00 c, $78 c, $44 c, $42 c, $42 c, $42 c, $42 c, $42 c, $44 c, $78 c, $00 c, $00 c,
( 0045 ) $00 c, $00 c, $7E c, $40 c, $40 c, $40 c, $78 c, $40 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 0046 ) $00 c, $00 c, $7E c, $40 c, $40 c, $40 c, $78 c, $40 c, $40 c, $40 c, $40 c, $00 c, $00 c,
( 0047 ) $00 c, $00 c, $3C c, $42 c, $40 c, $40 c, $40 c, $4E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 0048 ) $00 c, $00 c, $42 c, $42 c, $42 c, $42 c, $7E c, $42 c, $42 c, $42 c, $42 c, $00 c, $00 c,
( 0049 ) $00 c, $00 c, $7C c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 004A ) $00 c, $00 c, $1F c, $04 c, $04 c, $04 c, $04 c, $04 c, $04 c, $44 c, $38 c, $00 c, $00 c,
( 004B ) $00 c, $00 c, $42 c, $44 c, $48 c, $50 c, $60 c, $50 c, $48 c, $44 c, $42 c, $00 c, $00 c,
( 004C ) $00 c, $00 c, $40 c, $40 c, $40 c, $40 c, $40 c, $40 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 004D ) $00 c, $00 c, $82 c, $82 c, $C6 c, $AA c, $92 c, $92 c, $82 c, $82 c, $82 c, $00 c, $00 c,
( 004E ) $00 c, $00 c, $42 c, $42 c, $62 c, $52 c, $4A c, $46 c, $42 c, $42 c, $42 c, $00 c, $00 c,
( 004F ) $00 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 0050 ) $00 c, $00 c, $7C c, $42 c, $42 c, $42 c, $7C c, $40 c, $40 c, $40 c, $40 c, $00 c, $00 c,
( 0051 ) $00 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $42 c, $52 c, $4A c, $3C c, $02 c, $00 c,
( 0052 ) $00 c, $00 c, $7C c, $42 c, $42 c, $42 c, $7C c, $50 c, $48 c, $44 c, $42 c, $00 c, $00 c,
( 0053 ) $00 c, $00 c, $3C c, $42 c, $40 c, $40 c, $3C c, $02 c, $02 c, $42 c, $3C c, $00 c, $00 c,
( 0054 ) $00 c, $00 c, $FE c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $00 c, $00 c,
( 0055 ) $00 c, $00 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 0056 ) $00 c, $00 c, $82 c, $82 c, $44 c, $44 c, $44 c, $28 c, $28 c, $28 c, $10 c, $00 c, $00 c,
( 0057 ) $00 c, $00 c, $82 c, $82 c, $82 c, $82 c, $92 c, $92 c, $92 c, $AA c, $44 c, $00 c, $00 c,
( 0058 ) $00 c, $00 c, $82 c, $82 c, $44 c, $28 c, $10 c, $28 c, $44 c, $82 c, $82 c, $00 c, $00 c,
( 0059 ) $00 c, $00 c, $82 c, $82 c, $44 c, $28 c, $10 c, $10 c, $10 c, $10 c, $10 c, $00 c, $00 c,
( 005A ) $00 c, $00 c, $7E c, $02 c, $04 c, $08 c, $10 c, $20 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 005B ) $00 c, $00 c, $3C c, $20 c, $20 c, $20 c, $20 c, $20 c, $20 c, $20 c, $3C c, $00 c, $00 c,
( 005C ) $00 c, $00 c, $80 c, $80 c, $40 c, $20 c, $10 c, $08 c, $04 c, $02 c, $02 c, $00 c, $00 c,
( 005D ) $00 c, $00 c, $78 c, $08 c, $08 c, $08 c, $08 c, $08 c, $08 c, $08 c, $78 c, $00 c, $00 c,
( 005E ) $00 c, $00 c, $10 c, $28 c, $44 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 005F ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $FE c, $00 c,
( 0060 ) $00 c, $10 c, $08 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0061 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 0062 ) $00 c, $00 c, $40 c, $40 c, $40 c, $5C c, $62 c, $42 c, $42 c, $62 c, $5C c, $00 c, $00 c,
( 0063 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $42 c, $40 c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 0064 ) $00 c, $00 c, $02 c, $02 c, $02 c, $3A c, $46 c, $42 c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 0065 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $42 c, $7E c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 0066 ) $00 c, $00 c, $1C c, $22 c, $20 c, $20 c, $7C c, $20 c, $20 c, $20 c, $20 c, $00 c, $00 c,
( 0067 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3A c, $44 c, $44 c, $38 c, $40 c, $3C c, $42 c, $3C c,
( 0068 ) $00 c, $00 c, $40 c, $40 c, $40 c, $5C c, $62 c, $42 c, $42 c, $42 c, $42 c, $00 c, $00 c,
( 0069 ) $00 c, $00 c, $00 c, $10 c, $00 c, $30 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 006A ) $00 c, $00 c, $00 c, $04 c, $00 c, $0C c, $04 c, $04 c, $04 c, $04 c, $44 c, $44 c, $38 c,
( 006B ) $00 c, $00 c, $40 c, $40 c, $40 c, $44 c, $48 c, $70 c, $48 c, $44 c, $42 c, $00 c, $00 c,
( 006C ) $00 c, $00 c, $30 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 006D ) $00 c, $00 c, $00 c, $00 c, $00 c, $EC c, $92 c, $92 c, $92 c, $92 c, $82 c, $00 c, $00 c,
( 006E ) $00 c, $00 c, $00 c, $00 c, $00 c, $5C c, $62 c, $42 c, $42 c, $42 c, $42 c, $00 c, $00 c,
( 006F ) $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 0070 ) $00 c, $00 c, $00 c, $00 c, $00 c, $5C c, $62 c, $42 c, $62 c, $5C c, $40 c, $40 c, $40 c,
( 0071 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3A c, $46 c, $42 c, $46 c, $3A c, $02 c, $02 c, $02 c,
( 0072 ) $00 c, $00 c, $00 c, $00 c, $00 c, $5C c, $22 c, $20 c, $20 c, $20 c, $20 c, $00 c, $00 c,
( 0073 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $42 c, $30 c, $0C c, $42 c, $3C c, $00 c, $00 c,
( 0074 ) $00 c, $00 c, $00 c, $20 c, $20 c, $7C c, $20 c, $20 c, $20 c, $22 c, $1C c, $00 c, $00 c,
( 0075 ) $00 c, $00 c, $00 c, $00 c, $00 c, $44 c, $44 c, $44 c, $44 c, $44 c, $3A c, $00 c, $00 c,
( 0076 ) $00 c, $00 c, $00 c, $00 c, $00 c, $44 c, $44 c, $44 c, $28 c, $28 c, $10 c, $00 c, $00 c,
( 0077 ) $00 c, $00 c, $00 c, $00 c, $00 c, $82 c, $82 c, $92 c, $92 c, $AA c, $44 c, $00 c, $00 c,
( 0078 ) $00 c, $00 c, $00 c, $00 c, $00 c, $42 c, $24 c, $18 c, $18 c, $24 c, $42 c, $00 c, $00 c,
( 0079 ) $00 c, $00 c, $00 c, $00 c, $00 c, $42 c, $42 c, $42 c, $46 c, $3A c, $02 c, $42 c, $3C c,
( 007A ) $00 c, $00 c, $00 c, $00 c, $00 c, $7E c, $04 c, $08 c, $10 c, $20 c, $7E c, $00 c, $00 c,
( 007B ) $00 c, $00 c, $0E c, $10 c, $10 c, $08 c, $30 c, $08 c, $10 c, $10 c, $0E c, $00 c, $00 c,
( 007C ) $00 c, $00 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $00 c, $00 c,
( 007D ) $00 c, $00 c, $70 c, $08 c, $08 c, $10 c, $0C c, $10 c, $08 c, $08 c, $70 c, $00 c, $00 c,
( 007E ) $00 c, $00 c, $24 c, $54 c, $48 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,

( spare codes )
( 007F ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0080 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0081 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0082 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0083 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0084 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0085 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0086 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0087 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0088 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0089 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 008A ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 008B ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 008C ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 008D ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 008E ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 008F ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0090 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0091 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0092 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0093 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0094 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0095 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0096 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0097 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0098 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 0099 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 009A ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 009B ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 009C ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 009D ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 009E ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 009F ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,

( 00A0 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00A1 ) $00 c, $00 c, $10 c, $00 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $10 c, $00 c, $00 c,
( 00A2 ) $00 c, $00 c, $10 c, $38 c, $54 c, $50 c, $50 c, $54 c, $38 c, $10 c, $00 c, $00 c, $00 c,
( 00A3 ) $00 c, $00 c, $1C c, $22 c, $20 c, $70 c, $20 c, $20 c, $20 c, $62 c, $DC c, $00 c, $00 c,
( 00A4 ) $00 c, $00 c, $00 c, $00 c, $42 c, $3C c, $24 c, $24 c, $3C c, $42 c, $00 c, $00 c, $00 c,
( 00A5 ) $00 c, $00 c, $82 c, $82 c, $44 c, $28 c, $7C c, $10 c, $7C c, $10 c, $10 c, $00 c, $00 c,
( 00A6 ) $00 c, $00 c, $10 c, $10 c, $10 c, $10 c, $00 c, $10 c, $10 c, $10 c, $10 c, $00 c, $00 c,
( 00A7 ) $00 c, $18 c, $24 c, $20 c, $18 c, $24 c, $24 c, $18 c, $04 c, $24 c, $18 c, $00 c, $00 c,
( 00A8 ) $00 c, $24 c, $24 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00A9 ) $00 c, $38 c, $44 c, $92 c, $AA c, $A2 c, $AA c, $92 c, $44 c, $38 c, $00 c, $00 c, $00 c,
( 00AA ) $00 c, $00 c, $38 c, $04 c, $3C c, $44 c, $3C c, $00 c, $7C c, $00 c, $00 c, $00 c, $00 c,
( 00AB ) $00 c, $00 c, $00 c, $12 c, $24 c, $48 c, $90 c, $48 c, $24 c, $12 c, $00 c, $00 c, $00 c,
( 00AC ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $7E c, $02 c, $02 c, $02 c, $00 c, $00 c, $00 c,
( 00AD ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00AE ) $00 c, $38 c, $44 c, $92 c, $AA c, $AA c, $B2 c, $AA c, $44 c, $38 c, $00 c, $00 c, $00 c,
( 00AF ) $00 c, $00 c, $7E c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00B0 ) $00 c, $00 c, $18 c, $24 c, $24 c, $18 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00B1 ) $00 c, $00 c, $00 c, $10 c, $10 c, $7C c, $10 c, $10 c, $00 c, $7C c, $00 c, $00 c, $00 c,
( 00B2 ) $00 c, $30 c, $48 c, $08 c, $30 c, $40 c, $78 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00B3 ) $00 c, $30 c, $48 c, $10 c, $08 c, $48 c, $30 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00B4 ) $00 c, $08 c, $10 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00B5 ) $00 c, $00 c, $00 c, $00 c, $00 c, $42 c, $42 c, $42 c, $42 c, $66 c, $5A c, $40 c, $00 c,
( 00B6 ) $00 c, $00 c, $3E c, $74 c, $74 c, $74 c, $34 c, $14 c, $14 c, $14 c, $14 c, $00 c, $00 c,
( 00B7 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $18 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00B8 ) $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c, $08 c, $18 c,
( 00B9 ) $00 c, $20 c, $60 c, $20 c, $20 c, $20 c, $70 c, $00 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00BA ) $00 c, $00 c, $30 c, $48 c, $48 c, $30 c, $00 c, $78 c, $00 c, $00 c, $00 c, $00 c, $00 c,
( 00BB ) $00 c, $00 c, $00 c, $90 c, $48 c, $24 c, $12 c, $24 c, $48 c, $90 c, $00 c, $00 c, $00 c,
( 00BC ) $00 c, $40 c, $C0 c, $40 c, $40 c, $42 c, $E6 c, $0A c, $12 c, $1A c, $06 c, $00 c, $00 c,
( 00BD ) $00 c, $40 c, $C0 c, $40 c, $40 c, $4C c, $F2 c, $02 c, $0C c, $10 c, $1E c, $00 c, $00 c,
( 00BE ) $00 c, $60 c, $90 c, $20 c, $10 c, $92 c, $66 c, $0A c, $12 c, $1A c, $06 c, $00 c, $00 c,
( 00BF ) $00 c, $00 c, $10 c, $00 c, $10 c, $10 c, $20 c, $40 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00C0 ) $00 c, $10 c, $08 c, $00 c, $18 c, $24 c, $42 c, $42 c, $7E c, $42 c, $42 c, $00 c, $00 c,
( 00C1 ) $00 c, $08 c, $10 c, $00 c, $18 c, $24 c, $42 c, $42 c, $7E c, $42 c, $42 c, $00 c, $00 c,
( 00C2 ) $00 c, $18 c, $24 c, $00 c, $18 c, $24 c, $42 c, $42 c, $7E c, $42 c, $42 c, $00 c, $00 c,
( 00C3 ) $00 c, $32 c, $4C c, $00 c, $18 c, $24 c, $42 c, $42 c, $7E c, $42 c, $42 c, $00 c, $00 c,
( 00C4 ) $00 c, $24 c, $24 c, $00 c, $18 c, $24 c, $42 c, $42 c, $7E c, $42 c, $42 c, $00 c, $00 c,
( 00C5 ) $00 c, $18 c, $24 c, $18 c, $18 c, $24 c, $42 c, $42 c, $7E c, $42 c, $42 c, $00 c, $00 c,
( 00C6 ) $00 c, $00 c, $6E c, $90 c, $90 c, $90 c, $9C c, $F0 c, $90 c, $90 c, $9E c, $00 c, $00 c,
( 00C7 ) $00 c, $00 c, $3C c, $42 c, $40 c, $40 c, $40 c, $40 c, $40 c, $42 c, $3C c, $08 c, $10 c,
( 00C8 ) $00 c, $10 c, $08 c, $00 c, $7E c, $40 c, $40 c, $78 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 00C9 ) $00 c, $08 c, $10 c, $00 c, $7E c, $40 c, $40 c, $78 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 00CA ) $00 c, $18 c, $24 c, $00 c, $7E c, $40 c, $40 c, $78 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 00CB ) $00 c, $24 c, $24 c, $00 c, $7E c, $40 c, $40 c, $78 c, $40 c, $40 c, $7E c, $00 c, $00 c,
( 00CC ) $00 c, $20 c, $10 c, $00 c, $7C c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00CD ) $00 c, $08 c, $10 c, $00 c, $7C c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00CE ) $00 c, $18 c, $24 c, $00 c, $7C c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00CF ) $00 c, $44 c, $44 c, $00 c, $7C c, $10 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00D0 ) $00 c, $00 c, $78 c, $44 c, $42 c, $42 c, $E2 c, $42 c, $42 c, $44 c, $78 c, $00 c, $00 c,
( 00D1 ) $00 c, $64 c, $98 c, $00 c, $82 c, $C2 c, $A2 c, $92 c, $8A c, $86 c, $82 c, $00 c, $00 c,
( 00D2 ) $00 c, $20 c, $10 c, $00 c, $7C c, $82 c, $82 c, $82 c, $82 c, $82 c, $7C c, $00 c, $00 c,
( 00D3 ) $00 c, $08 c, $10 c, $00 c, $7C c, $82 c, $82 c, $82 c, $82 c, $82 c, $7C c, $00 c, $00 c,
( 00D4 ) $00 c, $18 c, $24 c, $00 c, $7C c, $82 c, $82 c, $82 c, $82 c, $82 c, $7C c, $00 c, $00 c,
( 00D5 ) $00 c, $64 c, $98 c, $00 c, $7C c, $82 c, $82 c, $82 c, $82 c, $82 c, $7C c, $00 c, $00 c,
( 00D6 ) $00 c, $44 c, $44 c, $00 c, $7C c, $82 c, $82 c, $82 c, $82 c, $82 c, $7C c, $00 c, $00 c,
( 00D7 ) $00 c, $00 c, $00 c, $00 c, $42 c, $24 c, $18 c, $18 c, $24 c, $42 c, $00 c, $00 c, $00 c,
( 00D8 ) $00 c, $02 c, $3C c, $46 c, $4A c, $4A c, $52 c, $52 c, $52 c, $62 c, $3C c, $40 c, $00 c,
( 00D9 ) $00 c, $20 c, $10 c, $00 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00DA ) $00 c, $08 c, $10 c, $00 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00DB ) $00 c, $18 c, $24 c, $00 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00DC ) $00 c, $24 c, $24 c, $00 c, $42 c, $42 c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00DD ) $00 c, $08 c, $10 c, $00 c, $44 c, $44 c, $28 c, $10 c, $10 c, $10 c, $10 c, $00 c, $00 c,
( 00DE ) $00 c, $00 c, $40 c, $7C c, $42 c, $42 c, $42 c, $7C c, $40 c, $40 c, $40 c, $00 c, $00 c,
( 00DF ) $00 c, $00 c, $38 c, $44 c, $44 c, $48 c, $50 c, $4C c, $42 c, $42 c, $5C c, $00 c, $00 c,
( 00E0 ) $00 c, $00 c, $10 c, $08 c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 00E1 ) $00 c, $00 c, $04 c, $08 c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 00E2 ) $00 c, $00 c, $18 c, $24 c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 00E3 ) $00 c, $00 c, $32 c, $4C c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 00E4 ) $00 c, $00 c, $24 c, $24 c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 00E5 ) $00 c, $18 c, $24 c, $18 c, $00 c, $3C c, $02 c, $3E c, $42 c, $46 c, $3A c, $00 c, $00 c,
( 00E6 ) $00 c, $00 c, $00 c, $00 c, $00 c, $6C c, $12 c, $7C c, $90 c, $92 c, $6C c, $00 c, $00 c,
( 00E7 ) $00 c, $00 c, $00 c, $00 c, $00 c, $3C c, $42 c, $40 c, $40 c, $42 c, $3C c, $08 c, $10 c,
( 00E8 ) $00 c, $00 c, $10 c, $08 c, $00 c, $3C c, $42 c, $7E c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 00E9 ) $00 c, $00 c, $08 c, $10 c, $00 c, $3C c, $42 c, $7E c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 00EA ) $00 c, $00 c, $18 c, $24 c, $00 c, $3C c, $42 c, $7E c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 00EB ) $00 c, $00 c, $24 c, $24 c, $00 c, $3C c, $42 c, $7E c, $40 c, $42 c, $3C c, $00 c, $00 c,
( 00EC ) $00 c, $00 c, $20 c, $10 c, $00 c, $30 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00ED ) $00 c, $00 c, $10 c, $20 c, $00 c, $30 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00EE ) $00 c, $00 c, $30 c, $48 c, $00 c, $30 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00EF ) $00 c, $00 c, $48 c, $48 c, $00 c, $30 c, $10 c, $10 c, $10 c, $10 c, $7C c, $00 c, $00 c,
( 00F0 ) $00 c, $24 c, $18 c, $28 c, $04 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00F1 ) $00 c, $00 c, $32 c, $4C c, $00 c, $5C c, $62 c, $42 c, $42 c, $42 c, $42 c, $00 c, $00 c,
( 00F2 ) $00 c, $00 c, $20 c, $10 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00F3 ) $00 c, $00 c, $08 c, $10 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00F4 ) $00 c, $00 c, $18 c, $24 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00F5 ) $00 c, $00 c, $32 c, $4C c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00F6 ) $00 c, $00 c, $24 c, $24 c, $00 c, $3C c, $42 c, $42 c, $42 c, $42 c, $3C c, $00 c, $00 c,
( 00F7 ) $00 c, $00 c, $00 c, $10 c, $10 c, $00 c, $7C c, $00 c, $10 c, $10 c, $00 c, $00 c, $00 c,
( 00F8 ) $00 c, $00 c, $00 c, $00 c, $02 c, $3C c, $46 c, $4A c, $52 c, $62 c, $3C c, $40 c, $00 c,
( 00F9 ) $00 c, $00 c, $20 c, $10 c, $00 c, $44 c, $44 c, $44 c, $44 c, $44 c, $3A c, $00 c, $00 c,
( 00FA ) $00 c, $00 c, $08 c, $10 c, $00 c, $44 c, $44 c, $44 c, $44 c, $44 c, $3A c, $00 c, $00 c,
( 00FB ) $00 c, $00 c, $18 c, $24 c, $00 c, $44 c, $44 c, $44 c, $44 c, $44 c, $3A c, $00 c, $00 c,
( 00FC ) $00 c, $00 c, $28 c, $28 c, $00 c, $44 c, $44 c, $44 c, $44 c, $44 c, $3A c, $00 c, $00 c,
( 00FD ) $00 c, $00 c, $08 c, $10 c, $00 c, $42 c, $42 c, $42 c, $46 c, $3A c, $02 c, $42 c, $3C c,
( 00FE ) $00 c, $00 c, $00 c, $40 c, $40 c, $5C c, $62 c, $42 c, $42 c, $62 c, $5C c, $40 c, $40 c,
( 00FF ) $00 c, $00 c, $24 c, $24 c, $00 c, $42 c, $42 c, $42 c, $46 c, $3A c, $02 c, $42 c, $3C c,
decimal

: lcd-scroll ( -- )
\ ** scroll the LCD display so the enough space for a new line
\ ** of characters apear at the bottom of the display
  font-height lcd-vram-width-bytes * dup dup   \ u u u
  lcd-vram + swap                              \ u c-addr u

  lcd-vram-size swap -                         \ u c-addr n
  lcd-vram swap move                           \ u

  lcd-vram lcd-vram-size + 1-                  \ u c-addr
  over - swap 0 fill
;

: lcd-scroll> ( -- )
\ ** reverse scroll the LCD display so the enough space for a new line
\ ** of characters apear at the top of the display
  font-height lcd-vram-width-bytes * dup dup   \ u u u
  lcd-vram + swap                              \ u c-addr u

  lcd-vram-size swap -                         \ u c-addr n
  lcd-vram -rot                                \ u c-addr2 c-addr n
  move                                         \ u

  lcd-vram swap 0 fill
;

: lcd-cr ( -- )
\ ** adjust the current position so that is a the start of a character line
\ ** if this would be off the bottom of the display then scroll and
\ ** set the position to the start of the last line
  0 lcd-x !
  lcd-y @ font-height + dup lcd-height-pixels 1- > if
    drop
	lcd-scroll
  else
    lcd-y !
  then
;

: lcd-emit ( c -- )
\ ** output one character to the display, advancing to the next line
\ ** or scrolling if necessary
  lcd-x @ lcd-width-pixels 1- > if
    lcd-cr
  then
  lcd-y @ lcd-vram-width-bytes * lcd-vram +
  lcd-x @ 3 rshift +           ( c c-addr )
  swap                         ( c-addr c )
  font-height * font-8x13 +    ( lcd-addr font-addr )
  font-height 0 ?do
    2dup c@
    lcd-line-colour @ 0= if invert then
    swap c!
    char+ swap lcd-vram-width-bytes + swap
  loop 2drop
  font-width lcd-x +!
;

: lcd-space ( -- )
\ ** output a sincle space to the LCD
  bl lcd-emit ;

: lcd-spaces ( u -- )
\ ** output 'u' spaces to the LCD display
  dup 0> if
    0 ?do lcd-space loop
  else
    drop
  then ;

: lcd-type ( caddr u -- )
\ ** output a string to the LCD display
  0 ?do
    dup c@ lcd-emit char+
  loop drop ;

: lcd-." ( "ccc<quote>" -- )
\ ** output a constant string to the LCD display
  postpone s" postpone lcd-type ; immediate compile-only

: dabs ( d -- ud )
  dup 0< if dnegate then ;

\ OK, now I need to get these numbers working.

\ LCD numeric output
\ =================

\ these are all separate functions as most test programs
\ will output results to the console (emit . u.) etc.
\ and will display on the lcd.  Switching vectors around
\ will be slower.

: lcd-number ( n -- )
\ ** display a number on the LCD in fixed 8 digit format
  s>d <# # # # # # # # # #> lcd-type ;

: lcd-d. ( d -- )
\ ** display signed double number followed by a space on the LCD
  swap over dabs <# #s rot sign #> lcd-type lcd-space ;

: lcd-d.r ( d n -- )
\ ** display right-justified signed double number on the LCD
  >r swap over dabs <# #s rot sign #> r> over - lcd-spaces lcd-type ;

: lcd-.r ( n1 n2 -- )
\ ** display right-justified signed number on the LCD
  >r s>d r> lcd-d.r ;

: lcd-u. ( u -- )
\ ** display unsigned number followed by a space on the LCD
  0 <# #s #> lcd-type lcd-space ;

: lcd-u.r ( u n -- )
\ ** display right-justified unsigned number on the LCD
  >r 0 <# #s #> r> over - lcd-spaces lcd-type ;

: lcd-. ( n -- )
\ ** display unsigned number followed by a space on the LCD
  s>d lcd-d. ;

: lcd-dec. ( n -- )
\ ** as lcd-. but setting the bas to decimal
\ ** (base is preseved for calling program)
  base @ decimal swap lcd-. base ! ;

: lcd-hex. ( n -- )
\ ** as lcd-. but setting the bas to decimal
\ ** (base is preseved for calling program)
  base @ hex swap lcd-u. base ! ;

\ Buttons
\ =======

0
constant button-none ( -- u )
\ ** button-poll return if no buttons are pressed

2
constant button-left ( -- u )
\ ** the left (search) button

4
constant button-centre ( -- u )
\ ** the centre (history) button

1
constant button-right ( -- u )
\ ** the right (random) button

16
constant button-power ( -- u )
\ ** a short press on the power button (< 1 second)
\ ** Notes:
\ **   no power off is preformed by this
\ **   button-poll cannot detect this condition
\ **   a long press is a hardware shutdown so cannot be detected by software


\ CTP
\ ===

: ctp-char ( -- x y )
\ ** read the net position evenr and align it to a character position
\ ** (0, 0)  is the top left character
\ ** (-1, -1) => release
  ctp-pos dup 0< if exit then
  swap font-width /
  swap font-height /
;
