/*
 * Copyright (C) 1998, 1999 Wolfgang Moser
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Commodore CBM 1581 floppy disk copy util for PC's, 1581COPY.C
 *
 * Redirected contact addresses:
 *   Wolfgang Moser <womo@mindless.com>
 *   Wolfgang Moser <womo@fairlight.to>
 *    http://www.dollar8000.de/womo
 *
 * Current workplace (up to Dezember 1999):
 *   Wolfgang Moser <womo@advm2.gm.fh-koeln.de>
 *    http://advm2.gm.fh-koeln.de/~womo
 *
 *
 * Basic informations from Dan Fandrich <dan@fch.wimsey.bc.ca>.
 *   His readme of the cbmfs-0.3 driver for Linux explained me, what the
 *   difference between a DOS formatted 800 kb disk and a CBM 1581 disk is.
 *   (check: http://vanbc.wimsey.com/~danf/software/)
 *
 *
 * Basic implementations by Ciriaco Garca de Celis <ciri@gui.uva.es>
 *   His util 765Debug, Version 5.0 is great for learning dma based
 *   direct floppy disk controller programming.
 *   (check: ftp://ftp.gui.uva.es/pub/pc/2m/765d50sr.zip)
 *
 * Check out for his floppy disk utils 2M and 2MGUI, the last words
 * in improving floppy disk storage capacity.
 *   http://www.gui.uva.es/2m, ftp://ftp.gui.uva.es/pub/pc/2m
 *
 *
 * For additional informations to FDC programming check:
 *     http://developer.intel.com/design/periphrl/datashts/290468.htm
 *   and get the intel 82078 CHMOS Single-Chip Floppy Disk Controller
 *   PDF document:
 *     http://www.intel.nl/design/periphrl/datashts/29047403.pdf
 *   National Semiconductor has also some pages about their PC
 *   compatible controllers:
 *     http://www.national.com/pf/DP/DP8473.html
 *     http://www.national.com/pf/DP/DP8477B.html
 *
 * Another good source for floppy disk controller programming information
 * are the linux kernal sources, you could have a look into:
 *     http://www.cs.utexas.edu/users/peterson/linux/include/linux/fdreg.h
 *     http://www.cs.utexas.edu/users/peterson/linux/drivers/block/floppy.c
 *
 * Now, that I started talking about Linux: Check out the most massive
 * attack against floppy disks ever made, Alain Knaff's fdutils:
 *     http://fdutils.linux.lu/
 *     http://alain.knaff.linux.lu/
 */


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "1581hlvl.h"

int main(int argc, char *argv[]){
	int i,res;
   uint8 t,s;	// track and side
   clock_t opStart;
#ifdef TimeEstimation
   clock_t ETCbase;
   uint8 TracksToDo, TracksDone;
#endif
	unsigned int args=1, earg=1, drive=0, format=0, writing=0, SwapCheck=1,
#ifdef FDCintel82078
					 FnW=0,
#endif
	             MSecRead=0, tries=3, ileave=1, BAMcp=0, Multiple=0;
	xferM verify=0;
   unsigned char BAMmark[11];				// BAM copy track buffer (88 tracks)
#ifdef DoubleTrackBuffer
   unsigned char DblTrBuf[2*TrBufSize];
   unsigned char *buffer;
#else
	unsigned char buffer[TrBufSize];		// TrBufSize #defined in CENTRAL.H
#endif
	char *flpp=NULL, *file=NULL, *temp;
	FILE *fd;

	for(t=1;t<argc;t++){
		if(argv[t]!=NULL){
			if((*argv[t]=='/' || *argv[t]=='-')){
				if(argv[t][2]=='\0'){		// arguments with no parameter
					switch(argv[t][1]){
						case 'v':
						case 'V':
							verify=Verify;
							break;
						case 'f':
						case 'F':
							format=1;
							break;
						case 'm':
						case 'M':
							Multiple=1;
							break;
						case 'b':
						case 'B':
							BAMcp=1;
							break;
						case 'p':
						case 'P':
							MSecRead=1;
							break;
						case 'd':
						case 'D':
							SwapCheck=0;
							break;
						default:
							args=0xffff;
						}
					}
				else if(argv[t][2]==':'){	// arguments with parameter
					switch(argv[t][1]){
						case 'f':
						case 'F':
#ifdef FDCintel82078
                  	if(argv[t][4]=='\0' && (argv[t][3]=='w' || argv[t][3]=='W')){
                     	format=1;
                     	FnW=1;
                     	break;
                     	}
#endif
							format=atoi(argv[t]+3);
							if(format<1 || format>2) args=0xffff;
							break;
						case 't':
						case 'T':
							tries=atoi(argv[t]+3);
							if(tries<1)	args=0xffff;
							break;
						case 'i':
						case 'I':
							ileave=atoi(argv[t]+3);
							if(ileave>9) args=0xffff;
							break;
						default:
							args=0xffff;
						}
					}
				else args=0xffff;
				}
			else{
         	earg++;
				switch(args++){				// file name arguments
					case 1:
						flpp=argv[t];
						break;
					case 2:
						file=argv[t];
					}
				}
			}
		else args=0xffff;
		if(args>3) break;
		}
	if(file!=NULL && file[0]!='\0' && file[1]==':' && file[2]=='\0') writing=1;

	if(writing){	// swap pointers
		temp=file;
		file=flpp;
		flpp=temp;
		}

   earg=(earg!=1);

	printf(_1581COPY_version_ "\n");
   do{
		if(args==0xffff){
		   args=0xfffe;
		   if(earg) printf("Error: Wrong parameter detected or help needed");
			break;
         }
		   // convert and check flpp
	   if(flpp==NULL || flpp[1]!=':' || flpp[2]!='\0'){
		   args=0xfffe;
         if(earg) printf("Error: Floppy device parameter is wrong");
         break;
         }
		drive=*flpp-((*flpp>'Z')?'a':'A');
	   if((drive & ~0x01)!=0x00){		// only drives A and B are valid
		   args=0xfffe;
     	   if(earg) printf("Error: Only floppy device parameter A: and B: are valid");
        	break;
         }

	   if(!format || writing){
		   if(args!=3){
		      args=0xfffe;
            if(earg) printf("Error: Source and destination parameters are needed");
	         break;
            }
		   }
	   else if(args!=2){					// no image file needed
	      args=0xfffe;
		   if(earg) printf("Error: Formatting needs the floppy device parameter only");
         break;
		   }
#ifdef FDCintel82078
      else if(FnW){	// args==2
	      args=0xfffe;
		   if(earg) printf("Error: Format'n'writing needs source and destination parameter");
         break;
      	}
#endif

      if(Multiple && writing){		// writing of multiple disks is not allowed
	      args=0xfffe;
         earg=1;
		   printf("Error: When using multiple disks, writing is not allowed");
         break;
   	   }
      if(verify && !writing && !format){
	      args=0xfffe;
         earg=1;
		   printf("Error: Verify cannot be used with read operations");
         break;
      	}
      }while(0);
	if(args==0xfffe && earg) printf("!\n\n");

	if(args>3){
		printf("1581COPY [/F[:x]][/V][/B][/M][/P][/D][/I:n][/T:nnn] [SOURCE] DESTINATION\n\n"
				 "SOURCE       - select floppy drive A: or B: or image file name\n"
				 "DESTINATION  - select image file name or floppy drives A: or B:\n\n"
				 "/F[:x]       - format before writing, no source needed, if formatting only\n"
				 "    x        - select desired format (2 - CMD FD 2000, default: 1 - CBM 1581)\n"
#ifdef FDCintel82078
				 "/F:W         - select 82078 format'n'write (presumably won't work, RTFM)\n"
#endif
				 "/V           - select verify mode\n\n"
				 "/B           - BAM copy, only transfer allocated tracks (no FD2000 support)\n"
				 "/M           - use multiple disks, when reading or formatting only\n"
				 "/P           - use \"read/write multiple sectors\" FDC feature\n"
				 "/D           - disable the \"swapped side ID\" CBM format check\n"
				 "/I:n         - interleave for higher speed on slow machines (0...9, default 1)\n"
				 "/T:nnn       - number of retries (> 0, default 3)\n\n"
				 "Examples:      1581COPY      A: 1581IMG.D81\n"
				 "               1581COPY /F   A:\n"
				 "               1581COPY      TESTDISK.D81 B:\n"
				 "               1581COPY /F:2 NEWDISK.D2M  A: /P /V\n");
		return 0;
		}

	CBMrecalibrate(drive);

#ifdef FDCintel82078
   switch((i=check4intelFDC())&0xFF00){
   	case 0x0100:	// extended controller
      	temp="extended floppy disk controller";
//      	FnW=0;	// don't support this feature
      	break;
   	case 0x0200:	// intel 82078 compatible controller
      	temp="dubios intel 82078 compatible controller";
//      	FnW=0;	// don't support this feature
      	break;
   	case 0x0300:	// fully 82078 compatible or original controller
      	temp="fully intel 82078 compatible or original FDC";
      	break;

   	case 0x0000:	// standard NEC PD765 compatible FDC
      	temp="NEC PD765, intel 8278, NS DP8473 or compatible FDC";
      	FnW=0;	// don't support this feature
         break;
      default:		// timeout or general error
      	temp="timeout or general error, couldn't detect";
      	FnW=0;	// don't support this feature
   	}
	if(!format || !writing) FnW=0;


	if(FnW) verify=Verify;	// Because this feature couldn't be tested yet
   								// I have to wait for some success reports and analyses

	printf("1581-Copy setup:\n - FDC type        : %s, version 0x%02X\n - Drive           : %c:\n - Retries         : %d\n - Interleave      : %d\n - Multiple sectors: %s\n - Verify          : %s\n - Formatting      : %s\n - Format'n'Write  : %s\n\n",
			 temp, i&0xFF,'A'+drive, tries, ileave, MSecRead?"on":"off", verify?"on":"off", format?"on":"off", FnW?"on (verifying forced, RTFM)":"off");
#else
	printf("1581-Copy setup:\n - Drive           : %c:\n - Retries         : %d\n - Interleave      : %d\n - Multiple sectors: %s\n - Verify          : %s\n - Formatting      : %s\n\n",
			 'A'+drive, tries, ileave, MSecRead?"on":"off", verify?"on":"off", format?"on":"off");
#endif

	printf("Low-level-%s CBM disk in drive %c:\n\n", writing?"writing":(format?"formatting":"reading"), 'A'+drive);
 	if(!waitForDiskChng(drive, 0)){
			// user abort while waiting for the first disk
      printf("User abort.\n");
		CBMrecalibrate(drive);
		switchMotorOff();
		return 2;
		}
   do{
#ifdef DoubleTrackBuffer
		buffer=DblTrBuf;
#endif
		// check source and destination formats
      do{
			if((res=CBMcheckDisk(drive,SwapCheck))<FDUNKNW){
	         printf("\nWrong drive type detected, this is no 3,5\" floppy, aborting.\n");
				CBMrecalibrate(drive);
			   switchMotorOff();
			   return 2;
            }
         if(format || res>FDUNKNW){	// CBM media detected in disk drive
            if(writing){	// search for compatible source image
					t=0;
               s=1;
               for(i=FDUNKNW;i<FDUNDEF;i++){
                  temp=changeExtension(file,0,i);
							// No problem, temp can only be NULL, if file is NULL
	               if(temp==NULL){	// new name cannot be generated
					      printf("Filename extension generation error.\n");
							CBMrecalibrate(drive);
							switchMotorOff();
					   	return 1;
   	               }

			         fd=fopen(temp,"r+b");
                  if(fd!=NULL){					// file found
                     t=1;
         	         fseek(fd,0L,SEEK_END);
                     switch(ftell(fd)){
            	         case  819200L:			// CBM 1581
                           s=2;					// CBM compatible file found
                           if(format) res=CBM1581;
               	         if(res==CBM1581) s=0;
                           break;
            	         case 1658880L:			// CMD FD2000
                           s=2;					// CBM compatible file found
                           if(format) res=CMDFD2M;
               	         if(res==CMDFD2M) s=0;
            	         }
			            fseek(fd,0L,SEEK_SET);	// seek to Track 01;00
                     if(!s) break;	// let the file open
						   fclose(fd);
                     }
                  }
               if(!s){
                  file=temp;
						break;			// let the file open (match)
                  }
               if(s==1){
                  if(!t) printf("File \"%s\" open error.\n",file);
						else printf("No CBM compatible disk images found.");
						CBMrecalibrate(drive);
					   switchMotorOff();
		   	      return 1;
                  }
               printf("No matching disk and image media types found. ");
               }
				else{
					switch(format){
                 	default:		// reading only
                  		// change extension to default
                        // force changing only, if multiple disks
                     file=changeExtension(file,Multiple,res);
						   if(file==NULL){	// new name cannot be generated
				      	   printf("Filename extension generation error.\n");
							   CBMrecalibrate(drive);
						      switchMotorOff();
				   	      return 1;
                  	   }
			            if(Multiple){				// generate new filename, if needed
                        do{
		                     fd=fopen(file,"rb");
                              // if reading disk, file mustn't exist
					            if(fd==NULL) break;	// new filename found
					            fclose(fd);
               	            // generate new filename
					            file=buildInxName(file);
                           if(file==NULL){	// new name cannot be generated
				                  printf("Filename generation error.\n");
							         CBMrecalibrate(drive);
						            switchMotorOff();
				   	            return 1;
                  	         }
            	            }while(1);
         	            }
	                  fd=fopen(file,"w+b");
	                  if(fd==NULL){
		                  printf("File \"%s\" open error.\n",file);
		                  return 1;
		                  }
							break;
                 	case 1:
							res=CBM1581;
							break;
                  case 2:
							res=CMDFD2M;
                  }
               break;
               }
            }
         else printf("CBM incompatible disk media type detected. ");
         res=FDUNKNW;
         }while(waitForDiskChng(drive, 1));
      if(res==FDUNKNW){	// user break;
				// user abort while waiting for a new disk
			CBMrecalibrate(drive);
			switchMotorOff();
			fclose(fd);
			return 2;
         }

      if(format && !writing) printf("Formatting ");
		else printf("Transferring ");
      switch(res){
#if (MESSAGES >= 0)
			default:
				printf("\n\nError after fixing the format routine, wrong drive type selected\n");
				return 10;
#endif
         case CBM1581:
            printf("CBM 1581");
            break;
         case CMDFD2M:
            printf("CMD FD2000");
         }
      if(format && !writing) printf(" disk in drive %c:\n",'A'+drive);
      else{
			printf(" disk from ");
         if(writing) printf("image \"%s\" to drive %c:\n",file,'A'+drive);
         else        printf("drive %c: to image \"%s\"\n",'A'+drive,file);
         }

      // presetting disk parameter table
      CBMpresetFDprm(drive,res);
#ifdef TimeEstimation
		TracksToDo=FDprm.Cylinders<<1;
#endif

		opStart=clock();
      if(BAMcp){
         if(res==CBM1581){
      	   printf("Reading BAM from source disk/image\n");
            if(writing){	// reading BAM from file
				   fseek(fd, 0x61800L, SEEK_SET);	// seek to Track 40;00
               	// BAM copy is only supported for CBM 1581 compatible disks
                  // buffer length needed: 0x400
				   if(fread(buffer,1,0x400,fd)!=0x400){
					   printf("File reading error.");
					   CBMrecalibrate(drive);
					   switchMotorOff();
					   fclose(fd);
					   return 1;
					   }
				   fseek(fd, 0x0L, SEEK_SET);		// seek to Track 01;00
         	   }
            else{
				   if(!MSecRead || !(res=CBMxferTrackMS(Read,drive,40,0,buffer,ileave,tries))){
               	res=CBMxferTrackSS(Read,drive,40,0,buffer,ileave,tries);
      	         }
			      if(res<=0){
         	      if(res<0) printf("User abort.\n");
                  else printf("Error while reading the BAM from disk.\n");
					   CBMrecalibrate(drive);
				      switchMotorOff();
				      fclose(fd);
				      return 2;
				      }
         	   }
            for(t=0;t<11;t++) BAMmark[t]=0;
         	   // Analyze buffer and get tracks to be copied
#ifdef TimeEstimation
			   TracksToDo=0;
            for(t=0,args=0x110;t<40;t++,args+=6) if(buffer[args]!=0x28){
					BAMmark[t>>3]|=1<<(t&0x07);
				   TracksToDo++;
               }
            for(args=0x210;t<80;t++,args+=6) if(buffer[args]!=0x28){
					BAMmark[t>>3]|=1<<(t&0x07);
				   TracksToDo++;
               }
			   TracksToDo<<=1;
#else
            for(t=0,args=0x110;t<40;t++,args+=6) if(buffer[args]!=0x28) BAMmark[t>>3]|=1<<(t&0x07);
            for(args=0x210;t<80;t++,args+=6) if(buffer[args]!=0x28) BAMmark[t>>3]|=1<<(t&0x07);
#endif

#if (MESSAGES >= 1)
            printf("Bitmask for BAM-Copy (80...01): 0x");
            for(t=9;t<9;t--) printf("%02X",BAMmark[t]);
            printf("\n");
#endif
   			}
			else{
            printf("BAM copying of this disk type is not supported yet.\n");
            for(t=0;t<11;t++) BAMmark[t]=0xFF;
#ifdef TimeEstimation
			   TracksToDo=162;
#endif
            }
      	}
#ifdef TimeEstimation
	   ETCbase=clock();
	   TracksDone=0;
#endif
		for(t=1;t<=FDprm.Cylinders;t++){
#ifdef DoubleTrackBuffer
			if(writing){
				if(fread(DblTrBuf,1,FDprm.Sectors*((2*128u) << FDprm.BpS),fd)!=
										  FDprm.Sectors*((2*128u) << FDprm.BpS)){
					printf("File reading error.");
					CBMrecalibrate(drive);
					switchMotorOff();
					fclose(fd);
					return 1;
					}
				}
			else{	// clear buffer, if there's a sector not readable,
					// 0x00 is written into the image file
				for(args=0;args<sizeof(DblTrBuf);args++) DblTrBuf[args]=0;
				}
			for(s=0,buffer=DblTrBuf;s<FDprm.Heads;s++,buffer+=FDprm.Sectors*(128u << FDprm.BpS)){
#else
			for(s=0;s<FDprm.Heads;s++){
			   if(writing){
				   if(fread(buffer,1,FDprm.Sectors*(128u << FDprm.BpS),fd)!=
				   						FDprm.Sectors*(128u << FDprm.BpS)){
					   printf("File reading error.");
					   CBMrecalibrate(drive);
					   switchMotorOff();
					   fclose(fd);
					   return 1;
					   }
				   }
			   else{	// clear buffer, if there's a sector not readable,
					   // 0x00 is written into the image file
				   for(args=0;args<sizeof(buffer);args++) buffer[args]=0;
				   }
#endif
			   if(!BAMcp || BAMmark[(t-1)>>3]&(1<<((t-1)&0x07))){
#if (MESSAGES >= 1)
        	      if(BAMcp) printf("BAM copying track %2d\n",t);
#endif
#ifdef TimeEstimation
               if(TracksDone!=0)
						printf("%2d %1d    ETC: %3lds\r",t,s,
							((long)(clock()-ETCbase)*(TracksToDo-TracksDone)*10 + 91)/(TracksDone*182)
							);
               TracksDone++;
#endif
					if(writing || format){
                  res=0;		// use normal format and/or write, if not Format'n'Writing or an error occurs
#ifdef FDCintel82078
                  if(FnW) res=CBMfwvTrack(verify,drive,t,s,buffer,tries);
#endif
                  if(!res && (!MSecRead || !(res=CBMxferTrackMS(verify|(writing?Write:0)|(format?Format:0),drive,t,s,buffer,ileave,tries)))){
							res=CBMxferTrackSS(verify|(writing?Write:0)|(format?Format:0),drive,t,s,buffer,ileave,tries);
                     }
					   }
               else{
					   if(!MSecRead || !(res=CBMxferTrackMS(Read,drive,t,s,buffer,ileave,tries))){
							res=CBMxferTrackSS(Read,drive,t,s,buffer,ileave,tries);
            		   }
               	}
					if(res==-2) break;
               if(res<0){
         	      printf("User abort.\n");
					   CBMrecalibrate(drive);
				      switchMotorOff();
						if(!writing && !format){
#ifdef DoubleTrackBuffer
							if(fwrite(DblTrBuf,1,FDprm.Sectors*((2*128u) << FDprm.BpS),fd)!=
														FDprm.Sectors*((2*128u) << FDprm.BpS)){
#else
						   if(fwrite(buffer,1,FDprm.Sectors*(128u << FDprm.BpS),fd)!=
						   						 FDprm.Sectors*(128u << FDprm.BpS)){
#endif
						      printf("File writing error.");
						      }
					      }
				      fclose(fd);
				      return 2;
         	      }
					else if(res==0) printf("Error in all attempts of handling side %1d, cylinder %2d of drive %d\n",s,t,drive);
   			   }
#ifndef  DoubleTrackBuffer
				if(!writing && !format){
				   if(fwrite(buffer,1,FDprm.Sectors*(128u << FDprm.BpS),fd)!=
				   						 FDprm.Sectors*(128u << FDprm.BpS)){
					   printf("File writing error.");
					   CBMrecalibrate(drive);
					   switchMotorOff();
					   fclose(fd);
					   return 1;
					   }
#else
			   }
			if(!writing && !format){
				if(fwrite(DblTrBuf,1,FDprm.Sectors*((2*128u) << FDprm.BpS),fd)!=
											FDprm.Sectors*((2*128u) << FDprm.BpS)){
					printf("File writing error.");
					CBMrecalibrate(drive);
					switchMotorOff();
					fclose(fd);
					return 1;
#endif
				   }
			   }
			if(res==-2) break;
         }
		if(format && !writing){
#ifdef DoubleTrackBuffer
			buffer=DblTrBuf;
#endif
         switch(format){
            case 1:
			      printf("Writing 1581 default BAM to disk in drive %c:\n", 'A'+drive);

			      C1581createBAM(buffer);

			      if(!MSecRead || !(res=CBMxferTrackMS(Write|verify,drive,40,0,buffer,ileave,tries))){
					   res=CBMxferTrackSS(Write|verify,drive,40,0,buffer,ileave,tries);
                  }
               if(res<0){
         	      printf("User abort.\n");
				      CBMrecalibrate(drive);
				      switchMotorOff();
				      return 2;
         	      }
               break;
            case 2:
			      printf("Writing FD2000 default system partition to disk in drive %c:\n", 'A'+drive);

					FD2000createSYS(buffer);

			      if(!MSecRead || !(res=CBMxferTrackMS(Write|verify,drive,81,0,buffer,ileave,tries))){
					   res=CBMxferTrackSS(Write|verify,drive,81,0,buffer,ileave,tries);
                  }
               if(res<0){
         	      printf("User abort.\n");
				      CBMrecalibrate(drive);
				      switchMotorOff();
				      return 2;
         	      }

			      printf("Writing FD2000 default BAM to disk in drive %c:\n", 'A'+drive);
					FD2000createBAM(buffer);

			      if(!MSecRead || !(res=CBMxferTrackMS(Write|verify,drive,1,0,buffer,ileave,tries))){
					   res=CBMxferTrackSS(Write|verify,drive,1,0,buffer,ileave,tries);
                  }
               if(res<0){
         	      printf("User abort.\n");
				      CBMrecalibrate(drive);
				      switchMotorOff();
				      return 2;
         	      }
               break;
            default:
	            printf("Writing a default BAM to this disk type is not supported yet.\n");
            }
			}
		fclose(fd);
		printf("Time needed for transferring the last disk: %ld ms\n\n",
				 (10000L*(long)(clock()-opStart))/182);
   	}while(Multiple && waitForDiskChng(drive, 1));
   CBMrecalibrate(drive);
	switchMotorOff();
	return 0;
	}
