/* DETECT.C chip's detection routines module.

   Copyright (c) 1995 Lin Ke-Fong, Ethereal Software.
*/

#include <conio.h>
#include <dos.h>
#include <string.h>
#include "mpvga.h"

/* %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */
int MPVGA_DetectAcer (void)

{
  int Old;

  Old = rdinx(GRC,0xFF);
  clrinx(GRC,0xFF,7);

  if (!testinx2(GRC,0x10,0x9F)) {
    clrinx(GRC,0xFF,7);
    if (testinx2(GRC,0x10,0x9F)) {
      MPVGA_ChipType = __Acer;
      switch (rdinx(GRC,0x0E) & 0x0C) {
        case 0 : MPVGA_VideoRAMSize = 256; break;
        case 1 : MPVGA_VideoRAMSize = 512; break;
        case 2 : MPVGA_VideoRAMSize = 1024; break;
        case 3 : MPVGA_VideoRAMSize = 2048; break;
        default : break;
      }
      wrinx(GRC,0xFF,Old);
      return Detected;
    }
  }
  wrinx(GRC,0xFF,Old);
  return NotDetected;
}


int MPVGA_DetectAhead (void)
/* note: possible detection using a memory checking scheme:
   0xC000:0x25 = "AHEAD" (5 bytes)
*/

{
  int Old;

  Old = rdinx(GRC,0x0F);
  wrinx(GRC,0x0F,0);

  if (!testinx2(GRC,0x0C,0xFB)) {
    wrinx(GRC,0x0F,0x20);
    if (testinx2(GRC,0x0C,0xFB)) {
      MPVGA_ChipType = __Ahead;
      switch (rdinx(GRC,0x0F) & 15) {
	case 0 : MPVGA_ChipVer = AH_A; break;
	case 1 : MPVGA_ChipVer = AH_B; break;
        default : break;
      }
      switch(rdinx(GRC,0x1F) & 3) {
        case 0 : MPVGA_VideoRAMSize = 256; break;
	case 1 : MPVGA_VideoRAMSize = 512; break;
	case 3 : MPVGA_VideoRAMSize = 1024; break;
        default : break;
      }
      wrinx(GRC,0x0F,Old);
      return Detected;
    }
  }
  wrinx(GRC,0x0F,Old);
  return NotDetected;
}


int MPVGA_DetectALG (void)

{
  int Old;

  Old = rdinx(CRTC,0x1A);
  clrinx(CRTC,0x1A,0x10);

  if (!testinx2(CRTC,0x19,0xCF)) {
    setinx(CRTC,0x1A,0x10);
    if (testinx2(CRTC,0x19,0xCF) && testinx2(CRTC,0x1A,0x3F)) {
      MPVGA_ChipType = __ALG;
      switch (rdinx(CRTC,0x1A) >> 6) {
        case 1 : MPVGA_ChipVer = ALG_2201; break;
        case 2 : {
	  if ((rdinx(CRTC,0x1B) & 4) > 0) MPVGA_ChipVer = ALG_2228;
	    else MPVGA_ChipVer = ALG_2301;
          break;
        }
        case 3 : MPVGA_ChipVer = ALG_2101; break;
        default : break;
      }
      switch (rdinx(CRTC,0x1E) & 3) {
	case 0 : MPVGA_VideoRAMSize = 256; break;
	case 1 : MPVGA_VideoRAMSize = 512; break;
	case 2 : MPVGA_VideoRAMSize = 1024; break;
	case 3 : MPVGA_VideoRAMSize = 2048; break;
        default : break;
      }
      wrinx(CRTC,0x1A,Old);
      return Detected;
    }
  }
  wrinx(CRTC,0x1A,Old);
  return NotDetected;
}


int MPVGA_DetectAlliance (void)

{
  char *ptr;

  if (rdinx(SEQ,0x11) == 0x41 && rdinx(SEQ,0x12) == 0x53) {
    MPVGA_ChipType = __Alliance;
    MPVGA_ChipVer = ALI_3210;

    wrinx(SEQ,0x10,0x12);
    setinx(SEQ,0x1C,8);
    modinx(SEQ,0x1B,7,1);
    ptr = (char *) ((0xA000 * 16) + 0xF0);
    MPVGA_VideoRAMSize = (*ptr)*64;
    clrinx(SEQ,0x1B,7);
    clrinx(SEQ,0x1C,8);
    return Detected;
  } else return NotDetected;
}


int MPVGA_DetectARK (void)

{
  int Old;

  Old = rdinx(SEQ,0x1D);
  wrinx(SEQ,0x1D,0);

  if (!(testinx(SEQ,0x11) && testinx(SEQ,0x12))) {
    wrinx(SEQ,0x1D,Old);
    if (testinx(SEQ,0x11) && testinx(SEQ,0x12)) {

      MPVGA_ChipType = __ARK;

      switch (rdinx(CRTC,0x50) & 0xF8) {
        case 0x88 : MPVGA_ChipVer = ARK_1000VL; break;
        case 0x90 : MPVGA_ChipVer = ARK_1000PV; break;
        case 0x98 : MPVGA_ChipVer = ARK_2000PV; break;
        default   : break;
      }
      if (MPVGA_ChipVer == ARK_2000PV)
        switch (rdinx(SEQ,0x10) >> 6) {
          case 0 : MPVGA_VideoRAMSize = 1024; break;
          case 1 : MPVGA_VideoRAMSize = 2048; break;
          case 2 : MPVGA_VideoRAMSize = 4096; break;
          case 3 : MPVGA_VideoRAMSize = 8192; break;
          default : break;
        }
      else if ((rdinx(SEQ,0x10) & 0x40) > 0) MPVGA_VideoRAMSize = 2048;
             else MPVGA_VideoRAMSize = 1024;
      wrinx(SEQ,0x1D,Old);
      return Detected;
    }
  }
  wrinx(SEQ,0x1D,Old);
  return NotDetected;
}

int MPVGA_DetectATI (void)

{
  const char ATI_BIOS_Str[9] = "761295520";
  char *Ptr;
  char *PtrW;
  int mall, mvga;
  long l;

  Ptr = (void *) 0xC0031;
  if (memcmp(Ptr,ATI_BIOS_Str,9) == 0) {
    MPVGA_ChipType = __ATI;
    PtrW = (void *) 0xC0040;
    switch (*PtrW) {
      case 0x3133 : {
        IOAdr = 0x1CE;
        Ptr = (void *) 0xC0043;
        switch (*Ptr) {
          case 0x31 : MPVGA_ChipVer = ATI_18800; break;
	  case 0x32 : MPVGA_ChipVer = ATI_18800_1; break;
	  case 0x33 : MPVGA_ChipVer = ATI_28800_2; break;
	  case 0x34 : MPVGA_ChipVer = ATI_28800_4; break;
          case 0x35 :
          case 0x36 : {
            if ((rdinx(IOAdr,0xAA) & 15) == 6) MPVGA_ChipVer = ATI_28800_6;
              else MPVGA_ChipVer = ATI_28800_5;
            break;
          }
          case 0x20 : {
            switch (inpw(0x6EEC)) {
              case 0x57 : MPVGA_ChipVer = ATI_M64_CX; break;
              case 0xD7 : MPVGA_ChipVer = ATI_M64_GX; break;
              default : break;
            }
            break;
          }
          case 0x61 :
          case 0x62 :
          case 0x63 : {
	    switch (inpw(0xFAEE) & 0x3FF) {
	      case 0x2F7 : MPVGA_ChipVer = ATI_GUP_6; break;
	      case 0x177 : MPVGA_ChipVer = ATI_GUP_LX; break;
	      case 0x017 : MPVGA_ChipVer = ATI_GUP_AX; break;
	      case 0     : MPVGA_ChipVer = ATI_GUP_3; break;
              default : break;
            }
            break;
          }
        }
        switch (MPVGA_ChipVer) {
	  case ATI_18800 :
          case ATI_18800_1 : {
            if ((rdinx(IOAdr,0xBB) & 0x20) != 0) MPVGA_VideoRAMSize = 512;
            break;
          }
	  case ATI_28800_2 : {
            if ((rdinx(IOAdr,0xB0) & 0x10) != 0) MPVGA_VideoRAMSize = 512;
            break;
          }
          case ATI_28800_4 :
          case ATI_28800_5 :
          case ATI_28800_6 : {
            switch (rdinx(IOAdr,0xB0) & 0x18) {
 	      case 0    : MPVGA_VideoRAMSize = 256; break;
	      case 0x10 : MPVGA_VideoRAMSize = 512; break;
 	      case 8    :
              case 0x18 : MPVGA_VideoRAMSize = 1024; break;
              default : break;
            }
            break;
          }
          case ATI_GUP_3  :
          case ATI_GUP_6  :
          case ATI_GUP_AX :
          case ATI_GUP_LX : {
            switch (inp(0x36EE) & 0x0C) {
	      case 0 : mall = 512; break;
	      case 4 : mall = 1024; break;
	      case 8 : mall = 2048; break;
	      case 12 : mall = 4096; break;
              default : break;
            }
            mvga = mall;
            if ((inp(0x42EE) & 0x10) > 0) {
              mvga = (inp(0x42EE) & 0x0F) * 256;
              if (mvga > mall) mvga = mall;
            }
            MPVGA_VideoRAMSize = mvga;
            break;
          }
          case ATI_M64_GX : {
            l = inp(0x52EC);
            l += inp(0x52EE) << 16;
            switch (l & 7) {
              case 0 : mall = 512; break;
              case 1 : mall = 1024; break;
              case 2 : mall = 2048; break;
              case 3 : mall = 4096; break;
              case 4 : mall = 6144; break;
              case 5 : mall = 8192; break;
              default : break;
            }
            mvga = mall;
            if ((l & 0x40000) > 0) {
              switch ((l >> 16) & 3) {
                case 0 : mvga = 0; break;
                case 1 : mvga = 256; break;
                case 2 : mvga = 512; break;
                case 3 : mvga = 1024; break;
                default : break;
              }
              if (mvga > mall) mvga = mall;
            }
            MPVGA_VideoRAMSize = mvga;
            if (MPVGA_VideoRAMSize > 1024) MPVGA_VideoRAMSize = 1024;
          }
          default : break;
        }
      }
      case 0x3233 : return NotDetected; /* ATI EGA is not what we want */
      default : break;
    }

/* not implemented yet
    if (MPVGA_ChipVer >= ATI_GUP_3) {
      if (MPVGA_ChipVer >= ATI_M64_GX) MPVGA_ChipType = __Mach64;
        else MPVGA_ChipType = __Mach32;
      MPVGA_VideoRAMSize = mall;
      if (mvga < mall) MPVGA_VideoRAMSize = mall-mvga;
    } */
    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectChips (void)

{
  union REGS Registers;
  int x, prt;

  Registers.w.ax = 0x5F00;
  int386(0x10,&Registers,&Registers);

  if (Registers.h.al == 0x5F) {
    MPVGA_ChipType = __Chips;
    x = Registers.h.bl;
    switch (x >> 4) {
      case 0 : MPVGA_ChipVer = CT_451; break;
      case 1 : MPVGA_ChipVer = CT_452; break;
      case 2 : MPVGA_ChipVer = CT_455; break;
      case 3 : MPVGA_ChipVer = CT_453; break;
      case 4 : MPVGA_ChipVer = CT_450; break;
      case 5 : MPVGA_ChipVer = CT_456; break;
      case 6 : MPVGA_ChipVer = CT_457; break;
      case 7 : MPVGA_ChipVer = CT_65520; break;
      case 8 : MPVGA_ChipVer = CT_65530; break;
      case 9 : MPVGA_ChipVer = CT_65510; break;
      case 10 : MPVGA_ChipVer = CT_64200; break;
      case 11 : {
        if (x >= 0xB8) MPVGA_ChipVer = CT_64310;
          else MPVGA_ChipVer = CT_64300;
        break;
      }
      case 12 : MPVGA_ChipVer = CT_65535; break;
      case 13 : {
        if (x >= 0xD8) MPVGA_ChipVer = CT_65545;
          else MPVGA_ChipVer = CT_65540;
        break;
      }
      default : break;
    }
    if (MPVGA_ChipVer < CT_450) {
      prt = 0x46E8;
      outp(prt,0x1E);
      x = inp(0x103);
      outp(0x103,x | 0x80);
      outp(prt,0x0E);
      if ((x & 0x40) == 0) IOAdr = 0x3D6;
        else IOAdr = 0x3B6;
    } else IOAdr = 0x3D6;

    switch (rdinx(IOAdr,4) & 3) {
      case 1 : MPVGA_VideoRAMSize = 512; break;
      case 2 :
      case 3 : {
        MPVGA_VideoRAMSize = 1024;
        if (MPVGA_ChipVer >= CT_64300 && (rdinx(IOAdr,0x0F) & 3) == 3)
          MPVGA_VideoRAMSize = 2048;
      }
      default : break;
    }
    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectCirrus54 (void)

{
  int Old, x;

  Old = rdinx(SEQ,6);
  wrinx(SEQ,6,0);

  if (rdinx(SEQ,6) == 0x0F) {
    wrinx(SEQ,6,0x12);
    if (rdinx(SEQ,6) == 0x12 && testinx2(SEQ,0x1E,0x3F)) {
      MPVGA_ChipType = __Cirrus54;
      switch (rdinx(SEQ,0x0A) & 0x18) {
	case 0  : MPVGA_VideoRAMSize = 256; break;
	case 8  : MPVGA_VideoRAMSize = 512; break;
        case 16 : MPVGA_VideoRAMSize = 1024; break;
        case 24 : MPVGA_VideoRAMSize = 2048; break;
        default : break;
      }
      x = rdinx(CRTC,0x27);
      if (testinx(GRC,9)) {
        switch (x) {
          case 0x18 : MPVGA_ChipVer = CL_AVGA2; break;
          case 0x88 : MPVGA_ChipVer = CL_GD5402; break;
          case 0x89 : MPVGA_ChipVer = CL_GD5402r1; break;
          case 0x8A : MPVGA_ChipVer = CL_GD5420; break;
          case 0x8B : MPVGA_ChipVer = CL_GD5420r1; break;
          case 0x8C :
          case 0x8D :
          case 0x8E :
          case 0x8F : MPVGA_ChipVer = CL_GD5422; break;
          case 0x90 :
          case 0x91 :
          case 0x92 :
          case 0x93 : MPVGA_ChipVer = CL_GD5426; break;
          case 0x94 :
          case 0x95 :
          case 0x96 :
          case 0x97 : MPVGA_ChipVer = CL_GD5424; break;
          case 0x98 :
          case 0x99 :
          case 0x9A :
          case 0x9B : MPVGA_ChipVer = CL_GD5428; break;
          case 0x9C :
          case 0x9D :
          case 0x9E :
          case 0x9F : MPVGA_ChipVer = CL_GD5429; break;
          case 0xA0 :
          case 0xA1 :
          case 0xA2 :
          case 0xA3 : MPVGA_ChipVer = CL_GD5430; break;
          case 0xA8 :
          case 0xA9 :
          case 0xAA :
          case 0xAB : MPVGA_ChipVer = CL_GD5434; break;
          case 0x2C :
          case 0x2D :
          case 0x2E :
          case 0x2F : MPVGA_ChipVer = CL_GD7542; break;
          case 0x30 :
          case 0x31 :
          case 0x32 :
          case 0x33 : MPVGA_ChipVer = CL_GD7543; break;
          case 0x34 :
          case 0x35 :
          case 0x36 :
          case 0x37 : MPVGA_ChipVer = CL_GD7541; break;
          default : break;
	}
        if (MPVGA_ChipVer >= CL_GD7541)
          switch (rdinx(SEQ,0x0A) & 15) {
            case 0 : MPVGA_VideoRAMSize = 256; break;
            case 1 : MPVGA_VideoRAMSize = 512; break;
            case 2 : MPVGA_VideoRAMSize = 1024; break;
            case 3 : MPVGA_VideoRAMSize = 2048; break;
            case 4 : MPVGA_VideoRAMSize = 4096; break;
            default : break;
          }

        if (MPVGA_ChipVer >= CL_GD5430)
          switch (rdinx(SEQ,0x15) & 15) {
            case 0 : MPVGA_VideoRAMSize = 256; break;
            case 1 : MPVGA_VideoRAMSize = 512; break;
            case 2 : MPVGA_VideoRAMSize = 1024; break;
            case 3 : MPVGA_VideoRAMSize = 2048; break;
            case 4 : MPVGA_VideoRAMSize = 4096; break;
            default : break;
          }
      } else if (testinx(SEQ,0x19)) {
	switch (x >> 6) {
	  case 0 : MPVGA_ChipVer = CL_GD6205; break;
	  case 1 : MPVGA_ChipVer = CL_GD6235; break;
	  case 2 : MPVGA_ChipVer = CL_GD6215; break;
	  case 3 : MPVGA_ChipVer = CL_GD6225; break;
          default : break;
        }
        MPVGA_VideoRAMSize = 512;
      } else {
	MPVGA_ChipVer = CL_AVGA2;
	switch  (rdinx(SEQ,0x0A) & 3) {
	  case 0 : MPVGA_VideoRAMSize = 256; break;
	  case 1 : MPVGA_VideoRAMSize = 512; break;
	  case 2 : MPVGA_VideoRAMSize = 1024; break;
          default : break;
        }
      }
      wrinx(SEQ,6,Old);
      return Detected;
    }
  }
  wrinx(SEQ,6,Old);
  return NotDetected;
}


int MPVGA_DetectCirrus64 (void)

{
  int Old;

  Old = rdinx(GRC,0x0A);
  wrinx(GRC,0x0A,0xCE);

  if (rdinx(GRC,0x0A) == 0) {
    wrinx(GRC,0x0A,0xEC);
    if (rdinx(GRC,0x0A) == 1) {
      MPVGA_ChipType = __Cirrus64;

      switch (rdinx(GRC,0xAA) >> 4) {
	case 4 : MPVGA_ChipVer = CL_GD6440; break;
	case 5 : MPVGA_ChipVer = CL_GD6412; break;
	case 6 : MPVGA_ChipVer = CL_GD5410; break;
	case 7 : {
          if (testinx2(GRC,0x87,0x90)) MPVGA_ChipVer = CL_GD6420B;
            else MPVGA_ChipVer = CL_GD6420A;
          break;
        }
	case 8 : MPVGA_ChipVer = CL_GD6410; break;
        default : break;
      }
      switch (rdinx(GRC,0xBB) >> 6) {
	case 0 : MPVGA_VideoRAMSize = 256; break;
	case 1 : MPVGA_VideoRAMSize = 512; break;
	case 2 : MPVGA_VideoRAMSize = 768; break;
	case 3 : MPVGA_VideoRAMSize = 1024; break;
        default : break;
      }
      wrinx(GRC,0x0A,Old);
      return Detected;
    }
  }
  wrinx(GRC,0x0A,Old);
  return NotDetected;
}


int MPVGA_DetectCompaq (void)

{
  int Old;
  union REGS Registers;

  Old = rdinx(GRC,0x0F);
  wrinx(GRC,0x0F,0);

  if (!testinx(GRC,0x45)) {
    wrinx(GRC,0x0F,5);
    if (testinx(GRC,0x45)) {
      MPVGA_ChipType = __Compaq;
      switch (rdinx(GRC,0x0C) >> 3) {
	case 3    : MPVGA_ChipVer = CPQ_IVGS; break;
	case 5 	  : MPVGA_ChipVer = CPQ_AVGA; break;
	case 6 	  : MPVGA_ChipVer = CPQ_QV1024; break;
        case 0x0E : {
	  if ((rdinx(GRC,0x56) & 4) != 0) MPVGA_ChipVer = CPQ_QV1280;
	    else MPVGA_ChipVer = CPQ_QV1024;
          break;
        }
        case 0x10 : MPVGA_ChipVer = CPQ_AVPORT; break;
	default : break;
      }
      if ((rdinx(GRC,0x0C) & 0xB8) == 0x30) {
	wrinx(GRC,0x0F,5);
	switch (rdinx(GRC,0x54)) {
	  case 0 : MPVGA_VideoRAMSize = 1024; break;
	  case 2 : MPVGA_VideoRAMSize = 512; break;
	  case 4 : MPVGA_VideoRAMSize = 1024; break;
	  case 8 : MPVGA_VideoRAMSize = 2048; break;
          default : break;
	}
      } else {
        Registers.w.ax = 0xBF03;
        Registers.w.bx = 0x0000;
        Registers.w.cx = 0x0000;
        int386(0x10,&Registers,&Registers);
	if ((Registers.h.ch & 64) == 0) MPVGA_VideoRAMSize = 512;
      }
      wrinx(GRC,0x0F,Old);
      return Detected;
    }
  }
  wrinx(GRC,0x0F,Old);
  return NotDetected;
}

int MPVGA_DetectHMC (void)

{
  if (testinx2(SEQ,0xE7,0x7F) && testinx2(SEQ,0xEE,0xF1)) {
    MPVGA_ChipType = __HMC;
    if ((rdinx(SEQ,0xE7) & 0x10) > 0) MPVGA_VideoRAMSize = 512;
      else MPVGA_VideoRAMSize = 256;
    if (testinx(SEQ,0xE7) && testinx(SEQ,0xEE)) MPVGA_ChipVer = HMC_304;
      else MPVGA_ChipVer = HMC_314;
    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectMatrox (void)

{
  union REGS Registers;

  Registers.w.ax = 0xBDFF;
  int386(0x10,&Registers,&Registers);
  if (Registers.w.ax == 0xBD) { 
    MPVGA_ChipType = __Matrox;
/* need more info */

    return Detected;
  }
  return NotDetected;
}

int MPVGA_DetectMXIC (void)

{
  int Old;

  Old = rdinx(SEQ,0xA7);
  wrinx(SEQ,0xA7,0);

  if (!testinx(SEQ,0xC5)) {
    wrinx(SEQ,0xA7,0x87);
    if (testinx(SEQ,0xC5)) {
      MPVGA_ChipType = __MXIC;
      if ((rdinx(SEQ,0x26) & 1) == 0) MPVGA_ChipVer = MX_86010;
        else MPVGA_ChipVer = MX_86000;
      switch ((rdinx(SEQ,0xC2) >> 2) & 3) {
	case 0 : MPVGA_VideoRAMSize = 256; break;
	case 1 : MPVGA_VideoRAMSize = 512; break;
	case 2 : MPVGA_VideoRAMSize = 1024; break;
        default : break;
      }
      wrinx(SEQ,0xA7,Old);
      return Detected;
    }
  }
  wrinx(SEQ,0xA7,Old);
  return NotDetected;
}


int MPVGA_DetectNCR (void)

{
  int x;

  if (testinx2(SEQ,5,5)) {
    wrinx(SEQ,5,0);
    if (!testinx(SEQ,0x10)) {
      wrinx(SEQ,5,1);
      if (testinx(SEQ,0x10)) {
	MPVGA_ChipType = __NCR;
        x = rdinx(SEQ,8);
	switch (x >> 4) {
	  case 0 : MPVGA_ChipVer = NCR_77c22; break;
	  case 1 : MPVGA_ChipVer = NCR_77c21; break;
	  case 2 : {
	    if ((x & 15) < 8) MPVGA_ChipVer = NCR_77c22e;
              else MPVGA_ChipVer = NCR_77c22ep;
            break;
          }
          case 3 : MPVGA_ChipVer = NCR_77c32blt; break;
	  default : break;
	}
/* no memory size detection yet */
	return Detected;
      }
    }
  }
  return NotDetected;
}


int MPVGA_DetectOak (void)

{
  int Old;

  if (testinx(0x3DE,9) || testinx2(0x3DE,0x0D,0x38)) {
    MPVGA_ChipType = __Oak;
    if (testinx2(0x3DE,0x23,0x1F)) {
      switch (rdinx(0x3DE,2) & 6) {
	case 0 : MPVGA_VideoRAMSize = 256; break;
	case 2 : MPVGA_VideoRAMSize = 512; break;
	case 4 : MPVGA_VideoRAMSize = 1024; break;
	case 6 : MPVGA_VideoRAMSize = 2048; break;
        default : break;
      }
      if ((rdinx(0x3DE,0) & 2) == 0) MPVGA_ChipVer = OAK_087;
 	else MPVGA_ChipVer = OAK_083;
    } else {
      switch (inp(0x3DE) >> 5) {
	case 0 : MPVGA_ChipVer = OAK_037; break;
	case 2 : MPVGA_ChipVer = OAK_067; break;
	case 5 : MPVGA_ChipVer = OAK_077; break;
	case 7 : MPVGA_ChipVer = OAK_057; break;
        default : break;
      }
      switch (rdinx(0x3DE,0x0D) >> 6) {
	case 2 : MPVGA_VideoRAMSize = 512; break;
        case 1 :
	case 3 : MPVGA_VideoRAMSize = 1024; break;
        default : break;
      }
    }
    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectPrimus (void)

{
  if (testinx2(GRC,0x3D,0x3F) && tstrg(0x3D6,0x1F) && tstrg(0x3D7,0x1F)) {
    MPVGA_ChipVer = PR_2000;
    MPVGA_ChipType = __Primus;
    /* unable to do memory size determination */
    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectRealtek (void)

{
  int x;
  union REGS Registers;

  if (testinx2(CRTC,0x1F,0x3F) && tstrg(0x3D6,0x0F) && tstrg(0x3D7,0x0F)) {
    MPVGA_ChipType = __Realtek;
    x = rdinx(CRTC,0x1A) >> 6;
    switch (x) {
      case 0 : MPVGA_ChipVer = RT_3103; break;
      case 1 : MPVGA_ChipVer = RT_3105; break;
      case 2 : MPVGA_ChipVer = RT_3106; break;
      default : break;
    }
    switch (rdinx(CRTC,0x1E) & 15) {
      case 0 : MPVGA_VideoRAMSize = 256; break;
      case 1 : MPVGA_VideoRAMSize = 512; break;
      case 2 : {
        if (x == 0) MPVGA_VideoRAMSize = 768;
	  else MPVGA_VideoRAMSize = 1024;
        break;
      }
      case 3 : {
        if (x == 0) MPVGA_VideoRAMSize = 1024;
	  else MPVGA_VideoRAMSize = 2048;
        break;
      }
      default : break;
    }
    Registers.w.ax = 0x5F02;
    int386(0x10,&Registers,&Registers);

    if (Registers.h.ah == 0)
      MPVGA_BiosVRAMSize = 256 + (Registers.h.al * 256);

    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectS3 (void)

{
  int x;

  wrinx(CRTC,0x38,0);
  if (!testinx2(CRTC,0x35,0x0F)) {
    wrinx(CRTC,0x38,0x48);
    if (testinx2(CRTC,0x35,0x0F)) {
      MPVGA_ChipType = __S3;
      x = rdinx(CRTC,0x30);
      switch (x) {
	case 0x81 : MPVGA_ChipVer = S3_911; break;
	case 0x82 : MPVGA_ChipVer = S3_924; break;
	case 0x90 : MPVGA_ChipVer = S3_928; break;
	case 0x91 : MPVGA_ChipVer = S3_928C; break;
	case 0x94 : MPVGA_ChipVer = S3_928D; break;
        case 0x95 : MPVGA_ChipVer = S3_928E; break;
	case 0xA0 : {
	  if ((rdinx(CRTC,0x36) & 2) > 0) MPVGA_ChipVer = S3_801AB;
	    else MPVGA_ChipVer = S3_805AB;
          break;
        }
        case 0xA2:
        case 0xA3:
	case 0xA4: {
	  if ((rdinx(CRTC,0x36) & 2) > 0) MPVGA_ChipVer = S3_801C;
	    else MPVGA_ChipVer = S3_805C;
	  break;
        }
        case 0xA5 :
	case 0xA7 : {
	  if ((rdinx(CRTC,0x36) & 2) > 0) MPVGA_ChipVer = S3_801D;
 	    else MPVGA_ChipVer = S3_805D;
          break;
        }
        case 0xA6 : {
	  if ((rdinx(CRTC,0x36) & 2) > 0) MPVGA_ChipVer = S3_801P;
	    else MPVGA_ChipVer = S3_805P;
          break;
        }
        case 0xA8 : {
	  if ((rdinx(CRTC,0x036) & 2) > 0) MPVGA_ChipVer = S3_801I;
	    else MPVGA_ChipVer = S3_805I;
          break;
        }
	case 0xB0 : MPVGA_ChipVer = S3_928PCI; break;
	case 0xC0 : MPVGA_ChipVer = S3_864; break;
	case 0xC1 : MPVGA_ChipVer = S3_864P; break;
	case 0xD0 : MPVGA_ChipVer = S3_964; break;

    	case 0xE0 :
	case 0xE1 : {
	  switch (rdinx(CRTC,0x2E)) {
            case 0x10 : MPVGA_ChipVer = S3_732; break;
            case 0x11 : MPVGA_ChipVer = S3_764; break;
            case 0x80 : MPVGA_ChipVer = S3_866; break;
            case 0x90 : MPVGA_ChipVer = S3_868; break;
            case 0xB0 :
	    case 0xF0 :  MPVGA_ChipVer = S3_968; break;
            default   : break;
	  }
          break;
        }
	default : break;
      }

      MPVGA_VideoRAMSize = 512;
      if ((rdinx(CRTC,0x36) & 0x20) == 0)
        if (x < 0x90) MPVGA_VideoRAMSize = 1024;
          else switch (rdinx(CRTC,0x036) >> 6) {
            case 0 : MPVGA_VideoRAMSize = 4096; break;
            case 1 : MPVGA_VideoRAMSize = 3072; break;
            case 2 : MPVGA_VideoRAMSize = 2048; break;
            case 3 : MPVGA_VideoRAMSize = 1024; break;
            default : break;
          }
      return Detected;
    }
  }
  return NotDetected;
}

int MPVGA_DetectSierra (void)

{
  int Old, i;

  Old = rdinx(SEQ,0x11);
  setinx(SEQ,0x11,0x20);
  if (!testinx(SEQ,0x15))
    i = rdinx(SEQ,0x11);
    outp(SEQ+1,i);
    outp(SEQ+1,i);
    outp(SEQ+1,i & 0xDF);
    if (testinx(SEQ,0x15)) {
      MPVGA_ChipType = __Sierra;
      setinx(SEQ,0x11,0x20);
      if ((rdinx(SEQ,7) >> 5) == 4) MPVGA_ChipVer = SC_15064;
/* cannot detect memory size */
      wrinx(SEQ,0x11,Old);
      return Detected;
    }
  wrinx(SEQ,0x11,Old);
  return NotDetected;
}


int MPVGA_DetectSiS (void)

{
  int Old;

  Old = rdinx(SEQ,5);
  wrinx(SEQ,5,0);

  if (rdinx(SEQ,5) == 0x21) {
    wrinx(SEQ,5,0x86);
    if (rdinx(SEQ,5) == 0xA1) {
      MPVGA_ChipType = __SiS;
      MPVGA_ChipVer = SIS_201;
      switch (rdinx(SEQ,0x0F) & 3) {
        case 0 : MPVGA_VideoRAMSize = 1024;
        case 1 : MPVGA_VideoRAMSize = 2048;
        case 2 : MPVGA_VideoRAMSize = 4096;
        default : break;
      }
      wrinx(SEQ,5,Old);
      return Detected;
    }
  }
  wrinx(SEQ,5,Old);
  return NotDetected;
}


int MPVGA_DetectTrident (void)

{
  int Old, Val, x;
  union REGS Registers;

  wrinx(SEQ,0x0B,0);
  x = inp(SEQ+1);
  Old = rdinx(SEQ,0x0E);
  outp(SEQ+1,Old ^ 0x55);
  Val = inp(SEQ+1);
  outp(SEQ+1,Old);

  if (((Val ^ Old) & 15) == 7) {
    MPVGA_ChipType = __Trident;
    outp(0x3C5,Old ^ 2);
    switch (x) {
      case 1    : MPVGA_ChipVer = TR_8800BR; break; /* unsupported */
      case 2    : MPVGA_ChipVer = TR_8800CS; break;
      case 3    : MPVGA_ChipVer = TR_8900B; break;
      case 4    :
      case 0x13 : MPVGA_ChipVer = TR_8900C; break;
      case 0x23 : MPVGA_ChipVer = TR_9000; break;
      case 0x33 : {
        if ((rdinx(CRTC,0x28) & 0x80) > 0) MPVGA_ChipVer = TR_8900CL;
          else MPVGA_ChipVer = TR_9000C;
        break;
      }
      case 0x43 : MPVGA_ChipVer = TR_9000i; break;
      case 0x53 : MPVGA_ChipVer = TR_9200CXr; break;
      case 0x63 : MPVGA_ChipVer = TR_LCD9100B; break;
      case 0x73 : MPVGA_ChipVer = TR_GUI9420; break;
      case 0x83 : MPVGA_ChipVer = TR_LX8200; break;
      case 0x93 : MPVGA_ChipVer = TR_9400CXi; break;
      case 0xA3 : MPVGA_ChipVer = TR_LCD9320; break;
      case 0xC3 : MPVGA_ChipVer = TR_GUI9420; break;
      case 0xD3 : MPVGA_ChipVer = TR_GUI9660; break;
      case 0xE3 : MPVGA_ChipVer = TR_GUI9440; break;
      case 0xF3 : MPVGA_ChipVer = TR_GUI9430; break;
      default : break;
    }
    switch (rdinx(CRTC,0x1F) & 3) {
      case 0 : MPVGA_VideoRAMSize = 256; break;
      case 1 : MPVGA_VideoRAMSize = 512; break;
      case 2 : MPVGA_VideoRAMSize = 768; break;
      case 3 : {
        if (MPVGA_ChipVer >= TR_8900CL && (rdinx(CRTC,0x1F) & 4) > 0)
          MPVGA_VideoRAMSize = 2048; else MPVGA_VideoRAMSize = 1024;
        break;
      }
      default : break;
    }
    Registers.h.ah = 0x12;
    Registers.h.bl = 0x12;
    int386(0x10,&Registers,&Registers);
    if (Registers.h.al == 0x12) MPVGA_BiosVRAMSize = Registers.h.ah * 256;

    return Detected;
  } else return NotDetected;
}


int MPVGA_DetectTseng (void)

{
  int x, vs;

  outp(0x3BF,3);
  outp(CRTC+4,0xA0);
  if (tstrg(0x3CD,0x3F)) {
    MPVGA_ChipType = __Tseng;
    if (testinx2(CRTC,0x33,0x0F))
      if (tstrg(0x3CB,0x33)) {
	switch (rdinx(0x217A,0xEC) >> 4) {
	  case 0 : MPVGA_ChipVer = ET_4W32; break;
          case 1 : MPVGA_ChipVer = ET_4W32i_a; break;
          case 2 : MPVGA_ChipVer = ET_4W32p_a; break;
          case 3 : MPVGA_ChipVer = ET_4W32i_b; break;
          case 5 : MPVGA_ChipVer = ET_4W32p_b; break;
          case 6 : MPVGA_ChipVer = ET_4W32p_d; break;
          case 7 : MPVGA_ChipVer = ET_4W32p_c; break;
          case 11 : MPVGA_ChipVer = ET_4W32i_c; break;
	  default : break;
	}

	switch (rdinx(CRTC,0x37) & 9) {
          case 0 : MPVGA_VideoRAMSize = 2048; break;
	  case 1 : MPVGA_VideoRAMSize = 4096; break;
	  case 8 : MPVGA_VideoRAMSize = 512; break;
	  case 9 : MPVGA_VideoRAMSize = 1024; break;
          default : break;
	}
        if (MPVGA_ChipVer != ET_4W32 && (rdinx(CRTC,0x32) & 0x80) > 0)
          MPVGA_VideoRAMSize *= 2;
      }
    else {
      MPVGA_ChipVer = ET_4000;
      switch (rdinx(CRTC,0x37) & 0x0B) {
	 case 3  :
	 case 9  : MPVGA_VideoRAMSize = 256; break;
	 case 10 : MPVGA_VideoRAMSize = 512; break;
	 case 11 : MPVGA_VideoRAMSize = 1024; break;
         default : break;
      }
    } else {
      MPVGA_ChipVer = ET_3000;
      /* no mem detection yet */
    }
    return Detected ;
  }
  return NotDetected;
}


int MPVGA_DetectUMC (void)

{
  int Old;

  Old = inp(0x3BF);
  outp(0x3BF,3);

  if (!testinx(SEQ,6)) {
    outp(0x3BF,0xAC);
    if (testinx(SEQ,6)) {
      MPVGA_ChipType = __UMC;
      switch (rdinx(SEQ,7) >> 6) {
        case 1 : MPVGA_VideoRAMSize = 512; break;
        case 2 :
	case 3 : MPVGA_VideoRAMSize = 1024; break;
        default : break;
      }
      if (testinx2(CRTC,0x35,0x0F)) {
        MPVGA_ChipVer = UMC_418;
        if ((rdinx(GRC,0xB) & 0x7F) == 0x2A) MPVGA_VideoRAMSize = 1024;
      } else MPVGA_ChipVer = UMC_408;
      outp(0x3BF,Old);
      return Detected;
    }
  }
  outp(0x3BF,Old);
  return NotDetected;
}


int MPVGA_DetectVideo7 (void)

{
  int Old,x;
  union REGS Registers;

  Registers.w.ax = 0x6F00;
  int386(0x10,&Registers,&Registers);

  if (Registers.w.bx == 0x5637) {
    MPVGA_ChipType = __Video7;
    Registers.w.ax = 0x06F07;
    int386(0x10,&Registers,&Registers);

    wrinx(SEQ,6,0xEA);
    x= (rdinx(SEQ,0x8F) << 8)+rdinx(SEQ,0x8E);
    if (x >= 0x8000 && x <= 0xFFFF)
      MPVGA_ChipVer = V7_VEGA;
    else if (x >= 0x7000 && x <= 0x70FF)
      MPVGA_ChipVer = V7_208_13;
    else if (x >= 0x7140 && x <= 0x714F)
      MPVGA_ChipVer = V7_208A;
    else switch (x) {
      case 0x7151 : MPVGA_ChipVer = V7_208B; break;
      case 0x7152 : MPVGA_ChipVer = V7_208CD; break;
      case 0x7760 : MPVGA_ChipVer = V7_216BC; break;
      case 0x7763 : MPVGA_ChipVer = V7_216D; break;
      case 0x7764 : MPVGA_ChipVer = V7_216E; break;
      case 0x7765 : MPVGA_ChipVer = V7_216F; break;
      default : break;
    }
    switch (Registers.h.ah & 127) {
      case 2 : MPVGA_VideoRAMSize = 512; break;
      case 4 : MPVGA_VideoRAMSize = 1024; break;
      default : break;
    }
    return Detected;
  }
  return NotDetected;
}


int MPVGA_DetectWD (void)

{
  int Old,Old2;

  Old = rdinx(GRC,0x0F);
  setinx(GRC,0x0F,0x17);

  if (!testinx2(GRC,9,0x7F)) {
    wrinx(GRC,0xF,5);
    if (testinx2(GRC,9,0x7F)) {
      MPVGA_ChipType = __WD;
      Old2 = rdinx(CRTC,0x29);
      modinx(CRTC,0x29,0x8F,0x85);
      if (!testinx(CRTC,0x2B)) MPVGA_ChipVer = WD_PVGA1A; else {
	wrinx(SEQ,6,0x48);
	if (!testinx2(SEQ,7,0xF0))
	  MPVGA_ChipVer = WD_90c00;
        else if (!testinx(SEQ,0x10)) {
          if (testinx2(CRTC,0x31,0x68)) MPVGA_ChipVer = WD_90c22;
            else if (testinx2(CRTC,0x31,0x90)) MPVGA_ChipVer = WD_90c20A;
              else MPVGA_ChipVer = WD_90c20;
	  wrinx(CRTC,0x34,0xA6);
	  if ((rdinx(CRTC,0x32) & 0x20) != 0) wrinx(CRTC,0x34,0);
	} else {
	  if (testinx2(SEQ,0x14,0x0F)) {
            wrinx(CRTC,0x34,0);
            wrinx(CRTC,0x35,0);
	    switch ((rdinx(CRTC,0x36) << 8) + rdinx(CRTC,0x37)) {
	      case 0x3234 : MPVGA_ChipVer = WD_90c24; break;
	      case 0x3236 : MPVGA_ChipVer = WD_90c26; break;
	      case 0x3330 : MPVGA_ChipVer = WD_90c30; break;
	      case 0x3331 : MPVGA_ChipVer = WD_90c31; break;
	      case 0x3333 : MPVGA_ChipVer = WD_90c33; break;
              default : break;
	    }
	  } else if (!testinx2(SEQ,0x10,4)) MPVGA_ChipVer = WD_90c10;
  		   else MPVGA_ChipVer = WD_90c11;
	}
      }
      wrinx(GRC,0x0F,5);
      switch (rdinx(GRC,0x0B) >> 6) {
        case 2 : MPVGA_VideoRAMSize = 512; break;
        case 3 : MPVGA_VideoRAMSize = 1024; break;
        default : break;
      }

      if (MPVGA_ChipVer >= WD_90c33 && (rdinx(CRTC,0x3E) & 0x80) > 0)
        MPVGA_VideoRAMSize = 2048;
      wrinx(CRTC,0x29,Old2);
      wrinx(GRC,0x0F,Old);
      return Detected;
    }
  }
  wrinx(GRC,0x0F,Old);
  return NotDetected;
}

int MPVGA_DetectChip (void)

{
  MPVGA_ChipType = __Unknown;
  MPVGA_ChipVer = UnknownChipVer;
  MPVGA_VideoRAMSize = -1; /* -1 means unknown */
  MPVGA_BiosVRAMSize = -1;

  if ((inp(0x3CC) & 1) == 0) CRTC = 0x3B4; else CRTC = 0x3D4;

  if (MPVGA_DetectAcer() == Detected)
    return Detected;
  else if (MPVGA_DetectAhead() == Detected)
    return Detected;
  else if (MPVGA_DetectAlliance() == Detected)
    return Detected;
  else if (MPVGA_DetectALG() == Detected)
    return Detected;
  else if (MPVGA_DetectARK() == Detected)
    return Detected;
  else if (MPVGA_DetectATI() == Detected)
    return Detected;
  else if (MPVGA_DetectChips() == Detected)
    return Detected;
  else if (MPVGA_DetectCirrus54() == Detected)
    return Detected;
  else if (MPVGA_DetectCirrus64() == Detected)
    return Detected;
  else if (MPVGA_DetectCompaq() == Detected)
    return Detected;
  else if (MPVGA_DetectHMC() == Detected)
    return Detected;
  else if (MPVGA_DetectMatrox() == Detected)
    return Detected;
  else if (MPVGA_DetectMXIC() == Detected)
    return Detected;
  else if (MPVGA_DetectNCR() == Detected)
    return Detected;
  else if (MPVGA_DetectOak() == Detected)
    return Detected;
  else if (MPVGA_DetectPrimus() == Detected)
    return Detected;
  else if (MPVGA_DetectRealtek() == Detected)
    return Detected;
  else if (MPVGA_DetectS3() == Detected)
    return Detected;
  else if (MPVGA_DetectSierra() == Detected)
    return Detected;
  else if (MPVGA_DetectSiS() == Detected)
    return Detected;
  else if (MPVGA_DetectTrident() == Detected)
    return Detected;
  else if (MPVGA_DetectTseng() == Detected)
    return Detected;
  else if (MPVGA_DetectUMC() == Detected)
    return Detected;
  else if (MPVGA_DetectVideo7() == Detected)
    return Detected;
  else if (MPVGA_DetectWD() == Detected)
    return Detected;
  else return NotDetected;
}