/*
   ** Copyright (C) 1995  Jonathan Paul Griffiths.  All rights reserved.
   **
   ** You may do anything with this code you wish EXCEPT sell it. You may sell
   ** any software you create using this code,  but you MUST NOT charge for
   ** the code itself.  See the file "jlib.doc" for more details.
 */

#include <stdlib.h>		/* malloc */
#include <stdio.h>
#include <string.h>		/* memset */
#include <jlib.h>
#include "../io/jio.h"

#ifndef NULL
#define NULL 0L;
#endif

/* this routine is private!! */
JINLINE int JLIB_do_intersect (USHORT x1, USHORT y1, UBYTE *r1, USHORT x2, USHORT y2, UBYTE *r2);


/*+----------------------------------------------------------------------+ */
/*|sprite_init - intitialises the sprite system.                         | */
/*+----------------------------------------------------------------------+ */

sprite_system *sprite_init (USHORT max_sprites, USHORT max_frames)
{
   sprite_system *tssp;
   int counter;

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_init()\n");

   /* create space for sprite system record */
   if ((tssp = (sprite_system *) malloc (sizeof (sprite_system))) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at sprite_init line %d\n", __LINE__);
	JLIB_PRINT_MESSAGE_STRING;

	return (sprite_system *) NULL;
     }

   /* fill in max sprite info */
   tssp->no_sprites = max_sprites;

   JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Number of sprites supported is %d\n", tssp->no_sprites);
   JLIB_PRINT_MESSAGE_STRING;

   /* create 'is active' array */
   if ((tssp->active_sprites =
	(UBYTE *) malloc (SPR_MAX_SPRITES (tssp) * sizeof (UBYTE))) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at sprite_init line %d\n", __LINE__);
	JLIB_PRINT_MESSAGE_STRING;

	return (sprite_system *) NULL;
     }

   /* turn 'em all off */
   memset (tssp->active_sprites, 0, (int) SPR_MAX_SPRITES (tssp));


   /* create 'sprite' array */
   if ((tssp->sprites =
	(sprite_record **) malloc (SPR_MAX_SPRITES (tssp) * sizeof (sprite_record *))) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at sprite_init line %d\n", __LINE__);
	JLIB_PRINT_MESSAGE_STRING;

	return (sprite_system *) NULL;
     }

   tssp->no_frames = max_frames;
   tssp->number_loaded = 0;
   
   JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Number of frames supported is %d\n", tssp->no_frames);
   JLIB_PRINT_MESSAGE_STRING;

   /* create 'sprite data' array */
   if ((tssp->sprite_data =
	(sprite_data_rec **) malloc (SPR_MAX_FRAMES (tssp) * sizeof (sprite_data_rec *))) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at sprite_init line %d\n", __LINE__);
	JLIB_PRINT_MESSAGE_STRING;

	return (sprite_system *) NULL;
     }

   /* NULL them all */
   for (counter = 0; counter < SPR_MAX_FRAMES (tssp); counter++)
     {
	tssp->sprite_data[counter] = NULL;
     }

   /* initialise the sprites */
   for (counter = 0; counter < SPR_MAX_SPRITES (tssp); counter++)
     {

	if ((tssp->sprites[counter] =
	     (sprite_record *) malloc (sizeof (sprite_record))) == NULL)
	  {
	     JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at sprite_init line %d\n", __LINE__);
	     JLIB_PRINT_MESSAGE_STRING;

	     return (sprite_system *) NULL;
	  }

	tssp->sprites[counter]->x = 0;
	tssp->sprites[counter]->y = 0;

	tssp->sprites[counter]->speed = 0;
	tssp->sprites[counter]->speed_counter = 0;
	tssp->sprites[counter]->xinc = 0;
	tssp->sprites[counter]->yinc = 0;

	tssp->sprites[counter]->animspeed = 0;
	tssp->sprites[counter]->animspeed_counter = 0;
	tssp->sprites[counter]->noframes = 0;
	tssp->sprites[counter]->frame = 0;

	tssp->sprites[counter]->animframes = NULL;

	if ((tssp->sprites[counter]->buffer =
	 (UBYTE *) malloc (SPR_MAX_X * SPR_MAX_Y * sizeof (UBYTE))) == NULL)
	  {
	     JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at sprite_init line %d\n",__LINE__);
	     JLIB_PRINT_MESSAGE_STRING;

	     return (sprite_system *) NULL;
	  }

	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Initialised sprite %d OK\n", counter);
	JLIB_PRINT_MESSAGE_STRING;

     }

   /* give back the handle */
   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_init()\n");
   return tssp;

}				/* init_sprite_system */


/*+----------------------------------------------------------------------+ */
/*|sprite_load - load a sprite file into a sprite system.                | */
/*+----------------------------------------------------------------------+ */

/* format of a sprite file -
   **
   **   USHORT = number of sprite frames in the file.
   **
   **   for each sprite frame:
   **      UBYTE          = width
   **      UBYTE          = height
   **      (width*height) UBYTES = data
   **      UBYTE          = number of bounding rectangles
   **      for each bounding rect:
   **          UBYTE          = x1
   **          UBYTE          = y1
   **          UBYTE          = x2
   **          UBYTE          = y2
   **      }
   **   }
   **  }

   Note:  Sprite files are stored using the device independant routines in "jio.h"
 */


#define _SHEIGHT  (ssys->sprite_data[count1]->height)
#define _SWIDTH   (ssys->sprite_data[count1]->width)
#define _SSIZE    (_SHEIGHT*_SWIDTH)

void generate_RLE (sprite_data_rec * srec);

UBYTE sprite_load (char *filename, sprite_system * ssys)
{
   FILE *spritefile;
   USHORT num_in_file;
   int count1, i, j;

   JLIB_PRINT_DEBUG_INFO ("Entering load_sprites()\n");

   if ((spritefile = fopen (filename, "rb")) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "couldn't open sprite file %s\n", filename);
	JLIB_PRINT_MESSAGE_STRING;
	return COULDNT_OPEN;
     }
   else
     {
        jio_read_elementary_type(spritefile,&num_in_file,sizeof(USHORT));

	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "%d sprites in file %s\n", num_in_file, filename);
	JLIB_PRINT_MESSAGE_STRING;

	if ((num_in_file+SPR_NUM_LOADED(ssys)) > SPR_MAX_FRAMES (ssys))
	  {
	     JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Not enough frames (%d) for sprites\n", SPR_MAX_FRAMES (ssys));
	     JLIB_PRINT_MESSAGE_STRING;
	     return TOO_MANY_IN_FILE;
	  }

	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "%d frames available for for sprites\n", SPR_MAX_FRAMES (ssys));
	JLIB_PRINT_MESSAGE_STRING;

	for (count1 = SPR_NUM_LOADED(ssys); count1 < (SPR_NUM_LOADED(ssys)+num_in_file); count1++)
	  {

	     if ((ssys->sprite_data[count1] =
	     (sprite_data_rec *) malloc (sizeof (sprite_data_rec))) == NULL)
	       {
		  JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at load_sprites line %d\n", __LINE__);
		  JLIB_PRINT_MESSAGE_STRING;
		  return OUT_OF_MEMORY;
	       }

             jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->width,sizeof(UBYTE));
             jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->height,sizeof(UBYTE));

	     if ((ssys->sprite_data[count1]->data =
		  (UBYTE *) malloc ((_SHEIGHT * _SWIDTH) * sizeof (UBYTE))) == NULL)
	       {
		  JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at load_sprites line %d\n", __LINE__);
		  JLIB_PRINT_MESSAGE_STRING;
		  return OUT_OF_MEMORY;

	       }

	     fread (ssys->sprite_data[count1]->data, _SSIZE, 1, spritefile);

           jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->no_rects,sizeof(UBYTE));
             
             /* if there are bounding rectangles, read them in */
             if(ssys->sprite_data[count1]->no_rects!=0){
             
	       if ((ssys->sprite_data[count1]->rect_coords =
		    (UBYTE *) malloc (((ssys->sprite_data[count1]->no_rects) * 4) * sizeof (UBYTE))) == NULL){
		    JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at load_sprites line %d\n", __LINE__);
		    JLIB_PRINT_MESSAGE_STRING;
		    return OUT_OF_MEMORY;
	       }

	       j = 0;

	       for (i = 0; i < ssys->sprite_data[count1]->no_rects; i++){
		  /* read in each bounding rectangles coords */
                  jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
		  j++;
                  jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
		  j++;
                  jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
		  j++;
                  jio_read_elementary_type(spritefile,&ssys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
		  j++;
               }	
             }
             else{
                  ssys->sprite_data[count1]->rect_coords == NULL;
             }
             
	     generate_RLE (ssys->sprite_data[count1]);
	  }
	  
	  /* add the loaded sprites into the system count */
	  ssys->number_loaded += num_in_file;

	fclose (spritefile);
     }

   JLIB_PRINT_DEBUG_INFO ("Leaving load_sprites()\n");

   return SUCCESS;
}

/* This is a private routine.  Just forget you ever saw it! */
void generate_RLE (sprite_data_rec * srec)
{
   UBYTE wcount, hcount, runlength, runtype, datum;
   UBYTE linereps, *linereparr;
   USHORT rlecount;

   JLIB_PRINT_DEBUG_INFO ("Entering generate_RLE()\n");

   /* create an array to hold the line reps */
   if ((linereparr = (UBYTE *) malloc (srec->height * sizeof (UBYTE))) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at generate_RLE line %d\n", __LINE__);
	JLIB_PRINT_MESSAGE_STRING;
	exit (-1);
     }

   rlecount = 0;

   /* this part is just to work out the size to malloc space for */
   for (hcount = 0; hcount < srec->height; hcount++)
     {
	linereps = 1;

	datum = srec->data[(hcount * srec->width)];

	if (datum > 0)
	  {			/* what is the ponsy bit-twiddle for this? */
	     datum = 1;
	  }

	runtype = datum;	/* this is important! */
	runlength = 1;

	for (wcount = 1; wcount < srec->width; wcount++)
	  {
	     datum = srec->data[(hcount * srec->width) + wcount];

	     if (datum > 0)
	       {
		  datum = 1;
	       }

	     if (datum == runtype)
	       {
		  ++runlength;
	       }
	     else
	       {
		  runtype = datum;

		  ++linereps;
		  runlength = 1;
	       }
	  }

	linereparr[hcount] = linereps;
	rlecount += (linereps * 2) + 1;

     }

   if ((srec->pattern = (UBYTE *) malloc (rlecount * sizeof (UBYTE))) == NULL)
     {
	JLIB_SPRINTF (JLIB_MESSAGE_STRING, "malloc failed at generate_RLE line %d\n", __LINE__);
	JLIB_PRINT_MESSAGE_STRING;
	exit (-1);
     }
   
   /* now we actually encode the data */
   rlecount = 0;

   for (hcount = 0; hcount < srec->height; hcount++)
     {

	srec->pattern[rlecount] = linereparr[hcount];

	++rlecount;
	datum = srec->data[(hcount * srec->width)];

	if (datum > 0)
	  {
	     datum = 1;
	  }

	srec->pattern[rlecount] = datum;
	++rlecount;

	runtype = datum;	/* this is important! */
	runlength = 1;

	for (wcount = 1; wcount < srec->width; wcount++)
	  {
	     datum = srec->data[(hcount * srec->width) + wcount];

	     if (datum > 0)
	       {
		  datum = 1;
	       }

	     if (datum == runtype)
	       {
		  ++runlength;
	       }
	     else
	       {
		  runtype = datum;

		  srec->pattern[rlecount] = runlength;
		  ++rlecount;
		  srec->pattern[rlecount] = runtype;
		  ++rlecount;

		  runlength = 1;
	       }
	  }
	srec->pattern[rlecount] = runlength;
	++rlecount;
     }

   free (linereparr);

   JLIB_PRINT_DEBUG_INFO ("Leaving generate_RLE()\n");
}


/*+----------------------------------------------------------------------+ */
/*|sprite_turn_on - turn on a sprite                                     | */
/*+----------------------------------------------------------------------+ */

JINLINE void sprite_turn_on (sprite_system * spr_sys, USHORT snum)
{
   JLIB_PRINT_DEBUG_INFO ("Entering sprite_turn_on()\n");

   JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Turning on sprite number %d\n", snum);
   JLIB_PRINT_MESSAGE_STRING;

   spr_sys->active_sprites[snum] = 1;

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_turn_on()\n");
}

/*+----------------------------------------------------------------------+ */
/*|sprite_turn_off - turn off a sprite                                   | */
/*+----------------------------------------------------------------------+ */

JINLINE void sprite_turn_off (sprite_system * spr_sys, USHORT snum)
{
   JLIB_PRINT_DEBUG_INFO ("Entering sprite_turn_off()\n");

   JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Turning off sprite number %d\n", snum);
   JLIB_PRINT_MESSAGE_STRING;

   spr_sys->active_sprites[snum] = 0;

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_turn_off()\n");
}


/*+----------------------------------------------------------------------+ */
/*|sprite_set_an_frame - set a sprites animation frame                   | */
/*+----------------------------------------------------------------------+ */

JINLINE void sprite_set_an_frame (sprite_system * ssys, USHORT snum, USHORT frame)
{
   JLIB_PRINT_DEBUG_INFO ("Entering sprite_set_an_frame()\n");

   JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Setting sprite %d's frame to %d\n", snum, frame);
   JLIB_PRINT_MESSAGE_STRING;

   ssys->sprites[snum]->frame = frame;

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_set_an_frame()\n");
}


/*+----------------------------------------------------------------------+ */
/*|sprite_set_xy - change a sprites position                             | */
/*+----------------------------------------------------------------------+ */

JINLINE void sprite_set_xy (sprite_system * spr_sys, USHORT snum, USHORT newx, USHORT newy)
{
   JLIB_PRINT_DEBUG_INFO ("Entering sprite_set_xy()\n");

   JLIB_SPRINTF (JLIB_MESSAGE_STRING, "Setting sprite %d's x/y to %d & %d\n", snum, newx, newy);
   JLIB_PRINT_MESSAGE_STRING;

   spr_sys->sprites[snum]->y = newy;
   spr_sys->sprites[snum]->x = newx;

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_set_xy()\n");
}


/*+----------------------------------------------------------------------+ */
/*|find_first_free - find the number of the first free sprite.           | */
/*+----------------------------------------------------------------------+ */

JINLINE USHORT sprite_find_first_free (sprite_system * spr_sys)
{
   int count;

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_find_first_free()\n");

   /* NOTE : this routine will DIE if there are no free sprites - always keep
      **        at least one free in the system.
    */
   for (count = 0; spr_sys->active_sprites[count] != 0; count++);

   return count;

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_find_first_free()\n");
}

/*+----------------------------------------------------------------------+ */
/*|set_move_info - Set up the movement info for a sprite.                | */
/*+----------------------------------------------------------------------+ */

JINLINE void sprite_set_move_info (sprite_system * spr_sys, USHORT snum,
				   UBYTE speed, BYTE xinc, BYTE yinc)
{
   JLIB_PRINT_DEBUG_INFO ("Entering sprite_set_move_info()\n");

   spr_sys->sprites[snum]->speed = speed;
   spr_sys->sprites[snum]->speed_counter = speed;
   spr_sys->sprites[snum]->xinc = xinc;
   spr_sys->sprites[snum]->yinc = yinc;

   JLIB_PRINT_DEBUG_INFO ("leaving sprite_set_move_info()\n");
}

/*+----------------------------------------------------------------------+ */
/*|set_anim_info - Set up the animation info for a sprite.               | */
/*+----------------------------------------------------------------------+ */

JINLINE void sprite_set_anim_info (sprite_system * spr_sys, USHORT snum,
			     UBYTE anspeed, UBYTE noframes, USHORT *animpat)
{

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_set_anim_info()\n");


   spr_sys->sprites[snum]->animspeed = anspeed;
   spr_sys->sprites[snum]->animspeed_counter = anspeed;
   spr_sys->sprites[snum]->noframes = noframes;
   spr_sys->sprites[snum]->frame = 0;
   spr_sys->sprites[snum]->frame_count = 0;
   spr_sys->sprites[snum]->animframes = animpat;


   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_set_anim_info()\n");
}

/*+---------------------------------------------------------------------+ */
/*|update_anim_and_move - update a sprites posistion and animframe.     | */
/*+---------------------------------------------------------------------+ */

JINLINE void sprite_update_anim_and_move (sprite_system * spr_sys, USHORT snum)
{
   sprite_record *current;

   current = spr_sys->sprites[snum];

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_update_anim_and_move()\n");

   if (SPR_IS_ON (spr_sys, snum))
     {
	/* check movement */
	if (current->speed)
	  {
	     if ((--current->speed_counter) == 0)
	       {
		  current->speed_counter = current->speed;
		  current->x += current->xinc;
		  current->y += current->yinc;
	       }
	  }

	/* check animation */
	if (current->animspeed)
	  {
	     if ((--current->animspeed_counter) == 0)
	       {
		  current->animspeed_counter = current->animspeed;

		  if ((++current->frame_count) == current->noframes)
		    {
		       current->frame_count = 0;
		    }
		  current->frame = current->animframes[current->frame_count];
	       }
	  }
     }

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_update_anim_and_move()\n");
}


/*+----------------------------------------------------------------------+ */
/*|sprite_update_all_anim_and_move                                       | */
/*+----------------------------------------------------------------------+ */
/*|update every active sprite.                                           | */
/*+----------------------------------------------------------------------+ */

void sprite_update_all_anim_and_move (sprite_system * spr_sys)
{
   int count;

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_update_all_anim_and_move()\n");


   for (count = 0; count < spr_sys->no_sprites; count++)
     {
	if (SPR_IS_ON (spr_sys, count))
	  {
	     sprite_update_anim_and_move (spr_sys, count);
	  }
     }

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_update_all_anim_and_move()\n");
}


/*+----------------------------------------------------------------------+ */
/*|sprite_do_all_anim_and_move_n_times                                   | */
/*+----------------------------------------------------------------------+ */
/*|update every active sprite n times.                                   | */
/*+----------------------------------------------------------------------+ */

void sprite_do_all_anim_and_move_n_times (sprite_system * spr_sys, int n)
{
   int count, i;

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_do_all_anim_and_move_n_times()\n");


   for (count = 0; count < spr_sys->no_sprites; count++)
     {
	if (SPR_IS_ON (spr_sys, count))
	  {
	     for (i = n; i != 0; i--)
	       {
		  sprite_update_anim_and_move (spr_sys, count);
	       }
	  }
     }

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_do_all_anim_and_move_n_times()\n");
}


/*+----------------------------------------------------------------------+ */
/*|JLIB_do_intersect - this routine is private!                          | */
/*+----------------------------------------------------------------------+ */
#ifndef SWAP
#define SWAP(a,b)       { a^=b; b^=a; a^=b; }
#endif

JINLINE int JLIB_do_intersect (USHORT x1, USHORT y1, UBYTE *r1, USHORT x2, USHORT y2, UBYTE *r2)
{
 if(((x1+r1[0]) < (x2 +r2[2])) &&
    ((x1+r1[2]) > (x2 +r2[0])) &&
    ((y1+r1[1]) < (y2 +r2[3])) &&
    ((y1+r1[3]) > (y2 +r2[1]))){
     return 1;
 }
 
 return 0;    

}


/*+----------------------------------------------------------------------+ */
/*|sprite_do_intersect                                                   | */
/*+----------------------------------------------------------------------+ */
/*|check if two sprites are colliding.                                   | */
/*+----------------------------------------------------------------------+ */

int sprite_do_intersect (sprite_system * spr_sys, USHORT s1, USHORT s2)
{
   USHORT s1x, s1y, s2x, s2y;
   UBYTE *s1rect, *s2rect;
   int s1count, s2count;


   JLIB_PRINT_DEBUG_INFO ("Entering sprite_do_intersect()\n");

   /* if either sprite is off,  return no */
   if ((!SPR_IS_ON (spr_sys, s1)) || (!SPR_IS_ON (spr_sys, s2)))
     {
	return 0;
     }

   /* set up preliminary information */
   s1x = spr_sys->sprites[s1]->x;
   s1y = spr_sys->sprites[s1]->y;
   s2x = spr_sys->sprites[s2]->x;
   s2y = spr_sys->sprites[s2]->y;

   s1rect = spr_sys->sprite_data[spr_sys->sprites[s1]->frame]->rect_coords;
   s2rect = spr_sys->sprite_data[spr_sys->sprites[s2]->frame]->rect_coords;

   /* for each bounding rectangle of sprite 1,  check each rect of sprite 2 */
   for (s1count = spr_sys->sprite_data[spr_sys->sprites[s1]->frame]->no_rects; s1count != 0; s1count--)
     {
	for (s2count = spr_sys->sprite_data[spr_sys->sprites[s2]->frame]->no_rects; s2count != 0; s2count--)
	  {
	     if (JLIB_do_intersect (s1x, s1y, s1rect, s2x, s2y, s2rect))
	       {
		  return 1;
	       }
	     else
	       {
		  /* move to the 2nd sprites next rectangle coordinates */
		  s2rect += 4;
	       }
	  }
	s2rect -= (spr_sys->sprite_data[spr_sys->sprites[s2]->frame]->no_rects * 4);
	
	/* move to the 1st sprites next rectangle coordinates */
	s1rect += 4;
     }


   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_do_intersect()\n");
   /* if we get down here,  they don't intersect */
   return 0;

}


/*+----------------------------------------------------------------------+ */
/*|sprite_free - free the memory used by a sprite system.                | */
/*+----------------------------------------------------------------------+ */

sprite_system *sprite_free (sprite_system * ssys)
{
   int counter;

   JLIB_PRINT_DEBUG_INFO ("Entering sprite_free()\n");

   if (ssys == NULL)
     {
	return NULL;
     }

   if (ssys->sprites != NULL)
     {
	for (counter = 0; counter < SPR_MAX_SPRITES (ssys); counter++)
	  {
	     free (ssys->sprites[counter]->buffer);
	     free (ssys->sprites[counter]);
	  }
     }

   free (ssys->sprites);

   if (ssys->sprite_data != NULL)
     {
	for (counter = 0; counter < SPR_MAX_FRAMES (ssys); counter++)
	  {
	     if (ssys->sprite_data[counter] != NULL)
	       {
		  free (ssys->sprite_data[counter]->data);
		  free (ssys->sprite_data[counter]->pattern);
		  free (ssys->sprite_data[counter]->rect_coords);
		  free (ssys->sprites[counter]);
	       }
	  }
     }

   free (ssys->sprite_data);

   free (ssys->active_sprites);

   free (ssys);

   JLIB_PRINT_DEBUG_INFO ("Leaving sprite_free()\n");

   return NULL;
}
