// c_sample_inline.h: implementation of the c_sample class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined(AFX_C_SAMPLE_INLINE_H__2691C105_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_SAMPLE_INLINE_H__2691C105_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_

coefs first_coefs(double y1, double y2, double y3)
{
	double s2 ;
	double s3 ;

	double t1 ;
	double t2 ;

	coefs cfs ;

	s2 = sqrt(1 + (y2 - y1) * (y2 - y1)) ;
	s3 = sqrt(1 + (y3 - y2) * (y3 - y2)) ;


	t1 = (y2 - y1) / s2 ;
	t2 = (s2 * (y3 - y2) + s3 * (y2 - y1)) / (s2 + s3) ;

	cfs.a = y1 ;
	cfs.b = t1 ;
	cfs.c = 3 * (y2 - y1) - 2 * t1 - t2 ;
	cfs.d = 2 * (y1 - y2) + t1 + t2 ;

	cfs.a *= 65536.0 ;
	cfs.c /= 65536.0 ;
	cfs.d /= 65536.0 ;
	cfs.d /= 65536.0 ;

	return cfs ;
}

coefs other_coefs(double y0, double y1, double y2, double y3)
{
	double s1 ;
	double s2 ;
	double s3 ;

	double t1 ;
	double t2 ;

	coefs cfs ;

	s1 = sqrt(1 + (y1 - y0) * (y1 - y0)) ;
	s2 = sqrt(1 + (y2 - y1) * (y2 - y1)) ;
	s3 = sqrt(1 + (y3 - y2) * (y3 - y2)) ;


	t1 = (s1 * (y2 - y1) + s2 * (y1 - y0)) / (s1 + s2) ;
	t2 = (s2 * (y3 - y2) + s3 * (y2 - y1)) / (s2 + s3) ;

	cfs.a = y1 ;
	cfs.b = t1 ;
	cfs.c = 3 * (y2 - y1) - 2 * t1 - t2 ;
	cfs.d = 2 * (y1 - y2) + t1 + t2 ;

	cfs.a *= 65536.0 ;
	cfs.c /= 65536.0 ;
	cfs.d /= 65536.0 ;
	cfs.d /= 65536.0 ;

	return cfs ;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_sample::c_sample(FILE *module_file)
{
	init_variables() ;
	
	// buffers
	signed char sc ;
	unsigned char uc ;
	signed short ss ;
	unsigned short us ;
	unsigned long ul ;
	
	{
		
		/* Verification of the IT sample marker 'IMPS' */
		
		char buffer [5] ;
		size_t return_value ;
		
		return_value = fread (buffer, (size_t) 1, (size_t) 4, module_file) ;
		if (return_value == (size_t) -1)
		{
			engine_error = "Cannot read the sample" ;
			return ;
		}
		
		if (strncmp ("IMPS", buffer, 4) != 0)
		{
			engine_error = "Bad sample marker" ;
			return ;
		}
		/* End of the marker verification */
		
	}
	
	/* Now we can assume (at least for now) that we
	have a regular IT sample */
	
	/* dos file name */
	(void) fread (m_dos_file_name, (size_t) 1, (size_t) 12, module_file) ;
	
	/* going to global volume field */
	(void) fseek (module_file, 1L, SEEK_CUR) ;
	
	/* global volume */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_global_volume = (signed long)uc ;
	
	/* flags */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	
	/* sample quality */
	m_is_16_bits_sample = (uc & 2) > 0 ;
	
	/* sample type */
	m_is_stereo_sample = (uc & 4) > 0 ;
	
	/* sample loop */
	m_use_loop = (uc & 16) > 0 ;
	
	/* sample sustain loop */
	m_use_sustain_loop = (uc & 32) > 0 ;
	
	/* sample loop direction */
	m_loop_direction = (uc & 64) > 0 ;
	
	/* sample sustain loop direction */
	m_sustain_loop_direction = (uc & 128) > 0 ;
	
	/* default volume */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_default_volume = (signed long) uc ;
	
	/* sample true name */
	(void) fread (m_name, (size_t) 1, (size_t) 26, module_file) ;
	
	/* convertion flags */
	(void) fread (&us, (size_t) 2, (size_t) 1, module_file) ;
	
	m_is_signed_sample = (us & 1) > 0 ;
	
	/* sample length */
	(void) fread (&ul, (size_t) 4, (size_t) 1, module_file) ;
	m_length = ul ; 
	
	/* loop begin */
	(void) fread (&ul, (size_t) 4, (size_t) 1, module_file) ;
	m_loop_begin = ul ; 
	
	/* loop end */
	(void) fread (&ul, (size_t) 4, (size_t) 1, module_file) ;
	m_loop_end = ul ; 
	
	/* C-5 speed */
	(void) fread (&ul, (size_t) 4, (size_t) 1, module_file) ;
	m_C5_speed = ul ; 
	
	/* sustain loop begin */
	(void) fread (&ul, (size_t) 4, (size_t) 1, module_file) ;
	m_sustain_loop_begin = ul ; 
	
	/* sustain loop end */
	(void) fread (&ul, (size_t) 4, (size_t) 1, module_file) ;
	m_sustain_loop_end = ul ; 
	
	{
		
		/* Sample data reading */
		
		long current_file_position ;
		long sample_data_position ;
		
		/* reading sample pointer */
		(void) fread (&sample_data_position, (size_t) 4, (size_t) 1, module_file) ;
		
		/* saving the current position of the file pointer */
		current_file_position = ftell (module_file) ;
		
		/* going to sample data */
		(void) fseek (module_file, sample_data_position, SEEK_SET) ;
		
		/* allocating memory for a mono sample */
		m_sample_data = new signed short[m_length + 256] ;
		
		if ((void *)m_sample_data == NULL)
		{
			engine_error = "Not enough memory to store sample data" ;
			return ;
		}
		
		if (m_is_16_bits_sample)
		{
			if (m_is_signed_sample)
			{
				for (unsigned long i = 0 ; i < m_length ; i ++)
				{
					/* reading a signed 16 bits sample */
					(void) fread (&ss, (size_t) 2, (size_t) 1, module_file) ;
					
					/* saving this sample with no conversion */
					m_sample_data[i] = (signed short) ss ;
				}
			}
			else
			{
				signed long converted_data ;
				
				for (unsigned long i = 0 ; i < m_length ; i ++)
				{
					/* reading a unsigned 16 bits sample */
					(void) fread (&us, (size_t) 2, (size_t) 1, module_file) ;
					
					converted_data = ((signed long) us) - 32767 ; /* drop the out-of-bound samples */
					if (converted_data > 32767)
					{
						converted_data = 32767 ;
					}
					
					/* saving this sample with an unsigned -> signed conversion */
					m_sample_data[i] = (signed short) converted_data ;
				}
			}
		}
		else /* 8 bits sample */
		{
			if (m_is_signed_sample)
			{
				for (unsigned long i = 0 ; i < m_length ; i ++)
				{
					/* reading a signed 8 bits sample */
					(void) fread (&sc, (size_t) 1, (size_t) 1, module_file) ;
					
					/* saving this sample with a 8 bits -> 16 bits conversion */
					m_sample_data[i] = 256 * (signed short) sc ;
				}
			}
			else
			{
				signed long converted_data ;
				
				for (unsigned i = 0 ; i < m_length ; i ++)
				{
					/* reading a unsigned 8 bits sample */
					(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
					
					converted_data = ((signed long) uc) - 127 ; /* drop the out-of-bound samples */
					if (converted_data > 127)
					{
						converted_data = 127 ;
					}
					
					/* saving this sample with a 8 bits -> 16 bits and unsigned -> signed conversion */
					m_sample_data[i] = 256 * (signed short)(converted_data) ;
				}
			}
			
		}
		
		if ((m_length >= 16) && !((m_use_loop) || (m_use_sustain_loop)))
		{
			for (unsigned long i = 0 ; i < 16 ; i++)
			{
				double temp = m_sample_data[m_length - i] ;
				temp*= i / 16.0 ;
				m_sample_data[m_length - i] = (signed short)temp ;
			}
		}
		
		
		fseek (module_file, current_file_position, SEEK_SET) ;
		/* restoring the file pointer after the
		'SamplePointer' field of the sample header */
		
		m_pointer_on_end = m_sample_data + (m_length - 1) ;
		
		if (m_use_loop)
		{
			m_pointer_on_loop_beginning = m_sample_data + m_loop_begin ;
			m_pointer_on_loop_end = m_sample_data + m_loop_end ;
		}
		else
		{
			m_pointer_on_loop_beginning = m_sample_data ;
			m_pointer_on_loop_end = m_pointer_on_end ;
		}
		
		if (m_use_sustain_loop)
		{
			m_pointer_on_sustain_loop_beginning = m_sample_data + m_sustain_loop_begin ;
			m_pointer_on_sustain_loop_end = m_sample_data + m_sustain_loop_end ;
		}
		else
		{
			m_pointer_on_sustain_loop_beginning = m_sample_data ;
			m_pointer_on_sustain_loop_end = m_pointer_on_end ;
		}
		
		m_end_of_loop_sample = *m_pointer_on_loop_end;
		m_end_of_sustain_loop_sample = *m_pointer_on_sustain_loop_end ;
	}
	
	/* vibrato speed */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_vibrato_speed = (signed long) uc ;
	
	/* vibrato depth */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_vibrato_depth = (signed long) uc ;
	
	/* vibrato rate */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_vibrato_rate = (signed long) uc ;
	
	/* vibrato waveform type */
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_vibrato_waveform = (signed long) uc ;
}

c_sample::c_sample(istream &is, bool linear_interpolation)
{
	init_variables() ;
	m_use_linear_interpolation = linear_interpolation ;
	
	// buffers
	signed char sc ;
	unsigned char uc ;
	signed short ss ;
	unsigned short us ;
	signed long sl ;
	unsigned long ul ;
	
	{
		
		/* Verification of the IT sample marker 'IMPS' */
		
		char buffer [5] ;
		
		is.read(buffer, 4) ;
		
		if (strncmp ("IMPS", buffer, 4) != 0)
		{
			engine_error = "Bad sample marker" ;
			return ;
		}
		/* End of the marker verification */
		
	}    
	
	/* Now we can assume (at least for now) that we
	have a regular IT sample */
	
	/* dos file name */
	is.read(m_dos_file_name, 12) ;
	
	/* going to global volume field */
	is.seekg((streamoff) 1, ios::cur) ;
	
	/* global volume */
	is.read((char *)&uc, 1) ;
	m_global_volume = (signed long)uc ;
	
	/* flags */
	is.read((char *)&uc, 1) ;

	/* sample on/off ? */
	m_is_on = (uc & 1) > 0 ;

	/* sample quality */
	m_is_16_bits_sample = (uc & 2) > 0 ;
	
	/* sample type */
	m_is_stereo_sample = (uc & 4) > 0 ;

	/* sample compression */
	m_is_compressed = (uc & 8) > 0 ;
	
	/* sample loop */
	m_use_loop = (uc & 16) > 0 ;
	
	/* sample sustain loop */
	m_use_sustain_loop = (uc & 32) > 0 ;
	
	/* sample loop direction */
	m_loop_direction = (uc & 64) > 0 ;
	
	/* sample sustain loop direction */
	m_sustain_loop_direction = (uc & 128) > 0 ;
	
	/* default volume */
	is.read((char *)&uc, 1) ;
	m_default_volume = (signed long) uc ;
	
	/* sample true name */
	is.read (m_name, 26) ;
	
	/* convertion flags */
	is.read((char *)&uc, 1) ;
	
	/* sample is signed or unsigned ? */
	m_is_signed_sample = (uc & 1) > 0 ;

	/* byte order : false -> lo/hi, true -> hi/lo */
	m_big_endian_byte_order = (uc & 2) > 0 ;

	/* sample encoding : false -> PCM, true -> Delta */
	m_delta_encoding = (uc & 4) > 0 ;
	
	/* default panning */	
	is.read((char *)&uc, 1) ;
	m_default_panning = uc & 0x7F ;
	m_default_panning_on = (uc & 128) > 0 ;

	/* sample length */
	is.read((char *)&ul, 4) ;
	m_length = ul ; 

	/* loop begin */
	is.read((char *)&ul, 4) ;
	m_loop_begin = ul ; 
	
	/* loop end */
	is.read((char *)&ul, 4) ;
	m_loop_end = ul ; 
	
	/* C-5 speed */
	is.read((char *)&ul, 4) ;
	m_C5_speed = ul ; 
	
	/* sustain loop begin */
	is.read((char *)&ul, 4) ;
	m_sustain_loop_begin = ul ; 
	
	/* sustain loop end */
	is.read((char *)&ul, 4) ;
	m_sustain_loop_end = ul ; 
	
	if ((m_length == 0) || m_big_endian_byte_order || m_delta_encoding)
	{
		/* 
		   * big endian byte order is not supported
		   * delta encoding is not supported
		   */
		m_is_on = false ;
	}
	
	if (m_is_on)
	{
		
		/* Sample data reading */
		
		streampos current_file_position ;
		streampos sample_data_position ;
		
		/* reading sample pointer */
		is.read((char *)&sl, 4) ;
		sample_data_position = (streampos)sl ;
		
		/* saving the current position of the file pointer */
		current_file_position = is.tellg() ;
		
		/* going to sample data */
		is.seekg(sample_data_position) ;
		
		/* allocating memory for a mono sample */
		m_sample_data = new signed short[m_length + 256] ;
		
		if ((void *)m_sample_data == NULL)
		{
			engine_error = "Not enough memory to store sample data" ;
		}

		if (!m_is_compressed)
		{
			/* <= IT 2.13 uncompressed sample */
			if (m_is_16_bits_sample)
			{
				if (m_is_signed_sample)
				{
					for (unsigned long i = 0 ; i < m_length ; i ++)
					{
						/* reading a signed 16 bits sample */
						is.read((char *)&ss, 2) ;
						
						/* saving this sample with no conversion */
						m_sample_data[i] = ss ;
					}
				}
				else
				{				
					for (unsigned long i = 0 ; i < m_length ; i ++)
					{
						signed long converted_data ;
						
						/* reading a unsigned 16 bits sample */
						is.read((char *)&us, 2) ;
						
						converted_data = ((signed long) us) - 32767 ; /* drop the out-of-bound samples */
						if (converted_data > 32767)
						{
							converted_data = 32767 ;
						}
						
						/* saving this sample with an unsigned -> signed conversion */
						m_sample_data[i] = (signed short) converted_data ;
					}
				}
			}
			else /* 8 bits sample */
			{
				if (m_is_signed_sample)
				{
					for (unsigned long i = 0 ; i < m_length ; i ++)
					{
						/* reading a signed 8 bits sample */
						is.read((char *)&sc, 1) ;
						
						/* saving this sample with a 8 bits -> 16 bits conversion */
						m_sample_data[i] = (signed short)(256 * sc) ;

					}
				}
				else
				{				
					for (unsigned i = 0 ; i < m_length ; i ++)
					{
						signed long converted_data ;
						
						/* reading a unsigned 8 bits sample */
						is.read((char *)&uc, 1) ;
						
						converted_data = ((signed long) uc) - 127 ; /* drop the out-of-bound samples */
						if (converted_data > 127)
						{
							converted_data = 127 ;
						}
						
						m_sample_data[i] = (signed short) (256 * converted_data) ;
					}
				}
				
			}
		}
		else
		{
			/* IT 2.14 compressed sample */
			if (m_is_16_bits_sample)
			{
				/* 16bit sample */
				signed short *decomp = m_sample_data ;

				for (unsigned long i = 0 ; i <= ((m_length - 1) / 16384) ; i++)
				{

					is.read((char *)&us, 2) ;

					char *comp = new char[us];

					is.read(comp, us) ;

					if (i < ((m_length - 1) / 16384))
					{
						decompress_16(comp, decomp, (unsigned short)16384) ;
					}
					else
					{
						decompress_16(comp, decomp, 1 + (unsigned short)((m_length - 1) % 16384)) ;
					}

					decomp += 16384 ;

					delete[] comp ;
				}

			}
			else
			{
				/* 8bit sample */
				signed short *decomp = m_sample_data ;
				for (unsigned long j = 0 ; j < m_length ; j++)
				{
					m_sample_data[j] = 0 ;
				}

				for (unsigned long i = 0 ; i <= ((m_length - 1) / 32768) ; i++)
				{
					is.read((char *)&us, 2) ;

					char *comp = new char[us];

					is.read(comp, us) ;

					if (i < ((m_length - 1) / 32768))
					{
						decompress_8(comp, decomp, (unsigned short)32768) ;
					}
					else
					{
						decompress_8(comp, decomp, 1 + (unsigned short)((m_length - 1) % 32768)) ;
					}

					decomp += 32768 ;
					delete[] comp ;
				}

			}
		}
		
		if ((m_length >= 16) && !((m_use_loop) || (m_use_sustain_loop)))
		{
			for (unsigned long i = 0 ; i < 16 ; i++)
			{
				double temp = m_sample_data[m_length - i] ;
				temp*= i / 16.0 ;
				m_sample_data[m_length - i] = (signed short)temp ;
			}
		}
		
		/* restoring the file pointer after the
		'SamplePointer' field of the sample header */
		is.seekg(current_file_position) ;
		
		if (!m_use_linear_interpolation)
		{
			m_coefs = new coefs[m_length] ;
			
			m_coefs[0] = first_coefs(m_sample_data[0], m_sample_data[1], m_sample_data[2]) ;
			
			m_sample_data[m_length] = 0 ;
			m_sample_data[m_length + 1] = 0 ;
			m_sample_data[m_length + 2] = 0 ;
			m_sample_data[m_length + 3] = 0 ;

			unsigned long i ;
			for (i = 1 ; i < m_length ; i++)
			{
				m_coefs[i] = other_coefs(m_sample_data[i - 1], m_sample_data[i], 
					m_sample_data[i + 1], m_sample_data[i + 2]) ;
			}

			m_coefs_end = m_coefs + (m_length - 1);
			if (m_use_loop)
			{
				m_coefs_loop_b = m_coefs + m_loop_begin ;
				m_coefs_loop_e = m_coefs + m_loop_end ;

				if (!m_loop_direction)
				{
					m_loop_beginning_coefs = other_coefs(m_sample_data[m_loop_end - 1], 
						m_sample_data[m_loop_begin], m_sample_data[m_loop_begin + 1], m_sample_data[m_loop_begin + 2]);
					
					m_loop_end_coefs = other_coefs(m_sample_data[m_loop_end - 2], 
						m_sample_data[m_loop_end - 1], m_sample_data[m_loop_begin], m_sample_data[m_loop_begin + 1]);
					
					m_loop_end_coefs_2 = other_coefs(m_sample_data[m_loop_end - 3], 
						m_sample_data[m_loop_end - 2], m_sample_data[m_loop_end - 1], m_sample_data[m_loop_begin]);
					
					m_loop_beginning_coefs_save = *m_coefs_loop_b;
					m_loop_end_coefs_save = *(m_coefs_loop_e - 1);
					m_loop_end_coefs_save_2 = *(m_coefs_loop_e - 2);
				}
				else
				{
					m_loop_beginning_coefs = other_coefs(m_sample_data[m_loop_begin + 1], 
						m_sample_data[m_loop_begin], m_sample_data[m_loop_begin + 1], 
						m_sample_data[m_loop_begin + 2]) ;
					m_loop_end_coefs = other_coefs(m_sample_data[m_loop_end - 2], 
						m_sample_data[m_loop_end - 1], m_sample_data[m_loop_end - 2], 
						m_sample_data[m_loop_end - 3]) ;
					m_loop_end_coefs_2 = other_coefs(m_sample_data[m_loop_end - 3], 
						m_sample_data[m_loop_end - 2], m_sample_data[m_loop_end - 1], 
						m_sample_data[m_loop_end - 2]) ;
					m_loop_beginning_coefs_save = *m_coefs_loop_b;
					m_loop_end_coefs_save = *(m_coefs_loop_e - 1);
					m_loop_end_coefs_save_2 = *(m_coefs_loop_e - 2);
				}
			}
			else
			{
				m_coefs_loop_b = m_coefs ;
				m_coefs_loop_e = m_coefs_end ;
				m_loop_beginning_coefs_save = *m_coefs_loop_b;
				m_loop_end_coefs_save = *(m_coefs_loop_e - 1);
				m_loop_end_coefs_save_2 = *(m_coefs_loop_e - 2);
			}
			
			if (m_use_sustain_loop)
			{
				m_coefs_sloop_b = m_coefs + m_loop_begin ;
				m_coefs_sloop_e = m_coefs + m_loop_end ;

				if (!m_sustain_loop_direction)
				{
					m_sustain_loop_beginning_coefs = other_coefs(m_sample_data[m_sustain_loop_end - 1], 
						m_sample_data[m_sustain_loop_begin], m_sample_data[m_sustain_loop_begin + 1], 
						m_sample_data[m_sustain_loop_begin + 2]);
					
					m_sustain_loop_end_coefs = other_coefs(m_sample_data[m_sustain_loop_end - 2], 
						m_sample_data[m_sustain_loop_end - 1], m_sample_data[m_sustain_loop_begin], 
						m_sample_data[m_sustain_loop_begin + 1]);
					
					m_sustain_loop_end_coefs_2 = other_coefs(m_sample_data[m_sustain_loop_end - 3], 
						m_sample_data[m_sustain_loop_end - 2], m_sample_data[m_sustain_loop_end - 1], 
						m_sample_data[m_sustain_loop_begin]);
					
					m_sustain_loop_beginning_coefs_save = *m_coefs_sloop_b;
					m_sustain_loop_end_coefs_save = *(m_coefs_sloop_e - 1);
					m_sustain_loop_end_coefs_save_2 = *(m_coefs_sloop_e - 2);
				}
				else
				{
					m_sustain_loop_beginning_coefs = other_coefs(m_sample_data[m_sustain_loop_begin + 1], 
						m_sample_data[m_sustain_loop_begin], m_sample_data[m_sustain_loop_begin + 1], 
						m_sample_data[m_sustain_loop_begin + 2]);
					
					m_sustain_loop_end_coefs = other_coefs(m_sample_data[m_sustain_loop_end - 2], 
						m_sample_data[m_sustain_loop_end - 1], m_sample_data[m_sustain_loop_end - 2], 
						m_sample_data[m_sustain_loop_end - 3]);
					
					m_sustain_loop_end_coefs_2 = other_coefs(m_sample_data[m_sustain_loop_end - 3], 
						m_sample_data[m_sustain_loop_end - 2], m_sample_data[m_sustain_loop_end - 1], 
						m_sample_data[m_sustain_loop_end - 2]);
					
					m_sustain_loop_beginning_coefs_save = *m_coefs_sloop_b;
					m_sustain_loop_end_coefs_save = *(m_coefs_sloop_e - 1);
					m_sustain_loop_end_coefs_save_2 = *(m_coefs_sloop_e - 2);
				}

			}
			else
			{
				m_coefs_sloop_b = m_coefs ;
				m_coefs_sloop_e = m_coefs_end ;
				m_sustain_loop_beginning_coefs_save = *m_coefs_sloop_b;
				m_sustain_loop_end_coefs_save = *(m_coefs_sloop_e - 1);
				m_sustain_loop_end_coefs_save_2 = *(m_coefs_sloop_e - 2);
			}

			delete m_sample_data ;
			m_sample_data = (signed short *)NULL ;
		}
		else
		{
			m_pointer_on_end = m_sample_data + (m_length - 1) ;
			
			if (m_use_loop)
			{
				m_pointer_on_loop_beginning = m_sample_data + m_loop_begin ;
				m_pointer_on_loop_end = m_sample_data + m_loop_end ;
			}
			else
			{
				m_pointer_on_loop_beginning = m_sample_data ;
				m_pointer_on_loop_end = m_pointer_on_end ;
			}
			
			if (m_use_sustain_loop)
			{
				m_pointer_on_sustain_loop_beginning = m_sample_data + m_sustain_loop_begin ;
				m_pointer_on_sustain_loop_end = m_sample_data + m_sustain_loop_end ;
			}
			else
			{
				m_pointer_on_sustain_loop_beginning = m_sample_data ;
				m_pointer_on_sustain_loop_end = m_pointer_on_end ;
			}
			
			m_end_of_loop_sample = *m_pointer_on_loop_end;
			m_end_of_sustain_loop_sample = *m_pointer_on_sustain_loop_end ;
		}
	}
	else
	{
		m_sample_data = (signed short *)NULL ;
	}
	
	/* vibrato speed */
	is.read((char *)&uc, 1) ;
	m_vibrato_speed = (signed long) uc ;
	
	/* vibrato depth */
	is.read((char *)&uc, 1) ;
	m_vibrato_depth = (signed long) uc ;
	
	/* vibrato rate */
	is.read((char *)&uc, 1) ;
	m_vibrato_rate = (signed long) uc ;
	
	/* vibrato waveform type */
	is.read((char *)&uc, 1) ;
	m_vibrato_waveform = (signed long) uc ;
}

c_sample::~c_sample()
{
	if (m_use_linear_interpolation)
	{
		if (m_sample_data != (signed short *)NULL)
		{
			if (m_is_on)
			{
				delete[] m_sample_data ;
			}
		}
	}
	else
	{
		if (m_coefs != (p_coefs)NULL)
		{
			if (m_is_on)
			{
				delete[] m_coefs ;
			}
		}
	}
}

void c_sample::set_output_C5_speed(double output_C5_speed)
{
	m_output_C5_speed = output_C5_speed ;
}

void c_sample::init_variables()
{
	m_use_linear_interpolation = true ;
	m_C5_speed = 8363 ;
	m_default_volume = 64 ;
	m_dos_file_name[0] = 0 ;
	m_end_of_loop_sample = 0 ;
	m_end_of_sustain_loop_sample = 0 ;
	m_global_volume = 64 ;
	m_is_16_bits_sample = false ;
	m_is_signed_sample = false ;
	m_is_stereo_sample = false ;
	m_length = 0 ;
	m_loop_begin = 0 ;
	m_loop_direction = false ;
	m_loop_end = 0 ;
	m_name[0] = 0 ;
	m_output_C5_speed = 1.0 ;
	m_pointer_on_end = NULL ;
	m_pointer_on_loop_beginning = NULL ;
	m_pointer_on_loop_end = NULL ;
	m_pointer_on_sustain_loop_beginning = NULL ;
	m_pointer_on_sustain_loop_end = NULL ;
	m_sample_data = NULL ;
	m_coefs = NULL ; 
	m_sustain_loop_begin = 0 ;
	m_sustain_loop_end = 0 ;
	m_use_loop = false ;
	m_use_sustain_loop = false ;
	m_vibrato_depth = 0 ;
	m_vibrato_rate = 0 ;
	m_vibrato_speed = 0 ;
	m_vibrato_waveform = 0 ;
}

const char *c_sample::get_name()
{
	return m_name ;
}

const char *c_sample::get_dos_name()
{
	return m_dos_file_name ;
}

signed long c_sample::get_volume()
{
	return m_global_volume ;
}

bool c_sample::is_on()
{
	return m_is_on ;
}

bool c_sample::is_16_bits()
{
	return m_is_16_bits_sample ;
}

bool c_sample::is_stereo()
{
	return m_is_stereo_sample ;
}

bool c_sample::is_loop_on()
{
	return m_use_loop ;
}

bool c_sample::is_sustain_loop_on()
{
	return m_use_sustain_loop ;
}

bool c_sample::is_loop_bidi()
{
	return m_loop_direction ;
}

bool c_sample::is_sustain_loop_bidi()
{
	return m_sustain_loop_direction ;
}

bool c_sample::is_signed()
{
	return m_is_signed_sample ;
}

signed long c_sample::get_default_volume()
{
	return m_default_volume ;
}

unsigned long c_sample::get_length()
{
	return m_length ;
}

unsigned long c_sample::get_loop_beginning()
{
	return m_loop_begin ;
}

unsigned long c_sample::get_loop_end()
{
	return m_loop_end ;
}

unsigned long c_sample::get_C5_speed()
{
	return m_C5_speed ;
}

double c_sample::get_output_C5_speed()
{
	return m_output_C5_speed ;
}

unsigned long c_sample::get_sustain_loop_beginning()
{
	return m_sustain_loop_begin ;
}

unsigned long c_sample::get_sustain_loop_end()
{
	return m_sustain_loop_end ;
}

signed long c_sample::get_vibrato_speed()
{
	return m_vibrato_speed ;
}

signed long c_sample::get_vibrato_depth()
{
	return m_vibrato_depth ;
}

signed long c_sample::get_vibrato_rate()
{
	return m_vibrato_rate ;
}

signed long c_sample::get_vibrato_waveform()
{
	return m_vibrato_waveform ;
}

signed short *c_sample::get_pointer_on_end()
{
	return m_pointer_on_end ;
}

signed short *c_sample::get_pointer_on_loop_beginning()
{
	return m_pointer_on_loop_beginning ;
}

signed short *c_sample::get_pointer_on_loop_end()
{
	return m_pointer_on_loop_end ;
}

signed short *c_sample::get_pointer_on_sustain_loop_beginning()
{
	return m_pointer_on_sustain_loop_beginning;
}

signed short *c_sample::get_pointer_on_sustain_loop_end()
{
	return m_pointer_on_sustain_loop_end ;
}

signed short c_sample::get_sample_data(unsigned long position)
{
	// No in-bounds checking there : this function is time critical
	return m_sample_data[position] ;
}

const signed short *c_sample::get_sample_data_pointer()
{
	return m_sample_data ;
}

// Call the following functions before accessing the sample data.
// This is needed for forward loops, to ensure that the first and
// the last sample of the loop are the same to avoid the interpolation
// routines to produce some clicks.
void c_sample::prepare_for_no_loop()
{
	if (m_use_linear_interpolation)
	{
		*m_pointer_on_loop_end = m_end_of_loop_sample ;
		*m_pointer_on_sustain_loop_end = m_end_of_sustain_loop_sample ;
	}
	else
	{
		m_coefs_loop_b[0] = m_loop_beginning_coefs_save ;
		m_coefs_loop_e[-1] = m_loop_end_coefs_save ; 
		m_coefs_loop_e[-2] = m_loop_end_coefs_save_2 ; 
		m_coefs_sloop_b[0] = m_sustain_loop_beginning_coefs_save ;
		m_coefs_sloop_e[-1] = m_sustain_loop_end_coefs_save ; 
		m_coefs_sloop_e[-2] = m_sustain_loop_end_coefs_save_2 ; 
	}
}

void c_sample::prepare_for_forward_loop()
{
	if (m_use_linear_interpolation)
	{
		*m_pointer_on_sustain_loop_end = m_end_of_sustain_loop_sample ;
		*m_pointer_on_loop_end = *m_pointer_on_loop_beginning ;
	}
	else
	{
		m_coefs_sloop_b[0] = m_sustain_loop_beginning_coefs_save ;
		m_coefs_sloop_e[-1] = m_sustain_loop_end_coefs_save ; 
		m_coefs_sloop_e[-2] = m_sustain_loop_end_coefs_save_2 ; 
		m_coefs_loop_b[0] = m_loop_beginning_coefs ;
		m_coefs_loop_e[-1] = m_loop_end_coefs ; 
		m_coefs_loop_e[-2] = m_loop_end_coefs_2 ; 
	}
}

void c_sample::prepare_for_bidi_loop()
{
	if (!m_use_linear_interpolation)
	{
		m_coefs_sloop_b[0] = m_sustain_loop_beginning_coefs_save ;
		m_coefs_sloop_e[-1] = m_sustain_loop_end_coefs_save ; 
		m_coefs_sloop_e[-2] = m_sustain_loop_end_coefs_save_2 ; 
		m_coefs_loop_b[0] = m_loop_beginning_coefs ;
		m_coefs_loop_e[-1] = m_loop_end_coefs ; 
		m_coefs_loop_e[-2] = m_loop_end_coefs_2 ; 
	}
}

void c_sample::prepare_for_forward_sustain_loop()
{
	if (m_use_linear_interpolation)
	{
		*m_pointer_on_loop_end = m_end_of_loop_sample ;
		*m_pointer_on_sustain_loop_end = *m_pointer_on_sustain_loop_beginning ;
	}
	else
	{
		m_coefs_loop_b[0] = m_loop_beginning_coefs_save ;
		m_coefs_loop_e[-1] = m_loop_end_coefs_save ; 
		m_coefs_loop_e[-2] = m_loop_end_coefs_save_2 ; 
		m_coefs_sloop_b[0] = m_sustain_loop_beginning_coefs ;
		m_coefs_sloop_e[-1] = m_sustain_loop_end_coefs ; 
		m_coefs_sloop_e[-2] = m_sustain_loop_end_coefs_2 ; 
	}
}

void c_sample::prepare_for_bidi_sustain_loop()
{
	if (!m_use_linear_interpolation)
	{
		m_coefs_loop_b[0] = m_loop_beginning_coefs_save ;
		m_coefs_loop_e[-1] = m_loop_end_coefs_save ; 
		m_coefs_loop_e[-2] = m_loop_end_coefs_save_2 ; 
		m_coefs_sloop_b[0] = m_sustain_loop_beginning_coefs ;
		m_coefs_sloop_e[-1] = m_sustain_loop_end_coefs ; 
		m_coefs_sloop_e[-2] = m_sustain_loop_end_coefs_2 ; 
	}
}

p_coefs c_sample::get_coefs()
{
	return m_coefs ;
}

p_coefs c_sample::get_coefs_end()
{
	return m_coefs_end ;
}
p_coefs c_sample::get_coefs_loop_b()
{
	return m_coefs_loop_b ;
}
p_coefs c_sample::get_coefs_loop_e()
{
	return m_coefs_loop_e ;
}
p_coefs c_sample::get_coefs_sloop_b()
{
	return m_coefs_sloop_b ;
}
p_coefs c_sample::get_coefs_sloop_e()
{
	return m_coefs_sloop_e ;
}

bool c_sample::use_linear_interpolation()
{
	return m_use_linear_interpolation ;
}


#endif // AFX_C_SAMPLE_INLINE_H__2691C105_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_