// c_envelope_inline.h: implementation of the c_envelope class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined(AFX_C_ENVELOPE_INLINE_H__2691C103_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_ENVELOPE_INLINE_H__2691C103_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_envelope::c_envelope()
{
	init_variables() ;
}


c_envelope::c_envelope(FILE *module_file, bool signed_envelope)
{
	init_variables()  ;
	
	// buffers
	unsigned char uc ;
	
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	
	m_envelope_on_off = (uc & 1) > 0 ;
	m_loop_on_off = (uc & 2) > 0 ;		
	m_sustain_loop_on_off = (uc & 4) > 0 ;
	
	
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_number_of_nodes = (signed long) uc ;
	
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_loop_begin = (signed long) uc ;
	
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_loop_end = (signed long) uc ;
	
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_sustain_loop_begin = (signed long) uc ;
	
	(void) fread (&uc, (size_t) 1, (size_t) 1, module_file) ;
	m_sustain_loop_end = (signed long) uc ;
	
	for (signed long node = 0 ; node < 25 ; node ++)
	{
		m_nodes[node] = new c_envelope_node (module_file, signed_envelope) ;	
		
		if ((void *)m_nodes[node] == NULL)
		{
			engine_error = "Not enough memory for allocating an envelope node" ;
		}
		
		if ((bool)engine_error)
		{
			return ;
		}
		
	}
	
	init_envelope() ;
	
}

c_envelope::c_envelope(istream &is, signed long envelope_type)
{
	/* 
	  envelope_type :
	     0 -> volume
		 1 -> panning
		 2 -> pitch
	  */
	init_variables() ;
	
	// buffers
	unsigned char uc ;
	
	is.read((char *)&uc, 1) ;
	
	m_envelope_on_off = (uc & 1) > 0 ;
	m_loop_on_off = (uc & 2) > 0 ;		
	m_sustain_loop_on_off = (uc & 4) > 0 ;

	if (envelope_type == 2)
	{
		/* Resonant filter envelope */
		m_filter_envelope_on_off = (uc & 128) > 0 ;
	}
	else
	{
		m_filter_envelope_on_off = false ;
	}
	
	/* Resonant filter is not supported */
	if (m_filter_envelope_on_off)
	{
		/* The envelope is disabled */
		m_envelope_on_off = false ;
	}
	
	is.read((char *)&uc, 1) ;
	m_number_of_nodes = (signed long) uc ;
	
	is.read((char *)&uc, 1) ;
	m_loop_begin = (signed long) uc ;
	
	is.read((char *)&uc, 1) ;
	m_loop_end = (signed long) uc ;
	
	is.read((char *)&uc, 1) ;
	m_sustain_loop_begin = (signed long) uc ;
	
	is.read((char *)&uc, 1) ;
	m_sustain_loop_end = (signed long) uc ;
	
	for (signed long node = 0 ; node < 25 ; node ++)
	{
		if (envelope_type == 0)
		{
			/* volume envelope : unsigned values */
			m_nodes[node] = new c_envelope_node (is, false) ;
		}
		else if (envelope_type == 1)
		{
			/* panning envelope : signed values */
			m_nodes[node] = new c_envelope_node (is, true) ;
		}
		else if (envelope_type == 2)
		{
			/* pitch envelope */

			if (m_filter_envelope_on_off)
			{
				/* pitch envelope used as filter envelope : unsigned values */
				m_nodes[node] = new c_envelope_node (is, false) ;
			}
			else
			{
				/* normal pitch envelope (signed values) */
				m_nodes[node] = new c_envelope_node (is, true) ;
			}
		}

		
		if ((void *)m_nodes[node] == NULL)
		{
			engine_error = "Not enough memory for allocating an envelope node" ;
		}
		
		if ((bool)engine_error)
		{
			return ;
		}
	}
	
	init_envelope() ;
	
}


c_envelope::~c_envelope()
{
	if (m_envelope_on_off && (m_interpolation != (double *)NULL))
	{
		delete[] m_interpolation ;
	}
	
	for (signed long counter = 0 ; counter < m_number_of_nodes ; counter++)
	{
		if ((void *)m_nodes[counter] != NULL)
		{
			delete m_nodes[counter] ;
		}
	}
}


void c_envelope::init_variables()
{
	m_envelope_on_off = false ;
	
	m_loop_on_off = false ;
	
	m_sustain_loop_on_off = false ;
	
	m_number_of_nodes = 0 ; 
	
	m_loop_begin = 0 ; 
	
	m_loop_end = 0 ; 
	
	m_sustain_loop_begin = 0 ;
	
	m_sustain_loop_end = 0 ;
	
	m_last_tick = 0;
	
	m_loop_begin_tick = 0 ;
	
	m_loop_end_tick = 0 ;
	
	m_sustain_loop_begin_tick = 0 ;
	
	m_sustain_loop_end_tick = 0 ;
	
	for (signed long counter = 0 ; counter < 25 ; counter ++)
	{
		m_nodes[counter] = (p_envelope_node) NULL ;
	}
	
	m_interpolation = (double *)NULL ;
}

void c_envelope::init_envelope()
{
	if (m_envelope_on_off)
	{ 
		m_last_tick = m_nodes[m_number_of_nodes - 1]->get_tick() ;
		
		m_loop_begin_tick = m_nodes[m_loop_begin]->get_tick() ;
		m_loop_end_tick = m_nodes[m_loop_end]->get_tick() ;
		
		m_sustain_loop_begin_tick = m_nodes[m_sustain_loop_begin]->get_tick() ;
		m_sustain_loop_end_tick = m_nodes[m_sustain_loop_end]->get_tick() ;
		
		m_interpolation = new double[m_last_tick + 1] ;
		
		if ((void *)m_interpolation == NULL)
		{
			engine_error = "Not enough memory for allocating an interpolated envelope" ;
			return ;
		}
		
		for (signed long current_node = 0 ; current_node < (m_number_of_nodes - 1) ; current_node ++)
		{
			p_envelope_node node_1 = m_nodes[current_node] ;
			p_envelope_node node_2 = m_nodes[current_node + 1] ;
			
			signed long tick_1 = node_1->get_tick() ;
			signed long tick_2 = node_2->get_tick() ;
			signed long number_of_ticks = tick_2 - tick_1 ;
			
			double value_1 = (double)node_1->get_y_value() ;
			double value_2 = (double)node_2->get_y_value() ;
			double delta = (value_2 - value_1) / number_of_ticks;
			
			for (signed long tick = tick_1 ; tick < tick_2 ; tick++)
			{
				m_interpolation[tick] = value_1 + delta * (tick - tick_1)  ;
			}				
		} 
		
		m_interpolation[m_last_tick] = (double)m_nodes[m_number_of_nodes - 1]->get_y_value() ;
		
	} 
	else
	{
		m_interpolation = (double *)NULL ;
	}
}


bool c_envelope::is_on()
{
	return m_envelope_on_off ;
}

bool c_envelope::is_loop_on()
{
	return m_loop_on_off ;
}

bool c_envelope::is_sustain_loop_on ()
{
	return m_sustain_loop_on_off ;
}

signed long c_envelope::get_number_of_nodes ()
{
	return m_number_of_nodes ;
}

signed long c_envelope::get_loop_beginning_node ()
{
	return m_loop_begin ;
}

signed long c_envelope::get_loop_end_node ()
{
	return m_loop_end ;
}

signed long c_envelope::get_sustain_loop_beginning_node()
{
	return m_sustain_loop_begin ;
}

signed long c_envelope::get_sustain_loop_end_node()
{
	return m_sustain_loop_end ;
}

signed long c_envelope::get_number_of_ticks()
{
	return m_last_tick + 1 ; // ticks are 0 based
}

signed long c_envelope::get_loop_beginning_tick ()
{
	return m_loop_begin_tick ;
}

signed long c_envelope::get_loop_end_tick ()
{
	return m_loop_end_tick ;
}

signed long c_envelope::get_sustain_loop_beginning_tick ()
{
	return m_sustain_loop_begin_tick ;
}

signed long c_envelope::get_sustain_loop_end_tick ()
{
	return m_sustain_loop_end_tick ;
}

p_envelope_node c_envelope::get_node (signed long node)
{
	if ((node >= 0) && (node < m_number_of_nodes))
	{
		return m_nodes[node] ;
	}
	else
	{
		return (p_envelope_node)NULL ;
	}
}

signed long c_envelope::get_node_tick (signed long node)
{
	if ((node >= 0) && (node < m_number_of_nodes))
	{
		return m_nodes[node]->get_tick() ;
	}
	else
	{
		return -1 ;
	}
}

signed long c_envelope::get_node_y_value (signed long node)
{
	if ((node >= 0) && (node < m_number_of_nodes))
	{
		return m_nodes[node]->get_y_value () ;
	}
	else
	{
		return 0 ;
	}
}

double c_envelope::get_tick_value (signed long tick)
{
	if ((tick >= 0) && (tick <= m_last_tick))
	{
		return m_interpolation[tick] ;
	}
	else
	{
		return 0 ;
	}
}

#endif // AFX_C_ENVELOPE_INLINE_H__2691C103_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_
